<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\ObjectWs\Version1\Attribute\Update;

use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Attribute\Update\Attribute;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\Attribute\Update\Request;
use Doctrine\Common\Collections\ArrayCollection;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $id;

    /**
     * @var string
     */
    private $name;

    /**
     * @var int
     */
    private $mandatory;

    /**
     * @var string
     */
    private $possibleValue;

    /**
     * @var string
     */
    private $defaultValue;

    /**
     * @var int
     */
    private $translatable;

    /**
     * @var string
     */
    private $groupId;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->id = $this->faker->sha1;
        $this->name = $this->faker->name;
        $this->mandatory = $this->faker->boolean ? 1 : 0;
        $this->possibleValue = $this->faker->word;
        $this->defaultValue = $this->faker->word;
        $this->translatable = $this->faker->boolean ? 1 : 0;
        $this->groupId = $this->faker->sha1;
    }

    public function testJsonSerialize()
    {
        $this->assertJsonStringEqualsJsonString($this->getRequestBody('json'), $this->setRequestBody('json'));
    }

    public function testXmlSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody('xml'), $this->setRequestBody('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function setRequestBody($type = 'xml')
    {
        $request = new Request();
        $attribute = new Attribute();

        $attribute
            ->setId($this->id)
            ->setName($this->name)
            ->setMandatory($this->mandatory)
            ->setPossibleValues(new ArrayCollection([$this->possibleValue]))
            ->setDefaultValue($this->defaultValue)
            ->setTranslatable($this->translatable)
            ->setGroupId($this->groupId);

        $request->setAttribute($attribute);

        return $this->serializer->serialize($request, ($type !== 'json' ? 'xml' : 'json'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getRequestBody($type = 'xml')
    {
        if ($type === 'json') {
            return '{
                "attributes": [
                    {
                        "id": "' . $this->id . '",
                        "name": "' . $this->name . '",
                        "mandatory": ' . $this->mandatory . ',
                        "possibleValues": [
                            "' . $this->possibleValue . '"
                        ],
                        "defaultValue": "' . $this->defaultValue . '",
                        "translatable": "' . $this->translatable . '",
                        "groupId": "' . $this->groupId . '"
                    }
                ]
            }';
        } else {
            return '<attributeUpdate>
                <attributes>
                    <attribute>
                        <id><![CDATA[' . $this->id . ']]></id>
                        <name><![CDATA[' . $this->name . ']]></name>
                        <mandatory>' . $this->mandatory . '</mandatory>
                        <possibleValues>
                            <value><![CDATA[' . $this->possibleValue . ']]></value>                        
                        </possibleValues>
                        <defaultValue><![CDATA[' . $this->defaultValue . ']]></defaultValue>
                        <translatable>' . $this->translatable . '</translatable>
                        <groupId><![CDATA[' . $this->groupId . ']]></groupId>
                    </attribute>
                </attributes>
            </attributeUpdate>';
        }
    }

    public function testGetters()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody('json'),
            Request::class,
            'json'
        );

        $attributeType = $requestModel->getAttributes()->get(0);

        $this->assertEquals($this->id, $attributeType->getId());
        $this->assertEquals($this->name, $attributeType->getName());
        $this->assertEquals($this->mandatory, $attributeType->getMandatory());
        $this->assertEquals($this->possibleValue, $attributeType->getPossibleValues()->get(0));
        $this->assertEquals($this->defaultValue, $attributeType->getDefaultValue());
        $this->assertEquals($this->translatable, $attributeType->getTranslatable());
        $this->assertEquals($this->groupId, $attributeType->getGroupId());
    }
}
