<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\ObjectWs\Version1\AttributeGroup\Update;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\AttributeGroup\Update;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest\Request\ObjectWs\Version1\AttributeGroup\Update
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var Update\Request
     */
    private $request;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpRequest();
    }

    private function setUpRequest()
    {
        $request = new Update\Request();
        $attributeGroup = new Update\AttributeGroup();

        $attributeGroup
            ->setId($this->faker->uuid)
            ->setName($this->faker->name)
            ->setAdd($this->faker->uuid)
            ->setAdd($this->faker->uuid)
            ->setDelete($this->faker->uuid)
            ->setDelete($this->faker->uuid)
            ->setDeleteReferences(true);

        $request->addAttributeGroup($attributeGroup);
        $this->request = $request;
    }

    public function testGetAndAddAttributeGroups()
    {
        $request = new Update\Request();

        $attributeGroup1 = new Update\AttributeGroup();
        $attributeGroup2 = new Update\AttributeGroup();

        $attributeGroup1Id = $this->faker->uuid;
        $attributeGroup1->setId($attributeGroup1Id);
        $attributeGroup2Id = $this->faker->uuid;
        $attributeGroup2->setId($attributeGroup2Id);

        $request
            ->addAttributeGroup($attributeGroup1)
            ->addAttributeGroup($attributeGroup2);

        static::assertEquals($request->getAttributeGroups()->count(), 2);
        static::assertEquals($request->getAttributeGroups()->get(0)->getId(), $attributeGroup1Id);
        static::assertEquals($request->getAttributeGroups()->get(1)->getId(), $attributeGroup2Id);
    }

    public function testJSONSerialize()
    {
        static::assertJsonStringEqualsJsonString($this->serializer->serialize($this->request, 'json'),
            $this->getSerializedRequestContent('json'));
    }

    public function testXMLSerialize()
    {
        static::assertXmlStringEqualsXmlString($this->serializer->serialize($this->request, 'xml'),
            $this->getSerializedRequestContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedRequestContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
           {
               "attributeGroups": [
                   {
                       "id": "{$this->request->getAttributeGroups()->get(0)->getId()}",
                       "name": "{$this->request->getAttributeGroups()->get(0)->getName()}",
                       "add": [
                           "{$this->request->getAttributeGroups()->get(0)->getAdd()->get(0)}",
                           "{$this->request->getAttributeGroups()->get(0)->getAdd()->get(1)}"
                       ],
                       "delete": [
                           "{$this->request->getAttributeGroups()->get(0)->getDelete()->get(0)}",
                           "{$this->request->getAttributeGroups()->get(0)->getDelete()->get(1)}"
                       ],
                       "deleteReferences": {$this->request->getAttributeGroups()->get(0)->isDeleteReferences()}
                   }
               ]
           }
JSON;
            case 'xml':
                return <<< XML
<?xml version="1.0" encoding="UTF-8"?>
<attributeGroupUpdate>
    <attributeGroups>
        <attributeGroup>
            <id>
                <![CDATA[{$this->request->getAttributeGroups()->get(0)->getId()}]]>
            </id>
            <name>
                <![CDATA[{$this->request->getAttributeGroups()->get(0)->getName()}]]>
            </name>
            <add>
                <id>
                    <![CDATA[{$this->request->getAttributeGroups()->get(0)->getAdd()->get(0)}]]>
                </id>
                <id>
                    <![CDATA[{$this->request->getAttributeGroups()->get(0)->getAdd()->get(1)}]]>
                </id>
            </add>
            <delete>
                <id>
                    <![CDATA[{$this->request->getAttributeGroups()->get(0)->getDelete()->get(0)}]]>
                </id>
                <id>
                    <![CDATA[{$this->request->getAttributeGroups()->get(0)->getDelete()->get(1)}]]>
                </id>
            </delete>
            <deleteReferences>{$this->booleanToString($this->request->getAttributeGroups()->get(0)->isDeleteReferences())}</deleteReferences>
        </attributeGroup>
    </attributeGroups>
</attributeGroupUpdate>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }

    /**
     * @param boolean $boolean
     *
     * @return string
     */
    private function booleanToString($boolean)
    {
        return var_export($boolean, true);
    }
}