<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserApplication\Delete;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplication\Delete\ApplicationType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplication\Delete\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplication\Delete\UserType;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $applicationName;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->userIdentifier = $this->faker->email;
        $this->applicationName = $this->faker->name;
    }

    public function testAddUserType()
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();

        $applicationType->setName($this->applicationName);

        $userType
            ->setIdentifier($this->userIdentifier)
            ->setApplicationType($applicationType);

        $request->addUserType($userType);

        static::assertNotEmpty($request->getUserType());
        static::assertEquals($request->getUserType()->get(0), $userType);
    }

    public function testJSONSerialize()
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();

        $applicationType->setName($this->applicationName);

        $userType
            ->setIdentifier($this->userIdentifier)
            ->setApplicationType($applicationType);

        $request->addUserType($userType);

        static::assertJsonStringEqualsJsonString($this->serializer->serialize($request, 'json'),
            $this->getSerializedRequestContent('json'));
    }

    public function testXMLSerialize()
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();

        $applicationType->setName($this->applicationName);

        $userType
            ->setIdentifier($this->userIdentifier)
            ->setApplicationType($applicationType);

        $request->addUserType($userType);

        static::assertXmlStringEqualsXmlString($this->serializer->serialize($request, 'xml'),
            $this->getSerializedRequestContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedRequestContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
                {
                    "users": [
                        {
                            "identifier": "$this->userIdentifier",
                            "applicationType": {
                                "name": "$this->applicationName"
                            }
                        }
                    ]
                }
JSON;
            case 'xml':
                return <<< XML
                <userApplicationDelete>
                    <users>
                        <userType>
                            <identifier><![CDATA[$this->userIdentifier]]></identifier>
                            <applicationType>
                                <name><![CDATA[$this->applicationName]]></name>
                            </applicationType>
                        </userType>
                    </users>
                </userApplicationDelete>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}