<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserApplicationAttribute\Add;

use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationAttribute\Add\ApplicationType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationAttribute\Add\AttributeType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationAttribute\Add\UserType;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserApplicationAttribute\Add\Request;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $identifier;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $type;

    /**
     * @var string
     */
    private $value;

    /**
     * @var string
     */
    private $value2;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->identifier = $this->faker->userName;
        $this->name = $this->faker->name;
        $this->type = $this->faker->word;
        $this->value = $this->faker->word;
        $this->value2 = $this->faker->word;
    }

    public function testJsonSerialize()
    {
        $this->assertJsonStringEqualsJsonString($this->getRequestBody('json'), $this->setRequestBody('json'));
    }

    public function testXmlSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody('xml'), $this->setRequestBody('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function setRequestBody($type = 'xml')
    {
        $request = new Request();
        $userType = new UserType();
        $applicationType = new ApplicationType();

        $userType->setIdentifier($this->identifier);
        $applicationType->setName($this->name);

        $attributeType = new AttributeType();
        $attributeType
            ->setName($this->name)
            ->setType($this->type)
            ->setValue($this->value)
            ->setUserType($userType)
            ->setApplicationType($applicationType);

        $request->setAttributeType($attributeType);

        $attributeType = new AttributeType();
        $attributeType
            ->setName($this->name)
            ->setType('many')
            ->setValue([$this->value, $this->value2])
            ->setUserType($userType)
            ->setApplicationType($applicationType);

        $request->setAttributeType($attributeType);

        return $this->serializer->serialize($request, ($type !== 'json' ? 'xml' : 'json'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getRequestBody($type = 'xml')
    {
        if ($type === 'json') {
            return '{
                "userAttributes": [
                    {
                        "name": "' . $this->name . '",
                        "type": "' . $this->type . '",
                        "value": "' . $this->value . '",
                        "userType": {
                            "identifier": "' . $this->identifier . '"
                        },
                        "applicationType": {
                            "name": "' . $this->name . '"
                        }
                    },
                    {
                        "name": "' . $this->name . '",
                        "type": "many",
                        "value": ["' . $this->value . '","' . $this->value2 . '"],
                        "userType": {
                            "identifier": "' . $this->identifier . '"
                        },
                        "applicationType": {
                            "name": "' . $this->name . '"
                        }
                    }
                ]
            }';
        } else {
            return '<userAttributeAdd>
                <userAttributes>
                    <attributeType>
                        <name><![CDATA[' . $this->name . ']]></name>
                        <type><![CDATA[' . $this->type . ']]></type>
                        <value><![CDATA[' . $this->value . ']]></value>
                        <userType>
                            <identifier><![CDATA[' . $this->identifier . ']]></identifier>
                        </userType>
                        <applicationType>
                            <name><![CDATA[' . $this->name . ']]></name>
                        </applicationType>
                    </attributeType>
                    <attributeType>
                        <name><![CDATA[' . $this->name . ']]></name>
                        <type><![CDATA[many]]></type>
                        <value><![CDATA[' . $this->value . ']]></value>
                        <value><![CDATA[' . $this->value2 . ']]></value>
                        <userType>
                            <identifier><![CDATA[' . $this->identifier . ']]></identifier>
                        </userType>
                        <applicationType>
                            <name><![CDATA[' . $this->name . ']]></name>
                        </applicationType>
                    </attributeType>
                </userAttributes>
            </userAttributeAdd>';
        }
    }

    public function testGetters()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody('json'),
            Request::class,
            'json'
        );

        $attributeType = $requestModel->getAttributeType()->get(0);

        $this->assertEquals($this->name, $attributeType->getName());
        $this->assertEquals($this->type, $attributeType->getType());
        $this->assertEquals($this->value, $attributeType->getValue());
        $this->assertEquals($this->identifier, $attributeType->getUserType()->getIdentifier());
        $this->assertEquals($this->name, $attributeType->getApplicationType()->getName());
    }
}
