<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserService\ActivateRequest;

use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\ActivateRequest;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;
use JMS\Serializer\SerializerInterface;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserService\ActivateRequest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var ActivateRequest\Request
     */
    private $request;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        $this->setUpRequest();
    }

    protected function setUpRequest()
    {
        $request = new ActivateRequest\Request();
        $userType1 = new ActivateRequest\UserType();
        $userType2 = new ActivateRequest\UserType();

        $userType1->setIdentifier($this->faker->email);
        $userType2->setIdentifier($this->faker->email);

        $request
            ->addUserType($userType1)
            ->addUserType($userType2);

        $this->request = $request;
    }

    public function testGetAndAddRoleTypes()
    {
        $request = new ActivateRequest\Request();
        $userType1 = new ActivateRequest\UserType();
        $userType2 = new ActivateRequest\UserType();

        $userType1Email = $this->faker->email;
        $userType1->setIdentifier($userType1Email);

        $userType2Email = $this->faker->email;
        $userType2->setIdentifier($userType2Email);

        $request->addUserType($userType1)
            ->addUserType($userType2);

        static::assertEquals($request->getUserTypes()->count(), 2);
        static::assertEquals($request->getUserTypes()->get(0)->getIdentifier(), $userType1Email);
        static::assertEquals($request->getUserTypes()->get(1)->getIdentifier(), $userType2Email);
    }

    public function testJSONSerialize()
    {
        static::assertJsonStringEqualsJsonString($this->serializer->serialize($this->request, 'json'),
            $this->getSerializedRequestContent('json'));
    }

    public function testXMLSerialize()
    {
        static::assertXmlStringEqualsXmlString($this->serializer->serialize($this->request, 'xml'),
            $this->getSerializedRequestContent('xml'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getSerializedRequestContent($type = 'xml')
    {
        switch ($type) {
            case 'json':
                return <<< JSON
{
	"userActivateRequest": [
		{
			"identifier": "{$this->request->getUserTypes()->get(0)->getIdentifier()}"
		},
		{
			"identifier": "{$this->request->getUserTypes()->get(1)->getIdentifier()}"
		}
	]
}
JSON;
            case 'xml':
                return <<< XML
<userServiceActivateRequest>
    <userActivateRequest>
        <userType>
            <identifier>{$this->request->getUserTypes()->get(0)->getIdentifier()}</identifier>
        </userType>
        <userType>
            <identifier>{$this->request->getUserTypes()->get(1)->getIdentifier()}</identifier>
        </userType>
    </userActivateRequest>
</userServiceActivateRequest>
XML;
            default:
                throw new \PHPUnit_Framework_Exception('Invalid request type (hint: not json/xml)');
        }
    }
}