<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Request\UserWs\Version1\UserService\PasswordConfirm;

use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\PasswordConfirm\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\PasswordConfirm\UserPasswordConfirm;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\UserService\PasswordConfirm\UserType;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $identifier;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $active;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $token;

    /**
     * @var string
     */
    private $password;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->identifier = $this->faker->email;
        $this->name = $this->faker->name;
        $this->active = $this->faker->boolean ? 1 : 0;
        $this->email = $this->faker->email;
        $this->token = $this->faker->sha256;
        $this->password = $this->faker->word;
    }

    public function testJsonSerialize()
    {
        $this->assertJsonStringEqualsJsonString($this->getRequestBody('json'), $this->setRequestBody('json'));
    }

    public function testXmlSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody('xml'), $this->setRequestBody('xml'));
    }

    public function testUserTypeGetter()
    {
        $request = new Request();
        $userPasswordConfirm = new UserPasswordConfirm();
        $userType = new UserType();

        $userType
            ->setEmail($this->email)
            ->setToken($this->token)
            ->setPassword($this->password);
        $userPasswordConfirm->addUserType($userType);

        $request->setUserPasswordConfirm($userPasswordConfirm);
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function setRequestBody($type = 'xml')
    {
        $request = new Request();
        $userPasswordConfirm = new UserPasswordConfirm();
        $userType = new UserType();

        $userType
            ->setEmail($this->email)
            ->setToken($this->token)
            ->setPassword($this->password);
        $userPasswordConfirm->addUserType($userType);

        $request->setUserPasswordConfirm($userPasswordConfirm);

        return $this->serializer->serialize($request, ($type !== 'json' ? 'xml' : 'json'));
    }

    /**
     * @param string $type (xml | json)
     * @return string
     */
    private function getRequestBody($type = 'xml')
    {
        if ($type === 'json') {
            return '{
                "userPasswordConfirm": {
                    "userType": [
                        {
                            "email": "' . $this->email . '",
                            "token": "' . $this->token . '",
                            "password": "' . $this->password . '"
                        }
                    ]
                }
            }';
        } else {
            return '<userServicePasswordConfirm>
                <userPasswordConfirm>
                    <userType>
                        <email><![CDATA[' . $this->email . ']]></email>
                        <token><![CDATA[' . $this->token . ']]></token>
                        <password><![CDATA[' . $this->password . ']]></password>
                    </userType>
                </userPasswordConfirm>
            </userServicePasswordConfirm>';
        }
    }

    public function testGetterUserType()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody('json'),
            Request::class,
            'json'
        );

        $userType = $requestModel->getUserPasswordConfirm()->getUserType()->get(0);

        $this->assertEquals($this->email, $userType->getEmail());
        $this->assertEquals($this->token, $userType->getToken());
        $this->assertEquals($this->password, $userType->getPassword());
    }

    public function testSetterGetterUserService()
    {
        $userPasswordConfirm = new UserPasswordConfirm();
        $userType = new UserType();

        $userType
            ->setEmail($this->email)
            ->setToken($this->token)
            ->setPassword($this->password);

        $userPasswordConfirm->addUserType($userType);

        $responseUserType = $userPasswordConfirm->getUserType()->get(0);

        $this->assertEquals($this->email, $responseUserType->getEmail());
        $this->assertEquals($this->token, $responseUserType->getToken());
        $this->assertEquals($this->password, $responseUserType->getPassword());
    }
}
