<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version1\Application\Show;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Show\Response as RestResponse;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Show\ApplicationType;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Sso\RestBundle\ModelRest\Common\Response\UserWs\ErrorCollector;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $description;

    /**
     * @var string
     */
    private $message;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->name = $this->faker->word;
        $this->description = $this->faker->sentence();
        $this->message = $this->faker->sentence();
    }

    public function testResponseJSONModel()
    {
        $this->assertJsonStringEqualsJsonString(
            $this->getSuccessResponse('json'),
            $this->getResponseModel('json')
        );
    }

    public function testResponseXMLModel()
    {
        $this->assertXmlStringEqualsXmlString(
            $this->getSuccessResponse('xml'),
            $this->getResponseModel('xml')
        );
    }

    public function testGetterSuccess()
    {
        /** @var  RestResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponse('json'),
            RestResponse::class,
            'json'
        );

        $this->assertEquals(200, $responseModel->getCode());
        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->scriptTimeSec, $responseModel->getScriptTimeSec());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());

        $applicationType = $responseModel->getApplicationType();
        $this->assertEquals(200, $applicationType->getCode());
        $this->assertEquals('OK', $applicationType->getStatus());
        $this->assertEquals($this->name, $applicationType->getName());
        $this->assertEquals($this->description, $applicationType->getDescription());
    }

    public function testSetterError()
    {
        $responseErrorModel = new RestResponse();
        $responseErrorModel
            ->setCode(400)
            ->setScriptTimeSec($this->scriptTimeSec)
            ->setDate(new \DateTime($this->date))
            ->setTrackId($this->trackId);

        $applicationType = new ApplicationType();
        $applicationType->setCode(400);

        $error = new ErrorCollector\Error();
        $error
            ->setCode(400)
            ->setMessage($this->message)
            ->setException('external');

        $applicationType->setError($error);
        $responseErrorModel->setApplicationType($applicationType);

        $this->assertJsonStringEqualsJsonString(
            $this->serializer->serialize($responseErrorModel, "json"),
            $this->getErrorResponse()
        );
    }

    public function testGetterError()
    {
        /** @var  RestResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponse('json'),
            RestResponse::class,
            'json'
        );

        $this->assertEquals(400, $responseModel->getCode());
        $this->assertEquals('Bad Request', $responseModel->getStatus());
        $this->assertEquals($this->scriptTimeSec, $responseModel->getScriptTimeSec());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());

        $applicationType = $responseModel->getApplicationType();
        $this->assertEquals(400, $applicationType->getCode());
        $this->assertEquals('Bad Request', $applicationType->getStatus());
        foreach ($applicationType->getError() as $error) {
            $this->assertEquals(400, $error->getCode());
            $this->assertEquals($this->message, $error->getMessage());
            $this->assertEquals('external', $error->getException());
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getResponseModel($type = 'json')
    {
        $response = new RestResponse();
        $response
            ->setCode(200)
            ->setStatus('OK')
            ->setTrackId($this->trackId)
            ->setDate(new \DateTime($this->date))
            ->setScriptTimeSec($this->scriptTimeSec);

        $response->setApplicationType($this->getApplicationType());
        $responseModel = $this->serializer->serialize($response, $type);

        return $responseModel;
    }

    /**
     * @return ApplicationType
     */
    private function getApplicationType()
    {
        $applicationType = new ApplicationType();
        $applicationType
            ->setCode(200)
            ->setStatus('OK');
        $applicationType
            ->setName($this->name)
            ->setDescription($this->description);

        return $applicationType;
    }

    /**
     * @param string $type
     * @return string
     */
    private function getSuccessResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                <application>
                  <code>200</code>
                  <status><![CDATA[OK]]></status>
                  <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                  <date><![CDATA[' . $this->date . ']]></date>
                  <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                  <application>
                    <code>200</code>
                    <status><![CDATA[OK]]></status>
                    <name><![CDATA[' . $this->name . ']]></name>
                    <description><![CDATA[' . $this->description . ']]></description>
                  </application>
                </application>';
        } else {
            return '{
              "code": 200,
              "status": "OK",
              "trackId": "' . $this->trackId . '",
              "date": "' . $this->date . '",
              "scriptTimeSec": ' . $this->scriptTimeSec . ',
              "application": {
                  "code": 200,
                  "status": "OK",
                  "name": "' . $this->name . '",
                  "description": "' . $this->description . '"
              }
            }';
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getErrorResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                      <application>
                          <code>400</code>
                          <status><![CDATA[Bad Request]]></status>
                          <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                          <date><![CDATA[' . $this->date . ']]></date>
                          <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                          <application>
                              <code>400</code>
                              <status><![CDATA[Bad Request]]></status>
                              <errors>
                                  <error>
                                      <code>400</code>
                                      <message><![CDATA[' . $this->message . ']]></message>
                                      <exception><![CDATA[external]]></exception>
                                  </error>
                              </errors>
                          </application>
                      </application>';
        } else {
            return '{
                      "code": 400,
                      "status": "Bad Request",
                      "trackId": "' . $this->trackId . '",
                      "date": "' . $this->date . '",
                      "scriptTimeSec": ' . $this->scriptTimeSec . ',
                      "application": {
                          "code": 400,
                          "status": "Bad Request",
                          "errors": [
                              {
                              "code": 400,
                              "message": "' . $this->message . '",
                              "exception": "external"
                              }
                          ]
                      }
                  }';
        }
    }
}
