<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelRest\Response\UserWs\Version1\UserService\PasswordRequest;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelRest\Common\Response\UserWs\ErrorCollector\Error;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserService\PasswordRequest\Response as RestResponse;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserService\PasswordRequest\Response;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserService\PasswordRequest\UserPasswordRequest;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\UserService\PasswordRequest\UserType;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $passwordToken;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
        $this->username = $this->faker->userName;
        $this->email = $this->faker->email;
        $this->passwordToken = $this->faker->sha256;
    }

    public function testResponseJSONModel()
    {
        $this->assertJsonStringEqualsJsonString($this->getSuccessResponse('json'), $this->getResponseModel('json'));
    }

    public function testResponseXMLModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse('xml'), $this->getResponseModel('xml'));
    }

    public function testWrongResponseJSONModel()
    {
        $this->assertJsonStringEqualsJsonString($this->getWrongResponse('json'), $this->getWrongResponseModel('json'));
    }

    public function testWrongResponseXMLModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getWrongResponse('xml'), $this->getWrongResponseModel('xml'));
    }

    private function getResponseModel($type = 'json')
    {
        $response = new RestResponse();
        $response
            ->setCode(200)
            ->setScriptTimeSec($this->scriptTimeSec)
            ->setTrackId($this->trackId)
            ->setDate(new \DateTime($this->date));
        $response->setUserPasswordRequest($this->getUserPasswordRequest());
        $responseModel = $this->serializer->serialize($response, $type);

        return $responseModel;
    }

    private function getWrongResponseModel($type = 'json')
    {
        $response = new RestResponse();
        $userPasswordRequest = new UserPasswordRequest();
        $userType = new UserType();
        $error = new Error();

        $error
            ->setCode(400)
            ->setMessage($this->message)
            ->setException('external');

        $userType
            ->setCode(400)
            ->setStatus('Bad Request');
        $userType->setError($error);

        $userPasswordRequest->addUserType($userType);

        $response
            ->setCode(400)
            ->setStatus('Bad Request')
            ->setScriptTimeSec($this->scriptTimeSec)
            ->setTrackId($this->trackId)
            ->setDate(new \DateTime($this->date));
        $response->setUserPasswordRequest($userPasswordRequest);

        $responseModel = $this->serializer->serialize($response, $type);

        return $responseModel;
    }

    /**
     * @return UserPasswordRequest
     */
    private function getUserPasswordRequest()
    {
        $userPasswordRequest = new UserPasswordRequest();
        $userType = new UserType();

        $userType
            ->setCode(200)
            ->setStatus('OK');
        $userType
            ->setUsername($this->username)
            ->setEmail($this->email)
            ->setPasswordToken($this->passwordToken)
            ->setPasswordTokenExpire($this->date);

        $userPasswordRequest->addUserType($userType);

        return $userPasswordRequest;
    }

    /**
     * @param string $type
     * @return string
     */
    private function getSuccessResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                <userServicePasswordRequest>
                  <code>200</code>
                  <status><![CDATA[OK]]></status>
                  <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                  <date><![CDATA[' . $this->date . ']]></date>
                  <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                  <userPasswordRequest>
                    <userType>
                        <code>200</code>
                        <status><![CDATA[OK]]></status>
                        <username><![CDATA[' . $this->username . ']]></username>
                        <email><![CDATA[' . $this->email . ']]></email>
                        <passwordToken><![CDATA[' . $this->passwordToken . ']]></passwordToken>
                        <passwordTokenExpire><![CDATA[' . $this->date . ']]></passwordTokenExpire>
                    </userType>
                  </userPasswordRequest>
                </userServicePasswordRequest>';
        } else {
            return '{
              "code": 200,
              "status": "OK",
              "trackId": "' . $this->trackId . '",
              "date": "' . $this->date . '",
              "scriptTimeSec": ' . $this->scriptTimeSec . ',
              "userPasswordRequest": {
                "userType": [
                    {
                      "code": 200,
                      "status": "OK",
                      "username": "' . $this->username . '",
                      "email": "' . $this->email . '",
                      "passwordToken": "' . $this->passwordToken . '",
                      "passwordTokenExpire": "' . $this->date . '"
                    }
                ]
              }
            }';
        }
    }

    /**
     * @param string $type
     * @return string
     */
    private function getWrongResponse($type = 'json')
    {
        if ($type === 'xml') {
            return '<?xml version="1.0" encoding="UTF-8"?>
                <userServicePasswordRequest>
                  <code>400</code>
                  <status><![CDATA[Bad Request]]></status>
                  <trackId><![CDATA[' . $this->trackId . ']]></trackId>
                  <date><![CDATA[' . $this->date . ']]></date>
                  <scriptTimeSec>' . $this->scriptTimeSec . '</scriptTimeSec>
                  <userPasswordRequest>
                      <userType>
                          <code>400</code>
                          <status><![CDATA[Bad Request]]></status>
                          <errors>
                              <error>
                                  <code>400</code>
                                  <message><![CDATA[' . $this->message . ']]></message>
                                  <exception><![CDATA[external]]></exception>
                              </error>
                          </errors>
                      </userType>
                  </userPasswordRequest>
                </userServicePasswordRequest>';
        } else {
            return '{
                "code": 400,
                "status": "Bad Request",
                "trackId": "' . $this->trackId . '",
                "date": "' . $this->date . '",
                "scriptTimeSec": ' . $this->scriptTimeSec . ',
                "userPasswordRequest": {
                      "userType": [
                          {
                              "code": 400,
                              "status": "Bad Request",
                              "errors": [
                                  {
                                      "code": 400,
                                      "message": "' . $this->message . '",
                                      "exception": "external"
                                  }
                              ]
                          }
                      ]
                }
            }';
        }
    }

    public function testGetters()
    {
        /** @var Response $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getResponseModel('json'),
            Response::class,
            'json'
        );

        $userType = $responseModel->getUserPasswordRequest()->getUserType()->get(0);

        $error = new Error();
        $error->setCode(400)->setMessage($this->message)->setException('external');
        $userType->setError($error);

        $errors = $userType->getErrors();
        $errorResponse = $errors->get(0);

        $this->assertEquals($this->username, $userType->getUsername());
        $this->assertEquals($this->email, $userType->getEmail());
        $this->assertEquals($this->passwordToken, $userType->getPasswordToken());
        $this->assertEquals(new \DateTime($this->date), $userType->getPasswordTokenExpire());
        $this->assertEquals(400, $errorResponse->getCode());
        $this->assertEquals($this->message, $errorResponse->getMessage());
        $this->assertEquals('external', $errorResponse->getException());
    }
}
