<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version1\UserApplicationAttribute\Add;

use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version1\UserApplicationAttribute\Add\Request;
use Sso\RestBundle\ModelXml\Request\UserWs\Version1\UserApplicationAttribute\Add\UserApplicationAttribute;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $serviceToken;

    /**
     * @var string
     */
    private $serviceName;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $langCode;

    /**
     * @var string
     */
    private $identifier;

    /**
     * @var string
     */
    private $appName;

    /**
     * @var string
     */
    private $attrName;

    /**
     * @var string
     */
    private $attrValue;

    /**
     * @var string
     */
    private $attrType;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->serviceToken = $this->faker->sha256;
        $this->serviceName = $this->faker->word;
        $this->userIdentifier = $this->faker->md5;
        $this->username = $this->faker->userName;
        $this->langCode = $this->faker->languageCode;
        $this->identifier = $this->faker->userName;
        $this->appName = $this->faker->word();
        $this->attrName = $this->faker->word();
        $this->attrType = $this->faker->word();
        $this->attrValue = $this->faker->word();
    }

    public function testSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody(), $this->setRequestBody());
    }

    /**
     * @return string
     */
    private function setRequestBody()
    {
        $request = new Request();
        $credentials = new Credentials();
        $userApplicationAttribute = new UserApplicationAttribute();

        $credentials->setCredentialsData(
            $this->serviceToken,
            $this->serviceName,
            $this->userIdentifier,
            $this->username,
            $this->langCode
        );

        $userApplicationAttribute->setUserApplicationAttribute(
            $this->identifier,
            $this->appName,
            $this->attrName,
            $this->attrType,
            $this->attrValue
        );

        $request->setCredentials($credentials);
        $request->setUserApplicationAttribute($userApplicationAttribute);

        return $this->serializer->serialize($request, 'xml');
    }

    /**
     * @return string
     */
    private function getRequestBody()
    {
        return '<SsoRequest>
            <Credentials>
                <ServiceProvider>
                    <ServiceToken>' . $this->serviceToken . '</ServiceToken>
                    <ServiceName>' . $this->serviceName . '</ServiceName>
                </ServiceProvider>
                <ServiceTrigger>
                    <UserIdentifier>' . $this->userIdentifier . '</UserIdentifier>
                    <Username>' . $this->username . '</Username>
                </ServiceTrigger>
                <ServiceCountry>
                    <Code>' . $this->langCode . '</Code>
                </ServiceCountry>
            </Credentials>
            <UserApplicationAttribute>
                <Add>
                    <Key>
                        <UserType>
                            <Identifier>' . $this->identifier . '</Identifier>
                        </UserType>
                        <ApplicationType>
                            <Name>' . $this->appName . '</Name>
                        </ApplicationType>
                    </Key>
                    <Value>
                        <AttributeType>
                            <Name>' . $this->attrName . '</Name>
                            <Type>' . $this->attrType . '</Type>
                            <Value>' . $this->attrValue . '</Value>
                        </AttributeType>
                    </Value>
                </Add>
            </UserApplicationAttribute>
        </SsoRequest>';
    }

    public function testGetters()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody(),
            Request::class,
            'xml'
        );

        $userApplicationAttribute = $requestModel->getUserApplicationAttribute()->getAdd();
        $userType = $userApplicationAttribute->getKey()->getUserType();
        $applicationType = $userApplicationAttribute->getKey()->getApplicationType();
        $attributeTypes = $userApplicationAttribute->getValue()->getAttributeTypes();
        $attributeType = reset($attributeTypes);

        $credentials = $requestModel->getCredentials();

        $this->assertEquals($this->identifier, $userType->getIdentifier());
        $this->assertEquals($this->appName, $applicationType->getName());
        $this->assertEquals($this->attrName, $attributeType->getName());
        $this->assertEquals($this->attrType, $attributeType->getType());
        $this->assertEquals($this->attrValue, $attributeType->getValue());
        $this->assertEquals($this->serviceToken, $credentials->getServiceProvider()->getServiceToken());
    }
}
