<?php

/**
 * Class RequestTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Request\UserWs\Version2\User\Add;

use Sso\RestBundle\ModelXml\Common\Request\Credentials\UserWs\Credentials;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\User\Add\Add;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\User\Add\Request;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\User\Add\User;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\User\Add\UserType;
use Sso\RestBundle\ModelXml\Request\UserWs\Version2\User\Add\Value;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Sso\RestBundle\Api\Manager as ApiManager;
use JMS\Serializer\SerializerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class RequestTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\RestBundle\Tests\ModelRest
 */
class RequestTest extends WebTestCase
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ApiManager
     */
    private $apiManager;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var string
     */
    private $serviceToken;

    /**
     * @var string
     */
    private $serviceName;

    /**
     * @var string
     */
    private $userIdentifier;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $firstName;

    /**
     * @var string
     */
    private $lastName;

    /**
     * @var string
     */
    private $password;

    /**
     * @var integer
     */
    private $active;

    /**
     * @var string
     */
    private $langCode;

    /**
     * @var string
     */
    private $authId;

    /**
     * @var integer
     */
    private $mfaEnabled;

    protected function setUp()
    {
        parent::setUp();
        static::bootKernel();
        $this->apiManager = new ApiManager(static::$kernel->getContainer());
        $this->serializer = $this->apiManager->serializer();
        $this->faker = FakerFactory::create();

        //set same random data
        $this->serviceToken = $this->faker->sha256;
        $this->serviceName = "Service";
        $this->userIdentifier = $this->faker->userName;
        $this->email = $this->faker->email;
        $this->username = $this->faker->userName;
        $this->firstName = $this->faker->firstName;
        $this->lastName = $this->faker->lastName;
        $this->password = 'd5zJwgnzU2uTtv97';
        $this->active = $this->faker->boolean ? 1 : 0;
        $this->langCode = $this->faker->languageCode;
        $this->authId = $this->faker->randomNumber();
        $this->mfaEnabled = $this->faker->boolean ? 1 : 0;
    }

    public function testSerialize()
    {
        $this->assertXmlStringEqualsXmlString($this->getRequestBody(), $this->setRequestBody());
    }

    /**
     * @return string
     */
    private function setRequestBody()
    {
        $request = new Request();
        $credentials = new Credentials();
        $user = new User();
        $add = new Add();
        $value = new Value();
        $userType = new UserType();

        $credentials->setCredentialsData(
            $this->serviceToken,
            $this->serviceName,
            $this->userIdentifier,
            $this->username,
            $this->langCode
        );

        $userType
            ->setEmail($this->email)
            ->setUsername($this->username)
            ->setFirstName($this->firstName)
            ->setLastName($this->lastName)
            ->setPassword($this->password)
            ->setActive($this->active)
            ->setAuthId($this->authId)
            ->setLdapSearchAttributes('')
            ->setLdapSearchValue('');
        $userType->setMfaEnabled($this->mfaEnabled);

        $value->setUserType($userType);
        $add->setValue($value);
        $user->setAdd($add);

        $request->setCredentials($credentials);
        $request->setUser($user);

        return $this->serializer->serialize($request, 'xml');
    }

    /**
     * @return string
     */
    private function getRequestBody()
    {
        return '<SsoRequest>
                <Credentials>
                    <ServiceProvider>
                        <ServiceToken><![CDATA[' . $this->serviceToken . ']]></ServiceToken>
                        <ServiceName><![CDATA[' . $this->serviceName . ']]></ServiceName>
                    </ServiceProvider>
                    <ServiceTrigger>
                        <UserIdentifier><![CDATA[' . $this->userIdentifier . ']]></UserIdentifier>
                        <Username><![CDATA[' . $this->username . ']]></Username>
                    </ServiceTrigger>
                    <ServiceCountry>
                        <Code><![CDATA[' . $this->langCode . ']]></Code>
                    </ServiceCountry>
                </Credentials>
                <User>
                <Add>
                    <Value>
                        <UserType>
                            <Email><![CDATA[' . $this->email . ']]></Email>
                            <Username><![CDATA[' . $this->username . ']]></Username>
                            <Firstname><![CDATA[' . $this->firstName . ']]></Firstname>
                            <Lastname><![CDATA[' . $this->lastName . ']]></Lastname>
                            <Password><![CDATA[' . $this->password . ']]></Password>
                            <Active>' . $this->active . '</Active>
                            <AuthId><![CDATA[' . $this->authId . ']]></AuthId>
                            <LdapSearchAttributes></LdapSearchAttributes>
                            <LdapSearchValue></LdapSearchValue>
                            <MfaEnabled>' . $this->mfaEnabled . '</MfaEnabled>
                        </UserType>
                    </Value>
                </Add>
            </User>
       </SsoRequest>';
    }

    public function testSettersGetters()
    {
        /** @var  Request $requestModel */
        $requestModel = $this->serializer->deserialize(
            $this->setRequestBody(),
            Request::class,
            'xml'
        );

        $user = $requestModel->getUser();
        $add = $user->getAdd();
        $value = $add->getValue();
        $userType = $value->getUserType();
        $user->setUserData(
            $this->email,
            $this->username,
            $this->firstName,
            $this->lastName,
            $this->password,
            $this->active,
            $this->authId,
            '',
            '',
            $this->mfaEnabled
        );

        $credentials = $requestModel->getCredentials();

        $this->assertEquals($this->email, $userType->getEmail());
        $this->assertEquals($this->username, $userType->getUsername());
        $this->assertEquals($this->firstName, $userType->getFirstName());
        $this->assertEquals($this->lastName, $userType->getLastName());
        $this->assertEquals($this->password, $userType->getPassword());
        $this->assertEquals($this->active, $userType->getActive());
        $this->assertEquals($this->authId, $userType->getAuthId());
        $this->assertEquals($this->mfaEnabled, $userType->getMfaEnabled());
        $this->assertEquals($this->serviceToken, $credentials->getServiceProvider()->getServiceToken());
    }
}
