<?php

/**
 * Class ResponseTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Response\ObjectWs\Version1\Object\Update;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Error\Error;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Success\Update as SuccessUpdate;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Success\Response as SuccessXmlResponse;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Success\Object as SuccessObject;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Success\ObjectType as SuccessObjectType;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Error\Update as ErrorUpdate;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Error\Response as ErrorXmlResponse;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\Object\Update\Error\Object as ErrorObject;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var ContainerInterface
     */
    private $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    private $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $description;

    /**
     * @var string
     */
    private $referenceId;

    /**
     * @var string
     */
    private $typeId;

    /**
     * @var string
     */
    private $guid;

    /**
     * @var int
     */
    private $active;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
        $this->name = $this->faker->name;
        $this->description = $this->faker->sentence();
        $this->referenceId = $this->faker->md5;
        $this->guid = $this->faker->uuid;
        $this->typeId = $this->faker->uuid;
        $this->active = $this->faker->boolean ? 1 : 0;
    }

    public function testSuccessResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse(), $this->getSuccessResponseModel());
    }

    public function testErrorResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getErrorResponse(), $this->getErrorResponseModel());
    }

    private function getSuccessResponseModel()
    {
        $response = new SuccessXmlResponse();
        $response->setStatus('OK');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $object = new SuccessObject();
        $objectType = new SuccessObjectType();
        $update = new SuccessUpdate();

        $objectType
            ->setGuid($this->guid)
            ->setReferenceId($this->referenceId)
            ->setTypeId($this->typeId)
            ->setParentGuid($this->guid)
            ->setName($this->name)
            ->setCreated(new \DateTime($this->date))
            ->setUpdated(new \DateTime($this->date))
            ->setActive($this->active);

        $update
            ->setStatus('OK')
            ->setMessage($this->message)
            ->setObjectType($objectType);

        $object->setUpdate($update);

        $response->setObject($object);

        return $this->serializer->serialize($response, 'xml');
    }

    private function getErrorResponseModel()
    {
        $response = new ErrorXmlResponse();
        $response->setStatus('Failure');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $object = new ErrorObject();
        $update = new ErrorUpdate();
        $error = new Error();

        $error->setErrorCode('fw003');
        $error->setErrorRef('hd007');
        $error->setShortMessage('InvalidXml');
        $error->setLongMessage($this->message);

        $update->setStatus('Failure')->setError($error);
        $object->setUpdate($update);

        $response->setObject($object);

        return $this->serializer->serialize($response, 'xml');
    }

    /**
     * @return string
     */
    private function getSuccessResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[OK]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <Object>
                    <Update>
                        <Status><![CDATA[OK]]></Status>
                        <Message><![CDATA[' . $this->message . ']]></Message>
                        <ObjectType>
                            <Guid><![CDATA[' . $this->guid . ']]></Guid>
                            <ReferenceId><![CDATA[' . $this->referenceId . ']]></ReferenceId>
                            <TypeId><![CDATA[' . $this->typeId . ']]></TypeId>
                            <ParentGuid><![CDATA[' . $this->guid . ']]></ParentGuid>
                            <Name><![CDATA[' . $this->name . ']]></Name>
                            <Created><![CDATA[' . $this->date . ']]></Created>
                            <Updated><![CDATA[' . $this->date . ']]></Updated>
                            <Active>' . $this->active . '</Active>
                        </ObjectType>
                    </Update>
                </Object>
            </SsoResponse>';
    }

    /**
     * @return string
     */
    private function getErrorResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[Failure]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <Object>
                    <Update>
                        <Status><![CDATA[Failure]]></Status>
                        <Errors>
                            <Error>
                                <ErrorCode><![CDATA[fw003]]></ErrorCode>
                                <ErrorRef><![CDATA[hd007]]></ErrorRef>
                                <ShortMessage><![CDATA[InvalidXml]]></ShortMessage>
                                <LongMessage><![CDATA[' . $this->message . ']]></LongMessage>    
                            </Error>
                        </Errors>
                    </Update>
                </Object>
            </SsoResponse>';
    }

    public function testSuccessGetters()
    {
        /** @var SuccessXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponseModel(),
            SuccessXmlResponse::class,
            'xml'
        );

        $objectType = $responseModel->getObject()->getUpdate()->getObjectType();

        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('OK', $responseModel->getObject()->getUpdate()->getStatus());
        $this->assertEquals($this->message, $responseModel->getObject()->getUpdate()->getMessage());

        $this->assertEquals($this->guid, $objectType->getGuid());
        $this->assertEquals($this->referenceId, $objectType->getReferenceId());
        $this->assertEquals($this->typeId, $objectType->getTypeId());
        $this->assertEquals($this->guid, $objectType->getParentGuid());
        $this->assertEquals($this->name, $objectType->getName());
        $this->assertEquals(new \DateTime($this->date), $objectType->getCreated());
        $this->assertEquals(new \DateTime($this->date), $objectType->getUpdated());
        $this->assertEquals($this->active, $objectType->getActive());
    }

    public function testErrorGetters()
    {
        /** @var ErrorXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponseModel(),
            ErrorXmlResponse::class,
            'xml'
        );

        $error = $responseModel->getObject()->getUpdate()->getErrors()->get(0);

        $this->assertEquals('Failure', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('Failure', $responseModel->getObject()->getUpdate()->getStatus());
        $this->assertEquals('fw003', $error->getErrorCode());
        $this->assertEquals('hd007', $error->getErrorRef());
        $this->assertEquals('InvalidXml', $error->getShortMessage());
        $this->assertEquals($this->message, $error->getLongMessage());
    }
}
