<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Response\UserWs\Version1\Application\Show;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Error\Error;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Success\ApplicationType;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Success\Show as SuccessShow;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Success\Response as SuccessXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Success\Application as SuccessApplication;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Error\Show as ErrorShow;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Error\Response as ErrorXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Show\Error\Application as ErrorApplication;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $name;

    /**
     * @var string
     */
    private $description;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
        $this->name = $this->faker->word;
        $this->description = $this->faker->sentence();
    }

    public function testSuccessResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse(), $this->getSuccessResponseModel());
    }

    public function testErrorResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getErrorResponse(), $this->getErrorResponseModel());
    }

    private function getSuccessResponseModel()
    {
        $response = new SuccessXmlResponse();
        $response->setStatus('OK');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $applicationRole = new SuccessApplication();
        $applicationRoleType = new ApplicationType();
        $show = new SuccessShow();

        $applicationRoleType
            ->setName($this->name)
            ->setDescription($this->description);

        $show
            ->setStatus('OK')
            ->setApplicationType($applicationRoleType);

        $applicationRole->setShow($show);
        $response->setApplication($applicationRole);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    private function getErrorResponseModel()
    {
        $response = new ErrorXmlResponse();
        $response->setStatus('Failure');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $applicationRole = new ErrorApplication();
        $show = new ErrorShow();
        $error = new Error();

        $error->setErrorCode('fw003');
        $error->setErrorRef('hd007');
        $error->setShortMessage('InvalidXml');
        $error->setLongMessage($this->message);

        $show->setStatus('Failure')->setError($error);
        $applicationRole->setShow($show);

        $response->setApplication($applicationRole);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    /**
     * @return string
     */
    private function getSuccessResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[OK]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <Application>
                    <Show>
                        <Status><![CDATA[OK]]></Status>
                        <ApplicationType>
                            <Name><![CDATA[' . $this->name . ']]></Name>
                            <Description><![CDATA[' . $this->description . ']]></Description>                        
                        </ApplicationType>
                    </Show>
                </Application>
            </SsoResponse>';
    }

    /**
     * @return string
     */
    private function getErrorResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[Failure]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <Application>
                    <Show>
                        <Status><![CDATA[Failure]]></Status>
                        <Errors>
                            <Error>
                                <ErrorCode><![CDATA[fw003]]></ErrorCode>
                                <ErrorRef><![CDATA[hd007]]></ErrorRef>
                                <ShortMessage><![CDATA[InvalidXml]]></ShortMessage>
                                <LongMessage><![CDATA[' . $this->message . ']]></LongMessage>    
                            </Error>
                        </Errors>
                    </Show>
                </Application>
            </SsoResponse>';
    }

    public function testSuccessGetters()
    {
        /** @var SuccessXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponseModel(),
            SuccessXmlResponse::class,
            'xml'
        );

        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('OK', $responseModel->getApplication()->getShow()->getStatus());
        $this->assertEquals($this->name, $responseModel->getApplication()->getShow()->getApplicationType()->getName());
        $this->assertEquals($this->description,
            $responseModel->getApplication()->getShow()->getApplicationType()->getDescription());
    }

    public function testErrorGetters()
    {
        /** @var ErrorXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponseModel(),
            ErrorXmlResponse::class,
            'xml'
        );

        $error = $responseModel->getApplication()->getShow()->getErrors()->get(0);

        $this->assertEquals('Failure', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('Failure', $responseModel->getApplication()->getShow()->getStatus());
        $this->assertEquals('fw003', $error->getErrorCode());
        $this->assertEquals('hd007', $error->getErrorRef());
        $this->assertEquals('InvalidXml', $error->getShortMessage());
        $this->assertEquals($this->message, $error->getLongMessage());
    }
}
