<?php

/**
 * Class ResponseTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Error\Error;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Success\Add as SuccessAdd;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Success\Response as SuccessXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Success\ApplicationAttribute as SuccessApplicationAttribute;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Error\Add as ErrorAdd;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Error\Response as ErrorXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\ApplicationAttribute\Add\Error\ApplicationAttribute as ErrorApplicationAttribute;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;


    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->message = $this->faker->sentence(2);
    }

    public function testSuccessResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse(), $this->getSuccessResponseModel());
    }

    public function testErrorResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getErrorResponse(), $this->getErrorResponseModel());
    }

    private function getSuccessResponseModel()
    {
        $response = new SuccessXmlResponse();
        $response->setStatus('OK');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $applicationAttribute = new SuccessApplicationAttribute();
        $add = new SuccessAdd();

        $add->setStatus('OK')->setMessage($this->message);

        $applicationAttribute->setAdd($add);

        $response->setApplicationAttribute($applicationAttribute);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    private function getErrorResponseModel()
    {
        $response = new ErrorXmlResponse();
        $response->setStatus('Failure');
        $response->setTrackId($this->trackId);
        $response->setDate(new \DateTime($this->date));

        $applicationAttribute = new ErrorApplicationAttribute();
        $add = new ErrorAdd();
        $error = new Error();

        $error->setErrorCode('fw003');
        $error->setErrorRef('hd007');
        $error->setShortMessage('InvalidXml');
        $error->setLongMessage($this->message);

        $add->setStatus('Failure')->setError($error);
        $applicationAttribute->setAdd($add);

        $response->setApplicationAttribute($applicationAttribute);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    /**
     * @return string
     */
    private function getSuccessResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[OK]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <ApplicationAttribute>
                    <Add>
                        <Status><![CDATA[OK]]></Status>
                        <Message><![CDATA[' . $this->message . ']]></Message>
                    </Add>
                </ApplicationAttribute>
            </SsoResponse>';
    }

    /**
     * @return string
     */
    private function getErrorResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[Failure]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <ApplicationAttribute>
                    <Add>
                        <Status><![CDATA[Failure]]></Status>
                        <Errors>
                            <Error>
                                <ErrorCode><![CDATA[fw003]]></ErrorCode>
                                <ErrorRef><![CDATA[hd007]]></ErrorRef>
                                <ShortMessage><![CDATA[InvalidXml]]></ShortMessage>
                                <LongMessage><![CDATA[' . $this->message . ']]></LongMessage>    
                            </Error>
                        </Errors>
                    </Add>
                </ApplicationAttribute>
            </SsoResponse>';
    }

    public function testSuccessGetters()
    {
        /** @var SuccessXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponseModel(),
            SuccessXmlResponse::class,
            'xml'
        );

        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('OK', $responseModel->getApplicationAttribute()->getAdd()->getStatus());
        $this->assertEquals($this->message, $responseModel->getApplicationAttribute()->getAdd()->getMessage());
    }

    public function testErrorGetters()
    {
        /** @var ErrorXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponseModel(),
            ErrorXmlResponse::class,
            'xml'
        );

        $error = $responseModel->getApplicationAttribute()->getAdd()->getErrors()->get(0);

        $this->assertEquals('Failure', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('Failure', $responseModel->getApplicationAttribute()->getAdd()->getStatus());
        $this->assertEquals('fw003', $error->getErrorCode());
        $this->assertEquals('hd007', $error->getErrorRef());
        $this->assertEquals('InvalidXml', $error->getShortMessage());
        $this->assertEquals($this->message, $error->getLongMessage());
    }
}
