<?php

/**
 * Class ResponseTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Tests\ModelXml\Response\UserWs\Version1\User\Add;

use Sso\RestBundle\Api;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Error\Error;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Success\Add as SuccessAdd;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Success\Response as SuccessXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Success\User as SuccessUser;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Error\Add as ErrorAdd;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Error\Response as ErrorXmlResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Error\User as ErrorUser;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Add\Success\UserType as SuccessUserType;
use GuzzleHttp\Client;
use Symfony\Bundle\FrameworkBundle\Test\WebTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Faker\Factory as FakerFactory;
use Faker\Generator as FakerGenerator;

/**
 * Class ResponseTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */
class ResponseTest extends WebTestCase
{
    /**
     * @var Api\Manager
     */
    public $apiM;

    /**
     * @var Client
     */
    protected $client;

    /**
     * @var ContainerInterface
     */
    protected $container;

    /**
     * @var \JMS\Serializer\Serializer
     */
    protected $serializer;

    /**
     * @var FakerGenerator
     */
    private $faker;

    /**
     * @var float
     */
    private $scriptTimeSec;

    /**
     * @var string
     */
    private $trackId;

    /**
     * @var string
     */
    private $date;

    /**
     * @var string
     */
    private $message;

    /**
     * @var string
     */
    private $guid;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $firstName;

    /**
     * @var string
     */
    private $lastName;

    /**
     * @var string
     */
    private $password;

    /**
     * @var boolean
     */
    private $active;

    /**
     * @var string
     */
    private $authId;

    protected function setUp()
    {
        $this->client = static::createClient();
        $this->container = $this->client->getContainer();
        $this->apiM = new Api\Manager($this->container);
        $this->serializer = $this->apiM->serializer();
        $this->faker = FakerFactory::create();

        // Set same random data
        $this->scriptTimeSec = $this->faker->randomFloat(2);
        $this->trackId = $this->faker->md5;
        $this->date = '2099-01-01T00:00:00+00:00';
        $this->guid = $this->faker->md5;
        $this->email = $this->faker->email;
        $this->username = $this->faker->userName;
        $this->firstName = $this->faker->firstName;
        $this->lastName = $this->faker->lastName;
        $this->password = 'd5zJwgnzU2uTtv97';
        $this->active = $this->faker->boolean ? 1 : 0;
        $this->authId = $this->faker->randomNumber();
        $this->message = $this->faker->sentence(2);
    }

    public function testSuccessResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getSuccessResponse(), $this->getSuccessResponseModel());
    }

    public function testErrorResponseModel()
    {
        $this->assertXmlStringEqualsXmlString($this->getErrorResponse(), $this->getErrorResponseModel());
    }

    private function getSuccessResponseModel()
    {
        $response = new SuccessXmlResponse();
        $response->setStatus('OK');
        $response->setTrackId($this->trackId);
        $response->setDate($this->date);

        $user = new SuccessUser();
        $add = new SuccessAdd();
        $userType = new SuccessUserType();

        $userType
            ->setGuid($this->guid)
            ->setUsername($this->username)
            ->setEmail($this->email)
            ->setFirstName($this->firstName)
            ->setLastName($this->lastName)
            ->setActive($this->active)
            ->setAuthId($this->authId)
            ->setLdapSearchAttributes('')
            ->setLdapSearchValue('');

        $add
            ->setStatus('OK')
            ->setMessage($this->message)
            ->setUserType($userType);

        $user->setAdd($add);
        $response->setUser($user);
        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    private function getErrorResponseModel()
    {
        $response = new ErrorXmlResponse();
        $response->setStatus('Failure');
        $response->setTrackId($this->trackId);
        $response->setDate($this->date);

        $user = new ErrorUser();
        $add = new ErrorAdd();
        $error = new Error();

        $error->setErrorCode('fw003');
        $error->setErrorRef('hd007');
        $error->setShortMessage('InvalidXml');
        $error->setLongMessage($this->message);

        $add
            ->setStatus('Failure')
            ->setError($error);

        $user->setAdd($add);
        $response->setUser($user);

        $responseModel = $this->serializer->serialize($response, 'xml');

        return $responseModel;
    }

    /**
     * @return string
     */
    private function getSuccessResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[OK]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <User>
                    <Add>
                        <Status><![CDATA[OK]]></Status>
                        <Message><![CDATA[' . $this->message . ']]></Message>
                        <UserType>
                            <Guid><![CDATA[' . $this->guid . ']]></Guid>
                            <Username><![CDATA[' . $this->username . ']]></Username>
                            <Email><![CDATA[' . $this->email . ']]></Email>
                            <Firstname><![CDATA[' . $this->firstName . ']]></Firstname>
                            <Lastname><![CDATA[' . $this->lastName . ']]></Lastname>
                            <Active>' . $this->active . '</Active>
                            <AuthId><![CDATA[' . $this->authId . ']]></AuthId>
                            <LdapSearchAttributes></LdapSearchAttributes>
                            <LdapSearchValue></LdapSearchValue>
                        </UserType>
                    </Add>
                </User>
            </SsoResponse>';
    }

    /**
     * @return string
     */
    private function getErrorResponse()
    {
        return '<?xml version="1.0" encoding="utf-8"?>
            <SsoResponse>
                <Status><![CDATA[Failure]]></Status>
                <Trackid><![CDATA[' . $this->trackId . ']]></Trackid>
                <Date><![CDATA[' . $this->date . ']]></Date>
                <User>
                    <Add>
                        <Status><![CDATA[Failure]]></Status>
                        <Errors>
                            <Error>
                                <ErrorCode><![CDATA[fw003]]></ErrorCode>
                                <ErrorRef><![CDATA[hd007]]></ErrorRef>
                                <ShortMessage><![CDATA[InvalidXml]]></ShortMessage>
                                <LongMessage><![CDATA[' . $this->message . ']]></LongMessage>    
                            </Error>
                        </Errors>
                    </Add>
                </User>
            </SsoResponse>';
    }

    public function testSuccessGetters()
    {
        /** @var SuccessXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getSuccessResponseModel(),
            SuccessXmlResponse::class,
            'xml'
        );

        $add = $responseModel->getUser()->getAdd();
        $userType = $add->getUserType();

        $this->assertEquals('OK', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('OK', $add->getStatus());
        $this->assertEquals($this->message, $add->getMessage());
        $this->assertEquals($this->guid, $userType->getGuid());
    }

    public function testErrorGetters()
    {
        /** @var ErrorXmlResponse $responseModel */
        $responseModel = $this->serializer->deserialize(
            $this->getErrorResponseModel(),
            ErrorXmlResponse::class,
            'xml'
        );

        $error = $responseModel->getUser()->getAdd()->getErrors()->get(0);

        $this->assertEquals('Failure', $responseModel->getStatus());
        $this->assertEquals($this->trackId, $responseModel->getTrackId());
        $this->assertEquals(new \DateTime($this->date), $responseModel->getDate());
        $this->assertEquals('Failure', $responseModel->getUser()->getAdd()->getStatus());
        $this->assertEquals('fw003', $error->getErrorCode());
        $this->assertEquals('hd007', $error->getErrorRef());
        $this->assertEquals('InvalidXml', $error->getShortMessage());
        $this->assertEquals($this->message, $error->getLongMessage());
    }
}
