<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\ObjectWs\Version1\AttributeGroup\Add;

use Sso\RestBundle\Api\ExceptionCollection\MainValidationException;
use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use GuzzleHttp\Exception\RequestException;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\AttributeGroup\Add\Request as RequestModel;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\AttributeGroup\Add\AttributeGroupType as RequestAttributeGroupType;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\AttributeGroup\Add\AttributeGroupType as ResponseAttributeGroupType;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\AttributeGroup\Add\Response as RestResponseModel;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Success\Response as SuccessXMLResponse;
use Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\AttributeGroup\Add\Error\Response as ErrorXMLResponse;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\ObjectWs\Version1\AttributeGroup\Add
 */
class Handler
{
    const STATUS = "status";
    const ERROR_EXTERNAL = "external";

    /**
     * @var UidentifyToken
     */
    private $userToken;

    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var ErrorApi
     */
    private $errorApi;

    /**
     * @var Errors
     */
    private $errors;

    /**
     * @var ErrorXMLResponse
     */
    private $errorXmlEntity;

    /**
     * @var SuccessXMLResponse
     */
    private $successXmlEntity;

    /**
     * @var RestResponseModel
     */
    private $restResponseModel;

    /**
     * @var bool
     */
    private $oneResponseSuccess = false;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->restResponseModel = $this->apiM->modelRest()->response()->objectWs()->version1()->attributeGroup()->add()->response();
        $this->userToken = $this->apiM->getToken();
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate main if there is any error it is allowed to use the default error response
        try {
            list($requestContent, $format) = $this->apiM->mainValidator()->validateRestRequest($request);
        } catch (MainValidationException $exc) {
            $this->errorApi->error()->errors()->setScriptSeconds($this->apiM->scriptTimeSeconds());

            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors(), 416);
        }

        // second: setting the request model
        $requestModelRest = $this->apiM->serializer()->deserialize($requestContent, RequestModel::class, $format);
//die(var_dump($requestContent));
        // loop the attributeGroupType
        foreach ($requestModelRest->getAttributeGroupType() as $attributeGroupType) {
            $responseAttributeGroupType = $this->sendXmlRequest($attributeGroupType);
            $this->restResponseModel->addAttributeGroupType($responseAttributeGroupType);
        }

        // reset the main code status
        $this->restResponseModel->setCode($this->oneResponseSuccess ? 200 : 400);

        return $this->apiM->restView()->setData($this->restResponseModel)->setStatusCode($this->oneResponseSuccess ? 200 : 400);
    }

    /**
     * @param RequestAttributeGroupType $requestAttributeGroupType
     * @return ResponseAttributeGroupType
     */
    private function sendXmlRequest(RequestAttributeGroupType $requestAttributeGroupType)
    {
        $requestModel = $this->apiM->modelXml()->request()->objectWs()->version1()->attributeGroup()->add()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->attributeGroup()->add()->credentials();
        $credentialsModelXml->setCredentialsData(
            $this->userToken->servicetoken,
            'AttributeGroup',
            $this->userToken->useridentifier,
            $this->userToken->username,
            'en'
        );
        $requestModel->setCredentials($credentialsModelXml);

        // add attribute group
        $objectModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->attributeGroup()->add()->attributeGroup();
        $objectModelXml->setAttributeGroup(
            $requestAttributeGroupType->getName(),
            $requestAttributeGroupType->getType()
        );
        $requestModel->setAttributeGroup($objectModelXml);

        // send request
        try {
            $clientResponse = $this->apiM->client()->post(
                $this->apiM->configuration()->objectWs()->getBaseUrl() . $this->apiM->configuration()->objectWs()->getServiceAction(),
                $this->apiM->clientHelper()->options()->getOptions(1, 'AttributeGroup', 'Add',
                    $this->apiM->serializer()->serialize($requestModel, 'xml'))
            );
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem([static::STATUS => false, 'xml' => $errorResponse]);
        }

        return $this->buildItem([static::STATUS => true, 'xml' => $clientResponse->getBody()->getContents()]);
    }

    /**
     * @param string $xmlResponse
     * @return ResponseAttributeGroupType
     */
    private function buildItem($xmlResponse)
    {
        $responseItem = $this->apiM->modelRest()->response()->objectWs()->version1()->attributeGroup()->add()->attributeGroupType();

        if (!$xmlResponse[static::STATUS]) {
            // build error response for a attributeGroupType
            $view = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for a attributeGroupType
            $view = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }

        return $view;
    }

    /**
     * @param string $xmlString
     * @param ResponseAttributeGroupType $responseItem
     * @return ResponseAttributeGroupType
     */
    private function buildSuccess($xmlString, ResponseAttributeGroupType $responseItem)
    {
        $this->successXmlEntity = $this->apiM->serializer()->deserialize($xmlString, SuccessXMLResponse::class, 'xml');

        if (!$this->successXmlEntity->getAttributeGroup()
            || ($this->successXmlEntity->getAttributeGroup()
                && $this->successXmlEntity->getAttributeGroup()->getAdd()->getStatus() != 'Success')
        ) {
            return $this->buildError($xmlString, $responseItem);
        }

        $this->oneResponseSuccess = true;

        /**
         * @var ResponseAttributeGroupType $responseItem
         */
        $responseItem = $this->apiM->helper()->mapper()->modelInToModelOut(
            $this->successXmlEntity->getAttributeGroup()->getAdd()->getAttributeGroupType(),
            $responseItem
        );

        // Add additional fields to response
        $this->restResponseModel
            ->setCode(200)
            ->setTrackId($this->successXmlEntity->getTrackId())
            ->setDate($this->successXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to success response
        $responseItem->setCode(200);

        return $responseItem;
    }

    /**
     * @param ResponseAttributeGroupType $responseItem
     * @param string $xmlString
     * @return ResponseAttributeGroupType
     */
    private function buildError($xmlString, ResponseAttributeGroupType $responseItem)
    {
        $this->errorXmlEntity = $this->apiM->serializer()->deserialize($xmlString, ErrorXMLResponse::class, 'xml');
        $errors = $this->errorXmlEntity->getAttributeGroup()->getAdd()->getErrors();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->objectWs()->version1()->attributeGroup()->add()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setMessage($message);
            $errorItem->setCode(400);
            $errorItem->setException(static::ERROR_EXTERNAL);
            $responseItem->setError($errorItem);
        }

        // Add additional fields to response
        $this->restResponseModel
            ->setCode(400)
            ->setTrackId($this->errorXmlEntity->getTrackId())
            ->setDate($this->errorXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to error response
        $responseItem->setCode(400);

        return $responseItem;
    }
}
