<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\ObjectWs\Version1\ObjectType\Add;

use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use LifeStyle\Tools\RestErrorBundle\Api as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use Symfony\Component\HttpFoundation\Request;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectType\Add\Request as RequestModel;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\ObjectWs\Version1\ObjectType\Add
 */
class Handler
{
    /**
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     * @var ApiManager
     */
    protected $apiM;

    /**
     * @var ErrorApi
     */
    protected $errorApi;

    /**
     * @var Errors
     */
    protected $errors;

    /**
     * @var \Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectType\Add\ObjectTypeAdd\Error\Response
     */
    protected $errorXmlObjectAddEntity;

    /**
     * @var \Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectType\Add\ObjectTypeAdd\Success\Response
     */
    protected $successXmlObjectAddEntity;

    /**
     * holds the complete response errors and success for every item
     * @var \Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\ObjectType\Add\Response
     */
    protected $restResponseEntity;

    /**
     * @var string
     */
    protected $format;

    /**
     * @var Utils\Factory
     */
    protected $utilsFactory;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->format = 'json';
        $this->userToken = $this->apiM->getToken();
        $this->restResponseEntity = $this->apiM->modelRest()->response()->objectWs()->version1()->objectType()->add()->response();
        $this->utilsFactory = new Utils\Factory($this->apiM);
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate input data
        $requestModelRest = $this->validateRestRequest($request);

        if ($this->errors->hasErrors()) {
            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors());
        }

        // second: send data to object ws ObjectTypeAdd
        foreach ($requestModelRest->getItem() as $item) {
            $responseItem = $this->utilsFactory->objectTypeAdd()->send($item);

            //now add the current item to the response
            $this->restResponseEntity->addItem($responseItem);
        }

        $view = RestView::create();
        $view->setData($this->restResponseEntity)->setStatusCode(200);

        return $view;
    }

    /**
     * @param Request $request
     * @return RequestModel
     */
    private function validateRestRequest(Request $request)
    {
        // check content
        $requestContent = $request->getContent();
        if (empty($requestContent)) {
            $this->errors->addError(204, 'No Content found in your request', 'external')->setCode(204);
            return;
        }

        $this->format = preg_match('/\.xml/i', $request->getUri()) ? 'xml' : 'json';

        //no validate the content
        if ($this->format == "xml") {
            $errors = $this->apiM->helper()->string()->validateXML($requestContent);
        } else {
            $errors = $this->apiM->helper()->string()->validateJson($requestContent);
        }
        // collect errors
        foreach ($errors as $error) {
            $this->errors->addError(400, $error, 'external', 'InvalidRequest')->setCode(400);
            return;
        }

        //now build request model
        $requestModel = $this->apiM->serializer()->deserialize($requestContent,
            'Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectType\Add\Request', $this->format);

        $validationErrors = $this->apiM->validator()->validate($requestModel);

        if (count($validationErrors) > 0) {
            foreach ($validationErrors as $error) {
                $message = $error->getPropertyPath() . ": " . $error->getMessage();
                $this->errors->addError(400, $message, 'external')->setCode(400);
            }
        }
        return $requestModel;
    }
}
