<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\ObjectWs\Version1\ObjectValue\Delete;

use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectValue\Delete\Request as RequestModel;
use Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectValue\Delete\Item as RequestItem;
use Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\ObjectValue\Delete\Item as ResponseItem;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\Client;

/**
 * Class Handler
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\RestBundle\Worker\ObjectWs\Version1\ObjectValue\DeleteObjectValue
 */
class Handler
{
    /**
     *
     * @var UidentifyToken
     */
    protected $userToken;

    /**
     *
     * @var ApiManager
     */
    protected $apiM;

    /**
     *
     * @var ErrorApi
     */
    protected $errorApi;

    /**
     *
     * @var Errors
     */
    protected $errors;

    /**
     *
     * @var \Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectValue\Delete\ObjectValueDelete\Error\Response
     */
    protected $errorXmlObjectValue;

    /**
     *
     * @var \Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectValue\Delete\ObjectValueDelete\Success\Response
     */
    protected $successXmlObjectValue;

    /**
     *
     * @var string
     */
    protected $format;

    /**
     * @var Client
     */
    protected $client;

    /**
     * holds the complete response errors and success for every item
     * @var \Sso\RestBundle\ModelRest\Response\ObjectWs\Version1\ObjectValue\Delete\Response
     */
    protected $restResponseEntity;

    /**
     *
     * @param ApiManager $apiM
     * @param Client $client
     */
    public function __construct(ApiManager $apiM, Client $client)
    {
        $this->apiM = $apiM;
        $this->client = $client;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->format = 'json';
        $this->userToken = $this->apiM->getToken();
        $this->restResponseEntity = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->delete()->response();
    }

    /**
     *
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate input data
        $requestModelRest = $this->validateRestRequest($request);

        if ($this->errors->hasErrors()) {
            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors());
        }

        // second: send data to object ws
        foreach ($requestModelRest->getItem() as $item) {
            $responseItem = $this->send($item);

            //now add the current item to the response
            $this->restResponseEntity->addItem($responseItem);

        }

        $view = RestView::create();
        $view->setData($this->restResponseEntity)->setStatusCode(200);
        return $view;
    }

    /**
     * @param Request $request
     * @return RequestModel
     */
    private function validateRestRequest(Request $request)
    {
        // check content
        $requestContent = $request->getContent();

        if (empty($requestContent)) {
            $this->errors->addError(204, 'No Content found in your request', 'external')->setCode(204);
            return;
        }

        $this->format = preg_match('/\.xml/i', $request->getUri()) ? 'xml' : 'json';

        //no validate the content
        if ($this->format == "xml") {
            $errors = $this->apiM->helper()->string()->validateXML($requestContent);
        } else {
            $errors = $this->apiM->helper()->string()->validateJson($requestContent);
        }
        // collect errors
        foreach ($errors as $error) {
            $this->errors->addError(400, $error, 'external', 'InvalidRequest')->setCode(400);
            return;
        }

        //now build request model
        $requestModel = $this->apiM->serializer()->deserialize($requestContent,
            'Sso\RestBundle\ModelRest\Request\ObjectWs\Version1\ObjectValue\Delete\Request', $this->format);

        $validationErrors = $this->apiM->validator()->validate($requestModel);

        if (count($validationErrors) > 0) {
            foreach ($validationErrors as $error) {
                $message = $error->getPropertyPath() . ": " . $error->getMessage();
                $this->errors->addError(400, $message, 'external')->setCode(400);
            }
        }

        return $requestModel;
    }

    /**
     * @param RequestItem $item
     * @return ResponseItem
     */
    public function send(RequestItem $item)
    {
        $requestModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->objectValue()->delete()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->objectValue()->delete()->credentials();
        $credentialsModelXml->setCredentialsData($this->userToken->servicetoken, 'MyServiceName',
            $this->userToken->useridentifier, $this->userToken->username, 'en');
        $requestModelXml->setCredentials($credentialsModelXml);

        // add objectValueType
        $objectValueModelXml = $this->apiM->modelXml()->request()->objectWs()->version1()->objectValue()->delete()->objectValue();
        $objectValueModelXml->setObjectValueTypeData($item->getId());
        $requestModelXml->setObjectValue($objectValueModelXml);

        $options = array(
            'headers' => array(
                'API-VERSION' => '1',
                'API-CONTROLLER' => 'ObjectValue',
                'API-ACTION' => 'Delete',
                'Content-Type' => 'text/xml; charset=UTF-8',
            ),
            'body' => $this->apiM->serializer()->serialize($requestModelXml, 'xml'),
        );

        try {
            $clientResponse = $this->client->post($this->apiM->configuration()->objectWs()->getBaseUrl() . $this->apiM->configuration()->objectWs()->getServiceAction(),
                $options);
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem(['status' => false, 'xml' => $errorResponse], $item);
        }

        return $this->buildItem(['status' => true, 'xml' => $clientResponse->getBody()->getContents()], $item);
    }

    /**
     *
     * @param string $xmlResponse
     * @param RequestItem $requestItem
     * @return ResponseItem
     */
    private function buildItem($xmlResponse, RequestItem $requestItem)
    {
        $responseItem = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->delete()->item();
        $responseItem->setId($requestItem->getId());

        if (!$xmlResponse['status']) {
            // build error response for item
            $responseItem = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for item
            $responseItem = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }
        return $responseItem;
    }

    /**
     *
     * @param string $xmlString
     * @param ResponseItem $responseItem
     * @return ResponseItem
     */
    private function buildSuccess($xmlString, ResponseItem $responseItem)
    {
        $responseItem->setCode(200);
        $this->successXmlObjectValue = $this->apiM->serializer()->deserialize($xmlString,
            'Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectValue\Delete\ObjectValueDelete\Success\Response',
            'xml');

        $responseItem->setId($this->successXmlObjectValue->getObjectValue()->getShow()->getObjectValueType()->getId());
        $responseItem->setTrackId($this->successXmlObjectValue->getTrackid());
        $responseItem->setScriptTimeSec($this->apiM->scriptTimeSeconds());


        return $responseItem;
    }

    /**
     *
     * @param string $xmlString
     * @param ResponseItem $responseItem
     * @return ResponseItem
     */
    private function buildError($xmlString, ResponseItem $responseItem)
    {
        $errorsItem = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->delete()->errors();
        $responseItem->setCode(400);

        $this->errorXmlObjectValue = $this->apiM->serializer()->deserialize($xmlString,
            'Sso\RestBundle\ModelXml\Response\ObjectWs\Version1\ObjectValue\Delete\ObjectValueDelete\Error\Response',
            'xml');

        $errorsItem->setTrackId($this->errorXmlObjectValue->getTrackid());
        $errorsItem->setCode(400);

        $errors = $this->errorXmlObjectValue->getObjectValue()->getShow()->getError();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->objectWs()->version1()->objectValue()->delete()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setMessage($message);
            $errorItem->setCode(400);
            $errorItem->setException('external');
            $errorsItem->addError($errorItem);
        }
        $responseItem->setScriptTimeSec($this->apiM->scriptTimeSeconds());
        $responseItem->setErrors($errorsItem);

        return $responseItem;
    }
}
