<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\UserWs\Version1\Application\Delete;

use Sso\RestBundle\Api\ExceptionCollection\MainValidationException;
use FOS\RestBundle\View\View as RestView;
use Sso\RestBundle\Api\Manager as ApiManager;
use GuzzleHttp\Exception\RequestException;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use Symfony\Component\HttpFoundation\Request;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\Application\Delete\Request as RequestModel;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\Application\Delete\Application as RequestApplication;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Delete\Application as ResponseApplication;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\Application\Delete\Response as RestResponseModel;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Delete\Success\Response as SuccessXMLResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\Application\Delete\Error\Response as ErrorXMLResponse;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\UserWs\Version1\Application\Delete
 */
class Handler
{
    const STATUS = "status";
    const ERROR_EXTERNAL = "external";

    /**
     * @var UidentifyToken
     */
    private $userToken;

    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var ErrorApi
     */
    private $errorApi;

    /**
     * @var Errors
     */
    private $errors;

    /**
     * @var ErrorXMLResponse
     */
    private $errorXmlEntity;

    /**
     * @var SuccessXMLResponse
     */
    private $successXmlEntity;

    /**
     * @var RestResponseModel
     */
    private $restResponseModel;

    /**
     * @var bool
     */
    private $oneResponseSuccess = false;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->restResponseModel = $this->apiM->modelRest()->response()->userWs()->version1()->application()->delete()->response();
        $this->userToken = $this->apiM->getToken();
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        // first: validate main if there is any error it is allowed to use the default error response
        try {
            list($requestContent, $format) = $this->apiM->mainValidator()->validateRestRequest($request);
        } catch (MainValidationException $exc) {
            $this->errorApi->error()->errors()->setScriptSeconds($this->apiM->scriptTimeSeconds());

            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors(), 416);
        }

        // second: setting the request model
        $requestModelRest = $this->apiM->serializer()->deserialize($requestContent, RequestModel::class, $format);

        // loop the application
        foreach ($requestModelRest->getApplication() as $application) {
            $responseApplication = $this->sendXmlRequest($application);
            $this->restResponseModel->setApplication($responseApplication);
        }

        // reset the main code status
        $this->restResponseModel->setCode($this->oneResponseSuccess ? 200 : 400);

        return $this->apiM->restView()->setData($this->restResponseModel)->setStatusCode($this->oneResponseSuccess ? 200 : 400);
    }

    /**
     * @param RequestApplication $requestApplication
     * @return ResponseApplication
     */
    private function sendXmlRequest(RequestApplication $requestApplication)
    {
        $requestModel = $this->apiM->modelXml()->request()->userWs()->version1()->application()->delete()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->userWs()->version1()->application()->delete()->credentials();
        $credentialsModelXml->setCredentialsData(
            $this->userToken->servicetoken,
            'Application',
            $this->userToken->useridentifier,
            $this->userToken->username,
            'en'
        );
        $requestModel->setCredentials($credentialsModelXml);

        // add application
        $objectModelXml = $this->apiM->modelXml()->request()->userWs()->version1()->application()->delete()->application();
        $objectModelXml->setApplication(
            $requestApplication->getName()
        );
        $requestModel->setApplication($objectModelXml);

        // send request
        try {
            $clientResponse = $this->apiM->client()->post(
                $this->apiM->configuration()->userWs()->getBaseUrl() . $this->apiM->configuration()->userWs()->getServiceAction(),
                $this->apiM->clientHelper()->options()->getOptions(1, 'Application', 'Delete',
                    $this->apiM->serializer()->serialize($requestModel, 'xml'))
            );
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem([static::STATUS => false, 'xml' => $errorResponse]);
        }

        return $this->buildItem([static::STATUS => true, 'xml' => $clientResponse->getBody()->getContents()]);
    }

    /**
     * @param string $xmlResponse
     * @return ResponseApplication
     */
    private function buildItem($xmlResponse)
    {
        $responseItem = $this->apiM->modelRest()->response()->userWs()->version1()->application()->delete()->application();

        if (!$xmlResponse[static::STATUS]) {
            // build error response for a application
            $view = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for a application
            $view = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }

        return $view;
    }

    /**
     * @param string $xmlString
     * @param ResponseApplication $responseItem
     * @return ResponseApplication
     */
    private function buildSuccess($xmlString, ResponseApplication $responseItem)
    {
        $this->successXmlEntity = $this->apiM->serializer()->deserialize($xmlString, SuccessXMLResponse::class, 'xml');

        if (!$this->successXmlEntity->getApplication()
            || ($this->successXmlEntity->getApplication()
                && $this->successXmlEntity->getApplication()->getDelete()->getStatus() != 'Success')
        ) {
            return $this->buildError($xmlString, $responseItem);
        }

        $this->oneResponseSuccess = true;

        /**
         * @var ResponseApplication $responseItem
         */
        $responseItem = $this->apiM->helper()->mapper()->modelInToModelOut(
            $this->successXmlEntity->getApplication()->getDelete(),
            $responseItem
        );

        // Delete additional fields to response
        $this->restResponseModel
            ->setCode(200)
            ->setTrackId($this->successXmlEntity->getTrackId())
            ->setDate($this->successXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Delete additional fields to success response
        $responseItem->setCode(200);

        return $responseItem;
    }

    /**
     * @param ResponseApplication $responseItem
     * @param string $xmlString
     * @return ResponseApplication
     */
    private function buildError($xmlString, ResponseApplication $responseItem)
    {
        $this->errorXmlEntity = $this->apiM->serializer()->deserialize($xmlString, ErrorXMLResponse::class, 'xml');
        $errors = $this->errorXmlEntity->getApplication()->getDelete()->getErrors();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->userWs()->version1()->application()->delete()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setMessage($message);
            $errorItem->setCode(400);
            $errorItem->setException(static::ERROR_EXTERNAL);
            $responseItem->setError($errorItem);
        }

        // Delete additional fields to response
        $this->restResponseModel
            ->setCode(400)
            ->setTrackId($this->errorXmlEntity->getTrackId())
            ->setDate($this->errorXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Delete additional fields to error response
        $responseItem->setCode(400);

        return $responseItem;
    }
}
