<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\RestBundle\Worker\UserWs\Version1\User\Delete;

use Sso\RestBundle\Api\ExceptionCollection\MainValidationException;
use Sso\RestBundle\Api\Manager as ApiManager;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\User\Delete\Request as RequestModel;
use Sso\RestBundle\ModelRest\Request\UserWs\Version1\User\Delete\UserType as RequestUserType;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\User\Delete\Response as RestResponseModel;
use Sso\RestBundle\ModelRest\Response\UserWs\Version1\User\Delete\UserType as ResponseUserType;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Delete\Error\Response as ErrorXMLResponse;
use Sso\RestBundle\ModelXml\Response\UserWs\Version1\User\Delete\Success\Response as SuccessXMLResponse;
use FOS\RestBundle\View\View as RestView;
use GuzzleHttp\Exception\RequestException;
use LifeStyle\Tools\RestAuthBundle\Security\Authentication\Token\UidentifyToken;
use LifeStyle\Tools\RestErrorBundle\Api\Error\Errors\Index as Errors;
use LifeStyle\Tools\RestErrorBundle\Api\Manager as ErrorApi;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class Handler
 * @package Sso\RestBundle\Worker\UserWs\Version1\User\Delete
 */
class Handler
{
    const STATUS = "status";
    const ERROR_EXTERNAL = "external";

    /**
     * @var UidentifyToken
     */
    private $userToken;

    /**
     * @var ApiManager
     */
    private $apiM;

    /**
     * @var ErrorApi
     */
    private $errorApi;

    /**
     * @var Errors
     */
    private $errors;

    /**
     * @var ErrorXMLResponse
     */
    private $errorXmlEntity;

    /**
     * @var SuccessXMLResponse
     */
    private $successXmlEntity;

    /**
     * @var RestResponseModel
     */
    private $restResponseModel;

    /**
     * @var bool
     */
    private $oneResponseSuccess = false;

    /**
     * @param ApiManager $apiM
     */
    public function __construct(ApiManager $apiM)
    {
        $this->apiM = $apiM;
        $this->errorApi = $this->apiM->errorManager();
        $this->errors = $this->errorApi->error()->errors();
        $this->restResponseModel = $this->apiM->modelRest()->response()->userWs()->version1()->user()->delete()->response();
        $this->userToken = $this->apiM->getToken();
    }

    /**
     * @param Request $request
     * @return RestView
     */
    public function init(Request $request)
    {
        try {
            list($requestContent, $format) = $this->apiM->mainValidator()->validateRestRequest($request);
        } catch (MainValidationException $exc) {
            $this->errorApi->error()->errors()->setScriptSeconds($this->apiM->scriptTimeSeconds());

            return $this->errorApi->error()->view()->getErrorsView($this->errors->getErrors(), 416);
        }

        /** @var RequestModel $requestModelRest */
        $requestModelRest = $this->apiM->serializer()->deserialize($requestContent, RequestModel::class, $format);

        foreach ($requestModelRest->getUserTypes() as $userType) {
            $responseUserType = $this->sendXmlRequest($userType);
            $this->restResponseModel->addUserType($responseUserType);
        }

        $this->restResponseModel->setCode($this->oneResponseSuccess ? 200 : 400);

        return $this->apiM->restView()->setData($this->restResponseModel)->setStatusCode($this->oneResponseSuccess ? 200 : 400);
    }

    /**
     * @param RequestUserType $requestUserType
     * @return ResponseUserType
     */
    private function sendXmlRequest(RequestUserType $requestUserType)
    {
        $requestModel = $this->apiM->modelXml()->request()->userWs()->version1()->user()->delete()->request();

        // add credentials
        $credentialsModelXml = $this->apiM->modelXml()->request()->userWs()->version1()->user()->delete()->credentials();
        $credentialsModelXml->setCredentialsData(
            $this->userToken->servicetoken,
            'User',
            $this->userToken->useridentifier,
            $this->userToken->username,
            'en'
        );
        $requestModel->setCredentials($credentialsModelXml);

        // add user
        $objectModelXml = $this->apiM->modelXml()->request()->userWs()->version1()->user()->delete()->user();
        $objectModelXml->setUserTypeData($requestUserType->getIdentifier());
        $requestModel->setUser($objectModelXml);

        // send request
        try {
            $clientResponse = $this->apiM->client()->delete(
                $this->apiM->configuration()->userWs()->getBaseUrl() . $this->apiM->configuration()->userWs()->getServiceAction(),
                $this->apiM->clientHelper()->options()->getOptions(1, 'User', 'Delete',
                    $this->apiM->serializer()->serialize($requestModel, 'xml'))
            );
        } catch (RequestException $ex) {
            $errorResponse = $ex->getResponse()->getBody()->getContents();
            $this->apiM->logger()->error($errorResponse);

            return $this->buildItem([static::STATUS => false, 'xml' => $errorResponse]);
        }

        return $this->buildItem([static::STATUS => true, 'xml' => $clientResponse->getBody()->getContents()]);
    }

    /**
     * @param string $xmlResponse
     * @return ResponseUserType
     */
    private function buildItem($xmlResponse)
    {
        $responseItem = $this->apiM->modelRest()->response()->userWs()->version1()->user()->delete()->userType();

        if (!$xmlResponse[static::STATUS]) {
            // build error response for a userType
            $view = $this->buildError($xmlResponse['xml'], $responseItem);
        } else {
            // fill the success response for a userType
            $view = $this->buildSuccess($xmlResponse['xml'], $responseItem);
        }

        return $view;
    }

    /**
     * @param string $xmlString
     * @param ResponseUserType $responseItem
     * @return ResponseUserType
     */
    private function buildSuccess($xmlString, ResponseUserType $responseItem)
    {
        $this->successXmlEntity = $this->apiM->serializer()->deserialize($xmlString, SuccessXMLResponse::class, 'xml');

        if (!$this->successXmlEntity->getUser()
            || ($this->successXmlEntity->getUser()
                && $this->successXmlEntity->getUser()->getDelete()->getStatus() != 'Success')
        ) {
            return $this->buildError($xmlString, $responseItem);
        }

        $this->oneResponseSuccess = true;

        /** @var ResponseUserType $responseItem */
        $responseItem = $this->apiM->helper()->mapper()->modelInToModelOut(
            $this->successXmlEntity->getUser()->getDelete(),
            $responseItem
        );

        // Add additional fields to response
        $this->restResponseModel
            ->setCode(200)
            ->setTrackId($this->successXmlEntity->getTrackId())
            ->setDate($this->successXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to user response
        $responseItem->setCode(200);

        return $responseItem;
    }

    /**
     * @param ResponseUserType $responseItem
     * @param string $xmlString
     * @return ResponseUserType
     */
    private function buildError($xmlString, ResponseUserType $responseItem)
    {
        $this->errorXmlEntity = $this->apiM->serializer()->deserialize($xmlString, ErrorXMLResponse::class, 'xml');
        $errors = $this->errorXmlEntity->getUser()->getDelete()->getErrors();

        foreach ($errors as $error) {
            $errorItem = $this->apiM->modelRest()->response()->userWs()->version1()->user()->add()->error();
            $message = 'ErrorCode: ' . $error->getErrorCode() . ' ErrorRef: ' . $error->getErrorRef() . ' ShortMessage: ' . $error->getShortMessage();
            $errorItem->setMessage($message);
            $errorItem->setCode(400);
            $errorItem->setException(static::ERROR_EXTERNAL);
            $responseItem->addError($errorItem);
        }

        // Add additional fields to response
        $this->restResponseModel
            ->setCode(400)
            ->setTrackId($this->errorXmlEntity->getTrackId())
            ->setDate($this->errorXmlEntity->getDate())
            ->setScriptTimeSec($this->apiM->scriptTimeSeconds());

        // Add additional fields to error response
        $responseItem->setCode(400);

        return $responseItem;
    }
}
