<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace ProcessLock;

use ProcessLock\Exception\AlreadyLockedException;
use ProcessLock\Exception\CheckLockIsExpiredException;
use ProcessLock\Exception\UnlockException;
use ProcessLock\Model\Config;

class Locker implements LockerInterface
{
    /**
     * @var Config
     */
    private $config;

    /**
     * Locker constructor.
     * @param Config $config
     */
    public function __construct(Config $config)
    {
        $this->config = $config;
    }

    /**
     * @throws AlreadyLockedException
     */
    public function lock()
    {
        $lockFile = $this->getLockFilePath();

        if ($this->isLockExpired()) {
            $this->unlock();
        }

        if ($this->isLocked()) {
            $processName = $this->config->getProcessName();
            throw new AlreadyLockedException("process ${processName} already locked!");
        }

        file_put_contents($lockFile, getmypid());
    }

    /**
     * @return bool
     */
    public function isLocked()
    {
        $lockFile = $this->getLockFilePath();

        return file_exists($lockFile);
    }

    /**
     * @throws UnlockException
     */
    public function unlock()
    {
        if ($this->isLocked()) {
            $lockFile = $this->getLockFilePath();
            $lockFileSuccessfullyUnlinked = unlink($lockFile);

            if (!$lockFileSuccessfullyUnlinked) {
                throw new UnlockException("unlock is in failure! lockfile ${lockFile} could not be deleted!");
            }
        }
    }

    /**
     * @return bool
     */
    private function isLockExpired()
    {
        if ($this->isLocked()) {
            $lockFile = $this->getLockFilePath();

            try {
                return ((time() - filemtime($lockFile)) >= $this->config->getProcessTTL());
            } catch (\Exception $exception) {
                throw new CheckLockIsExpiredException($exception->getMessage(), 0, $exception);
            }
        }

        return true;
    }

    /**
     * @return string
     */
    private function getLockFilePath()
    {
        $sysTmpDir = sys_get_temp_dir();
        $processName = $this->config->getProcessName();

        return "${sysTmpDir}/${processName}.loc";
    }
}
