<?php

/**
 *  LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 *  and is private by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license
 *  is a violation of the license agreement and will be prosecuted by
 *  civil and criminal law.
 *
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Vtours\Model\Booking\Test;

use PHPUnit\Framework\TestCase;
use Vtours\Model\Booking\Payment\BankAccount\BankAccount;
use Vtours\Model\Booking\Payment\Details\PaymentDetails;
use Vtours\Model\Booking\Payment\Paycode\Paycode;
use Vtours\Model\Booking\Payment\Payments;
use Vtours\Model\Booking\Payment\Wirecard\Wirecard;
use Vtours\Model\Booking\PriceHistory\PriceHistory;

/**
 * Class Payments
 * @package Vtours\Model\Booking\Payment
 */
class PaymentsTest extends TestCase
{
    public function test_add_payment_details()
    {
        $paymentDetails1 = new PaymentDetails();
        $paymentDetails1
            ->setType(PaymentDetails::TYPE_PREPAYMENT)
            ->setDate(new \DateTime())
            ->setGross(123.24)
            ->setUser('dbarlowe')
        ;

        $paymentDetails2 = new PaymentDetails();
        $paymentDetails2
            ->setType(PaymentDetails::TYPE_BALANCEPAYMENT)
            ->setDate(new \DateTime())
            ->setGross(131.24)
            ->setUser('dbarlowe')
        ;

        $paymentDetails3 = new PaymentDetails();
        $paymentDetails3
            ->setType(PaymentDetails::TYPE_PROVISION)
            ->setDate(new \DateTime())
            ->setGross(21.24)
            ->setUser('dbarlowe')
        ;

        $payments = new Payments();
        $payments
            ->addPaymentDetails($paymentDetails1)
            ->addPaymentDetails($paymentDetails2)
            ->addPaymentDetails($paymentDetails3)
        ;

       self::assertCount(3, $payments->getPaymentDetails());
       self::assertSame(PaymentDetails::TYPE_BALANCEPAYMENT, $payments->getPaymentDetails()[1]->getType());
    }

    public function test_add_history_to_payment()
    {
        $priceHistory1 = new PriceHistory();
        $priceHistory1
            ->setPriceBefore(123)
            ->setPriceAfter(150)
            ->setChangedBy('dbarlowe')
            ->setChangedDate(new \DateTime())
        ;

        $priceHistory2 = new PriceHistory();
        $priceHistory2
            ->setPriceBefore(100)
            ->setPriceAfter(200)
            ->setChangedBy('dbarlowe')
            ->setChangedDate(new \DateTime())
        ;

        $payments = new Payments();
        $payments
            ->addPriceHistory($priceHistory1)
            ->addPriceHistory($priceHistory2)
            ;

        self::assertCount(2, $payments->getPriceHistory());
        self::assertSame(200, $payments->getPriceHistory()[1]->getPriceAfter());
    }

    public function test_add_bank_account_to_payment()
    {
        $bankAccount = new BankAccount();
        $bankAccount
            ->setAccountHolder('Dennis Barlowe')
            ->setAccountNumber(123456)
            ->setBankSortCode(795550000)
            ->setBic('BILADEM1ASA')
            ->setIban('DE1239075901375901')
            ;

        $payments = new Payments();
        $payments->setBankAccount($bankAccount);

        self::assertSame(123456, $payments->getBankAccount()->getAccountNumber());
    }

    public function test_add_paycode_to_payment()
    {
        $paycode = new Paycode();
        $paycode
            ->setStatus('done')
            ->setUrl('https://life-style.de')
            ->setCreated(new \DateTime())
        ;

        $payments = new Payments();
        $payments->setPaycode($paycode);

        self::assertSame('done', $payments->getPaycode()->getStatus());
        self::assertInstanceOf('DateTime', $payments->getPaycode()->getCreated());
    }

    public function test_add_wirecard_to_payment()
    {
        $wirecard = new Wirecard();
        $wirecard
            ->setBalancePaymentToken('bxxx')
            ->setCancellationToken('cxxx')
            ->setPrePaymentToken('pxxx')
            ;

        $payments = new Payments();
        $payments->setWirecard($wirecard);

        self::assertSame('pxxx', $payments->getWirecard()->getPrePaymentToken());
        self::assertSame('cxxx', $payments->getWirecard()->getCancellationToken());
        self::assertSame('bxxx', $payments->getWirecard()->getBalancePaymentToken());
    }
}
