<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Vtours\ComputopPaymentSdk\Service;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use Vtours\ComputopPaymentSdk\Exception\ComputopPaymentServiceException;
use Vtours\ComputopPaymentSdk\Mapper\DirectDebit\DirectDebitMapperInitial;
use Vtours\ComputopPaymentSdk\Mapper\DirectDebit\DirectDebitMapperRecurring;
use Vtours\ComputopPaymentSdk\Model\InitialPayment\Request as InitialPaymentRequest;
use Vtours\ComputopPaymentSdk\Model\RecurringPayment\Request as RecurringPaymentRequest;
use Vtours\ComputopPaymentSdk\Model\Response;

/**
 * Class ComputopPaymentService
 * @package Vtours\ComputopPaymentSdk\Service
 */
class ComputopSepaPaymentService
{
    /**
     * URL for the Computop Payment Service.
     *
     * @var string
     */
    private $url;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var DirectDebitMapperInitial
     */
    private $directDebitMapperInitial;

    /**
     * @var DirectDebitMapperRecurring
     */
    private $directDebitMapperRecurring;

    /**
     * ComputopPaymentService constructor.
     * @param $url
     */
    public function __construct($url)
    {
        $this->url = $url;
        $this->client = new Client([
            'base_uri' => $url,
            'timeout' => 60,
        ]);
        $this->directDebitMapperInitial = new DirectDebitMapperInitial();
        $this->directDebitMapperRecurring = new DirectDebitMapperRecurring();
    }

    /**
     * @param InitialPaymentRequest $request
     * @return Response
     * @throws ComputopPaymentServiceException
     */
    public function payAsInitialPayment(InitialPaymentRequest $request): Response
    {
        try {
            $response = $this->client->request('POST', '/direct-debit/initial', [
                'json' => $request,
                'debug' => true,
            ]);
        } catch (GuzzleException $exception) {
            throw new ComputopPaymentServiceException('There has been a problem. ', 500, $exception);
        }

        return $this->directDebitMapperInitial->map($response);
    }

    /**
     * @param RecurringPaymentRequest $request
     * @return Response
     * @throws ComputopPaymentServiceException
     */
    public function payAsRecurringPayment(RecurringPaymentRequest $request): Response
    {
        try {
            $response = $this->client->request('POST', '/direct-debit/recurring', [
                'json' => $request,
                'debug' => true,
            ]);
        } catch (RequestException $exception) {
            $response = $exception->getResponse();
        } catch (GuzzleException $exception) {
            throw new ComputopPaymentServiceException('There has been a problem. ' . $exception->getMessage(), 500, $exception);
        }

        return $this->directDebitMapperRecurring->map($response);
    }


}
