<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Vtours\ComputopPaymentSdk\Service;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;
use Vtours\ComputopPaymentSdk\Exception\ComputopPaymentServiceException;
use Vtours\ComputopPaymentSdk\Mapper\DirectDebit\DirectDebitMapperInitial;
use Vtours\ComputopPaymentSdk\Mapper\DirectDebit\DirectDebitMapperRecurring;
use Vtours\ComputopPaymentSdk\Model\RequestInterface;
use Vtours\ComputopPaymentSdk\Model\Response;

/**
 * Class ComputopSepaPaymentService
 * @package Vtours\ComputopPaymentSdk\Service
 */
class ComputopSepaPaymentService
{
    /**
     * URL to the computop-payment-service.
     *
     * @var string
     */
    private $url;

    /**
     * @var Client
     */
    private $client;

    /**
     * @var boolean
     */
    private $debug;

    /**
     * @var DirectDebitMapperInitial
     */
    private $directDebitMapperInitial;

    /**
     * @var DirectDebitMapperRecurring
     */
    private $directDebitMapperRecurring;

    /**
     * ComputopPaymentService constructor.
     * @param string $url
     * @param bool $debug
     */
    public function __construct(string $url, bool $debug)
    {
        $this->url = $url;
        $this->debug = $debug;

        $this->client = new Client([
            'base_uri' => $url,
            'timeout' => 60,
        ]);
        $this->directDebitMapperInitial = new DirectDebitMapperInitial();
        $this->directDebitMapperRecurring = new DirectDebitMapperRecurring();
    }

    /**
     * @param RequestInterface $request
     * @return Response
     * @throws ComputopPaymentServiceException
     */
    public function payAsInitialPayment(RequestInterface $request): Response
    {
        try {
            $response = $this->client->request('POST', '/direct-debit/initial', [
                'json' => $request,
                'debug' => $this->debug,
            ]);
        } catch (GuzzleException $exception) {
            throw ComputopPaymentServiceException::withGuzzleException($exception);
        }

        return $this->directDebitMapperInitial->map($response);
    }

    /**
     * @param RequestInterface $request
     * @return Response
     * @throws ComputopPaymentServiceException
     */
    public function payAsRecurringPayment(RequestInterface $request): Response
    {
        try {
            $response = $this->client->request('POST', '/direct-debit/recurring', [
                'json' => $request,
                'debug' => $this->debug,
            ]);
        } catch (RequestException $exception) {
            $response = $exception->getResponse();
        } catch (GuzzleException $exception) {
            throw ComputopPaymentServiceException::withGuzzleException($exception);
        }

        return $this->directDebitMapperRecurring->map($response);
    }
}
