<?php
/**
 *  LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 *  and is private by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license
 *  is a violation of the license agreement and will be prosecuted by
 *  civil and criminal law.
 *
 * @copyright 2021 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Vtours\ComputopPaymentSdk\Mapper\DirectDebit\Initial;

use Psr\Http\Message\ResponseInterface as HttpResponseInterface;
use Vtours\ComputopPaymentSdk\Exception\ComputopPaymentServiceException;
use Vtours\ComputopPaymentSdk\Mapper\ResponseMapperAbstract;
use Vtours\ComputopPaymentSdk\Model\DirectDebit\Initial\InnerResponse;
use Vtours\ComputopPaymentSdk\Model\DirectDebit\Initial\Request;
use Vtours\ComputopPaymentSdk\Model\PaymentServiceResponse;
use Vtours\ComputopPaymentSdk\Model\RequestInterface as RequestModelInterface;
use Vtours\ComputopPaymentSdk\Model\ResponseInterface as ResponseModelInterface;

/**
 * Class DirectDebitMapperInitial
 * @package Vtours\ComputopPaymentSdk\Mapper\DirectDebit
 */
class DirectDebitMapperInitial extends ResponseMapperAbstract
{
    /**
     * @param HttpResponseInterface $response
     * @return PaymentServiceResponse
     * @throws ComputopPaymentServiceException
     */
    public function map(HttpResponseInterface $response): PaymentServiceResponse
    {
        $responseBody = $this->getResponseBody($response);

        return (new PaymentServiceResponse())
            ->setStatus($responseBody['status'])
            ->setStatusCode($responseBody['status_code'])
            ->setMessage($responseBody['message'])
            ->setErrorList($responseBody['errorList'] ?? [])
            ->setRequest($this->mapRequest($response))
            ->setResponse($this->mapResponse($response));
    }

    /**
     * @param HttpResponseInterface $response
     * @return RequestModelInterface
     */
    protected function mapRequest(HttpResponseInterface $response): RequestModelInterface
    {
        $responseBody = $this->getResponseBody($response);

        return (new Request())
            ->setAmount($responseBody['request_data']['amount'] ?? null)
            ->setCurrency($responseBody['request_data']['currency'] ?? null)
            ->setReferenceNumber($responseBody['request_data']['reference_number'] ?? '')
            ->setAccountOwner($responseBody['request_data']['account_owner'] ?? null)
            ->setIban($responseBody['request_data']['iban'] ?? null)
            ->setBankName($responseBody['request_data']['bank_name'] ?? null)
            ->setOrderDescription($responseBody['request_data']['order_description'] ?? null)
            ->setDateOfSignature($this->normalizeDateTime($responseBody['request_data']['date_of_signature']))
            ->setMandateId($responseBody['request_data']['mandate_id'] ?? null)
            ->setTransactionId($responseBody['request_data']['transaction_id'] ?? null);
    }

    /**
     * @param HttpResponseInterface $response
     * @return InnerResponse
     * @throws ComputopPaymentServiceException
     */
    protected function mapResponse(HttpResponseInterface $response): ResponseModelInterface
    {
        $responseBody = $this->getResponseBody($response);

        if (empty($responseBody['response_data'])) {
            throw ComputopPaymentServiceException::withMissingResponseData($response);
        }

        $innerResponseData = array_change_key_case($responseBody['response_data'], CASE_LOWER);

        if (isset($innerResponseData['dtofsgntr'])) {
            $mandateSignatureDate = $this->normalizeDateTime($innerResponseData['dtofsgntr']);
        }

        return (new InnerResponse())
            ->setMid($innerResponseData['mid'] ?? null)
            ->setPaymentId($innerResponseData['payid'] ?? null)
            ->setXid($innerResponseData['xid'] ?? null)
            ->setTransactionId($innerResponseData['transid'] ?? null)
            ->setIban($innerResponseData['iban'] ?? null)
            ->setAccountOwner($innerResponseData['accowner'] ?? null)
            ->setBankName($innerResponseData['accbank'] ?? 'undefined')
            ->setMandateId($innerResponseData['mandateid'] ?? null)
            ->setDateOfSignature($mandateSignatureDate ?? null)
            ->setMandateSequenceType($innerResponseData['mdtseqtype'] ?? null)
            ->setStatus($innerResponseData['status'] ?? null)
            ->setCode($innerResponseData['code'] ?? null)
            ->setDescription($innerResponseData['description'] ?? null)
            ->setRefNumber($innerResponseData['refnr'] ?? null);
    }
}
