<?php
/**
 *  LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 *  and is private by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license
 *  is a violation of the license agreement and will be prosecuted by
 *  civil and criminal law.
 *
 * @copyright 2021 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Vtours\ComputopPaymentSdk\Mapper\CreditCard\Recurring;

use Psr\Http\Message\ResponseInterface as HttpResponseInterface;
use Vtours\ComputopPaymentSdk\Exception\ComputopPaymentServiceException;
use Vtours\ComputopPaymentSdk\Mapper\ResponseMapperAbstract;
use Vtours\ComputopPaymentSdk\Model\CreditCard\Recurring\InnerResponse;
use Vtours\ComputopPaymentSdk\Model\CreditCard\Recurring\Request;
use Vtours\ComputopPaymentSdk\Model\PaymentServiceResponse;
use Vtours\ComputopPaymentSdk\Model\RequestInterface as RequestModelInterface;
use Vtours\ComputopPaymentSdk\Model\ResponseInterface as ResponseModelInterface;

/**
 * Class CreditCardMapperRecurring
 * @package Vtours\ComputopPaymentSdk\Mapper\CreditCard
 */
class CreditCardMapperRecurring extends ResponseMapperAbstract
{
    /**
     * @param HttpResponseInterface $response
     * @return PaymentServiceResponse
     * @throws ComputopPaymentServiceException
     */
    public function map(HttpResponseInterface $response): PaymentServiceResponse
    {
        $responseBody = $this->getResponseBody($response);

        return (new PaymentServiceResponse())
            ->setStatus($responseBody['status'])
            ->setStatusCode($responseBody['status_code'])
            ->setMessage($responseBody['message'])
            ->setRequest($this->mapRequest($response))
            ->setResponse($this->mapResponse($response));
    }

    /**
     * @param HttpResponseInterface $response
     * @return RequestModelInterface
     */
    protected function mapRequest(HttpResponseInterface $response): RequestModelInterface
    {
        $responseBody = $this->getResponseBody($response);

        return (new Request())
            ->setAmount($responseBody['request_data']['amount'] ?? null)
            ->setCurrency($responseBody['request_data']['currency'] ?? null)
            ->setReferenceNumber($responseBody['request_data']['reference_number'] ?? '')
            ->setTransactionId($responseBody['request_data']['transaction_id'] ?? null)
            ->setOrderDescription($responseBody['request_data']['order_description'] ?? null)
            ->setAccountOwner($responseBody['request_data']['account_owner'] ?? null)
            ->setAddressCountry($responseBody['request_data']['address_country'] ?? null)
            ->setCardNumber($responseBody['request_data']['card_number'])
            ->setCardBrand($responseBody['request_data']['card_brand'])
            ->setCardExpiryDate($responseBody['request_data']['card_expiry_date'] ?? null)
            ->setCardHolderName($responseBody['request_data']['card_holder_name'] ?? null)
            ->setCardHolderCountry($responseBody['request_data']['card_holder_country'] ?? null)
            ->setCardHolderCity($responseBody['request_data']['card_holder_city'] ?? null)
            ->setSchemeReferenceId($responseBody['request_data']['scheme_reference_id'] ?? null);
    }

    /**
     * @param HttpResponseInterface $response
     * @return InnerResponse
     * @throws ComputopPaymentServiceException
     */
    protected function mapResponse(HttpResponseInterface $response): ResponseModelInterface
    {
        $responseBody = $this->getResponseBody($response);

        if (empty($responseBody['response_data'])) {
            throw ComputopPaymentServiceException::withMissingResponseData($response);
        }

        return (new InnerResponse())
            ->setMid($responseBody['response_data']['mid'] ?? null)
            ->setPaymentId($responseBody['response_data']['PayID'] ?? null)
            ->setXid($responseBody['response_data']['XID'] ?? null)
            ->setTransactionId($responseBody['response_data']['TransID'] ?? null)
            ->setReferenceNumber($responseBody['response_data']['RefNr'] ?? null)
            ->setCard($responseBody['response_data']['card'] ?? null)
            ->setCode($responseBody['response_data']['Code'] ?? null)
            ->setStatus($responseBody['response_data']['Status'] ?? null)
            ->setDescription($responseBody['response_data']['Description'] ?? null);
    }
}
