<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg,Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ProtocolServiceSdk\Model;

use DateInterval;
use DateTimeImmutable;
use JMS\Serializer\Annotation as Serializer;
use LifeStyle\ProtocolServiceSdk\Exception\ProtocolServiceSdkException;

/**
 * Class ProtocolCriteria
 * @package LifeStyle\ProtocolServiceSdk\Model
 *
 * @Serializer\AccessType(type=JMS\Serializer\Metadata\PropertyMetadata::ACCESS_TYPE_PUBLIC_METHOD)
 * @Serializer\ExclusionPolicy(policy=Serializer\ExclusionPolicy::ALL)
 */
class ProtocolCriteria
{
    const ORDER_BY_FIELD_ID = 'id';
    const ORDER_BY_FIELD_CLIENT = 'client';
    const ORDER_BY_FIELD_SENT_FLAG = 'sentFlag';
    const ORDER_BY_FIELD_SENT_DATE = 'sentDate';
    const ORDER_BY_FIELD_CREATE_DATE = 'createDate';
    const ORDER_BY_FIELD_SOURCE = 'source';

    const ORDER_BY_DIRECTION_ASC = 'ASC';
    const ORDER_BY_DIRECTION_DESC = 'DESC';

    /**
     * @var boolean|null
     *
     * @Serializer\Type("bool")
     * @Serializer\SerializedName("sent_flag")
     * @Serializer\Expose()
     */
    private $sentFlag;

    /**
     * @var string|null
     *
     * @Serializer\Type("string")
     * @Serializer\SerializedName("client")
     * @Serializer\Expose()
     */
    private $client;

    /**
     * @var DateTimeImmutable
     *
     * @Serializer\Type("DateTimeImmutable")
     * @Serializer\SerializedName("time_anchor")
     * @Serializer\Expose()
     */
    private $timeAnchor;

    /**
     * @var DateInterval|null
     *
     * @Serializer\Type("DateInterval")
     * @Serializer\SerializedName("time_frame")
     * @Serializer\Expose()
     */
    private $timeFrame;

    /**
     * @var string|null
     *
     * @Serializer\Type("string")
     * @Serializer\SerializedName("source")
     * @Serializer\Expose()
     */
    private $source;

    /**
     * @var array|string[]|null
     *
     * @Serializer\Type("array<string, string>")
     * @Serializer\SerializedName("order_by")
     * @Serializer\Expose()
     */
    private $orderBy;

    /**
     * @var integer|null
     *
     * @Serializer\Type("integer")
     * @Serializer\SerializedName("limit")
     * @Serializer\Expose()
     */
    private $limit;

    /**
     * @var integer|null
     *
     * @Serializer\Type("integer")
     * @Serializer\SerializedName("offset")
     * @Serializer\Expose()
     */
    private $offset;

    /**
     * @return bool|null
     */
    public function getSentFlag(): ?bool
    {
        return $this->sentFlag;
    }

    /**
     * @param bool|null $sentFlag
     * @return ProtocolCriteria
     */
    public function setSentFlag(?bool $sentFlag): ProtocolCriteria
    {
        $this->sentFlag = $sentFlag;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasSentFlag(): bool
    {
        return null !== $this->sentFlag;
    }

    /**
     * @return string|null
     */
    public function getClient(): ?string
    {
        return $this->client;
    }

    /**
     * @param string|null $client
     * @return ProtocolCriteria
     */
    public function setClient(?string $client): ProtocolCriteria
    {
        $this->client = $client;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasClient(): bool
    {
        return !empty($this->client);
    }

    /**
     * @return DateTimeImmutable
     */
    public function getTimeAnchor(): DateTimeImmutable
    {
        return $this->timeAnchor ?? new DateTimeImmutable();
    }

    /**
     * @param DateTimeImmutable $timeAnchor
     * @return ProtocolCriteria
     */
    public function setTimeAnchor(DateTimeImmutable $timeAnchor): ProtocolCriteria
    {
        $this->timeAnchor = $timeAnchor;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasTimeAnchor(): bool
    {
        return !empty($this->timeAnchor);
    }

    /**
     * @return DateInterval|null
     */
    public function getTimeFrame(): ?DateInterval
    {
        return $this->timeFrame;
    }

    /**
     * @param DateInterval|null $timeFrame
     * @return ProtocolCriteria
     */
    public function setTimeFrame(?DateInterval $timeFrame): ProtocolCriteria
    {
        $this->timeFrame = $timeFrame;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasTimeFrame(): bool
    {
        return !empty($this->timeFrame);
    }

    /**
     * @return string|null
     */
    public function getSource(): ?string
    {
        return $this->source;
    }

    /**
     * @param string|null $source
     * @return ProtocolCriteria
     */
    public function setSource(?string $source): ProtocolCriteria
    {
        $this->source = $source;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasSource(): bool
    {
        return !empty($this->source);
    }

    /**
     * @return array|string[]|null
     */
    public function getOrderBy(): ?array
    {
        return $this->orderBy;
    }

    /**
     * @param array|string[] $orderBy
     * @return ProtocolCriteria
     * @throws ProtocolServiceSdkException
     */
    public function setOrderBy(array $orderBy): ProtocolCriteria
    {
        if (!empty($orderBy)) {
            $this->assertOrderBy($orderBy);
        }
        $this->orderBy = $orderBy;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasOrderBy(): bool
    {
        return !empty($this->orderBy);
    }

    /**
     * @return int|null
     */
    public function getLimit(): ?int
    {
        return $this->limit;
    }

    /**
     * @param int|null $limit
     * @return ProtocolCriteria
     */
    public function setLimit(?int $limit): ProtocolCriteria
    {
        $this->limit = $limit;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasLimit(): bool
    {
        return !empty($this->limit);
    }

    /**
     * @return int|null
     */
    public function getOffset(): ?int
    {
        return $this->offset;
    }

    /**
     * @param int|null $offset
     * @return ProtocolCriteria
     */
    public function setOffset(?int $offset): ProtocolCriteria
    {
        $this->offset = $offset;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasOffset(): bool
    {
        return !empty($this->offset);
    }

    /**
     * @param array $orderBy
     * @throws ProtocolServiceSdkException
     */
    protected function assertOrderBy(array $orderBy): void
    {
        $valid = array_reduce(array_map(function (/*mixed*/ $key, /*mixed*/ $value): bool {
            return is_string($key) && is_string($value);
        }, array_keys($orderBy), array_values($orderBy)), function (bool $carry, bool $valid): bool {
            return $carry && $valid;
        }, true);

        if (!$valid) {
            throw ProtocolServiceSdkException::fromInvalidOrderBy();
        }

        array_walk($orderBy, function (string $orderByDirection, string $orderByField) {
            $orderByDirection = strtoupper($orderByDirection);
            if (!$this->isOrderByDirectionAvailable($orderByDirection)) {
                throw ProtocolServiceSdkException::fromInvalidOrderByDirection($orderByDirection);
            }

            if (!$this->isOrderByFieldAvailable($orderByField)) {
                throw ProtocolServiceSdkException::fromInvalidOrderByField($orderByField);
            }
        });
    }

    /**
     * @param string $orderByDirection
     * @return bool
     */
    protected function isOrderByDirectionAvailable(string $orderByDirection): bool
    {
        return in_array($orderByDirection, [
            static::ORDER_BY_DIRECTION_ASC,
            static::ORDER_BY_DIRECTION_DESC,
        ]);
    }

    /**
     * @param string $orderByField
     * @return bool
     */
    protected function isOrderByFieldAvailable(string $orderByField): bool
    {
        return in_array($orderByField, [
            static::ORDER_BY_FIELD_ID,
            static::ORDER_BY_FIELD_CLIENT,
            static::ORDER_BY_FIELD_CREATE_DATE,
            static::ORDER_BY_FIELD_SENT_FLAG,
            static::ORDER_BY_FIELD_SENT_DATE,
            static::ORDER_BY_FIELD_SOURCE,
        ]);
    }
}
