<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg,Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace LifeStyle\ProtocolServiceSdk\Service;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\RequestOptions;
use JMS\Serializer\ArrayTransformerInterface;
use JMS\Serializer\SerializerInterface;
use LifeStyle\ProtocolServiceSdk\Exception\ProtocolServiceSdkException;
use LifeStyle\ProtocolServiceSdk\Model\Protocol;
use LifeStyle\ProtocolServiceSdk\Model\ProtocolCriteria;
use LifeStyle\ProtocolServiceSdk\Model\ProtocolLine;
use LifeStyle\ProtocolServiceSdk\Model\ProtocolList;
use Psr\Http\Client\ClientExceptionInterface;

/**
 * Class ProtocolService
 * @package LifeStyle\ProtocolServiceSdk\Service
 */
class ProtocolService
{
    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var string
     */
    private $baseUri;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ArrayTransformerInterface
     */
    private $arrayTransformer;

    /**
     * ProtocolService constructor.
     * @param string|null $baseUri
     * @param SerializerInterface $serializer
     * @param ArrayTransformerInterface $arrayTransformer
     * @throws ProtocolServiceSdkException
     */
    public function __construct(?string $baseUri, SerializerInterface $serializer, ArrayTransformerInterface $arrayTransformer)
    {
        if (null === $baseUri) {
            throw ProtocolServiceSdkException::fromInvalidBaseUri();
        }

        $this->client = $this->createClient($this->baseUri = $baseUri);

        $this->serializer = $serializer;
        $this->arrayTransformer = $arrayTransformer;
    }

    /**
     * @param string|null $baseUri
     * @return Client
     */
    protected function createClient(?string $baseUri): Client
    {
        return new Client([
            'base_uri' => $baseUri,
        ]);
    }

    /**
     * @param int $protocolId
     * @return Protocol
     * @throws ProtocolServiceSdkException
     */
    public function read(int $protocolId): Protocol
    {
        try {
            $response = $this->client->request('GET', "/protocol/{$protocolId}", []);

            return $this->serializer->deserialize((string)$response->getBody(), Protocol::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::fromGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::fromOtherException($exception);
        }
    }

    /**
     * @param ProtocolCriteria $criteria
     * @return ProtocolList
     * @throws ProtocolServiceSdkException
     */
    public function list(ProtocolCriteria $criteria): ProtocolList
    {
        try {
            $response = $this->client->request('GET', '/protocol/list', [
                RequestOptions::QUERY => $this->arrayTransformer->toArray($criteria),
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), ProtocolList::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::fromGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::fromOtherException($exception);
        }
    }

    /**
     * @param Protocol $protocol
     * @return Protocol
     * @throws ProtocolServiceSdkException
     */
    public function create(Protocol $protocol): Protocol
    {
        try {
            $response = $this->client->request('POST', '/protocol/', [
                RequestOptions::BODY => $this->serializer->serialize($protocol, 'json'),
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), Protocol::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::fromGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::fromOtherException($exception);
        }
    }

    /**
     * @param Protocol $protocol
     * @param ProtocolLine $protocolLine
     * @return ProtocolLine
     * @throws ProtocolServiceSdkException
     */
    public function createLine(Protocol $protocol, ProtocolLine $protocolLine): ProtocolLine
    {
        try {
            $response = $this->client->request('POST', "/protocol/{$protocol->getId()}/line", [
                RequestOptions::BODY => $this->serializer->serialize($protocolLine, 'json'),
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), ProtocolLine::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::fromGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::fromOtherException($exception);
        }
    }

    /**
     * @param Protocol $protocol
     * @param bool $sentFlag
     * @return Protocol
     * @throws ProtocolServiceSdkException
     */
    public function markAsSent(Protocol $protocol, bool $sentFlag = true): Protocol
    {
        try {
            $response = $this->client->request('PATCH', "/protocol/{$protocol->getId()}/mark-as-sent", [
                RequestOptions::QUERY => [
                    'sent_flag' => $sentFlag,
                ],
            ]);

            return $this->serializer->deserialize((string)$response->getBody(), Protocol::class, 'json');
        } catch (ClientExceptionInterface $exception) {
            throw ProtocolServiceSdkException::fromGuzzleException($exception);
        } catch (Exception $exception) {
            throw ProtocolServiceSdkException::fromOtherException($exception);
        }
    }

    /**
     * This information might be needed for comparison of multiple parallel protocol service instances.
     *
     * @return string
     */
    public function getBaseUri(): string
    {
        return $this->baseUri;
    }
}
