<?php
include_once('vendor/autoload.php');

/**
 * *
 *  LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 *  and is private by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license
 *  is a violation of the license agreement and will be prosecuted by
 *  civil and criminal law.
 *
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 * /
 */

use Lifestyle\BookingClassification\Service\BookingClassificationService;

/**
 * Class Actions
 */
class Actions extends Rula2Std
{

    const BOOKINGTABLE = 'VT_BOOKINGS';
    const DATETIMEFORMAT = 'Y-m-d H:i:s';
    const UPDATER = "System_R2";

    /**
     * @var string
     */
    private $table;

    /**
     * @var string
     */
    private $mapTable;

    /**
     * @var string
     */
    private $pk;

    /**
     * @var string
     */
    private $pk2;

    /**
     * @var int
     */
    private $countVmixActions;

    /**
     * @var int
     */
    private $countPacificActions;

    /**
     * @var int
     */
    private $countDbActions;

    /**
     * @var bool
     */
    public $bookingState = false;

    /**
     * @var bool
     */
    public $wsInvoiceWithMail = false;

    /**
     * @var array
     */
    private $_vmixActions = array();

    /**
     * @var array
     */
    private $_dbActions = array();

    /**
     * @var array
     */
    private $_pacificActions = array();

    /**
     * Actions constructor.
     */
    public function __construct()
    {
        $this->table = 'RULAS2_FINAL_ACTIONS';
        $this->mapTable = 'RULA2_PROZESS_TO_ACTION';
        $this->pk = 'RFA_ID';
        $this->pk2 = 'P_ID';

        parent::init();

        if (!class_exists('DB', false)) {
            $dir = $this->ini->pacific->dir;
            $file = str_replace('#date#', date('ymdHi'), $this->ini->pacific->propertyfname);
            $this->vflyFile = $dir . $file;
        }
    }

    /**
     * @param null|string $type
     * @return ActionsItem[]
     */
    public function getAllActions($type = null)
    {
        try {
            $stmt = "select * from $this->table ";

            if ($type) {
                $stmt .= " WHERE RFA_TYPE = '$type'";
            }

            $res = $this->db->getAll($stmt . " order by rfa_name_in asc");

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen der Actions');
            }
            foreach ($res as $number => $values) {
                $item[$number] = new ActionsItem();
                foreach ($values as $key => $value) {
                    $temp = strtolower($key);
                    $item[$number]->$temp = $value;
                }
            }
            return $item;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param string $id
     * @param array $valPost
     * @return ActionsItem
     * @throws Exception
     */
    public function getActionByID($id, $valPost)
    {
        try {

            if ($id == null || $id == '' || $id <= 0) {
                throw new Exception('Keine Id vorhanden um Aktion zu halten');
            }

            $stmt = "select * from 
				 $this->table
				 where $this->pk = '$id'
				 order by RFA_NAME_IN asc";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception(
                    'Fehler beim holen der Aktionen fuer einen Prozess. MSG:: ' . $this->db->errorMsg()
                );
            }

            $item = new ActionsItem();
            if (!empty($res)) {
                foreach ($res[0] as $key => $value) {
                    $item->$key = $value;
                }
            }

            if ($valPost != null && $valPost != '') {
                $item->rfa_value = $valPost;
            }
        } catch (Exception $e) {
            throw new Exception($e->getMessage);
        }

        return $item;
    }

    /**
     * @param int $id
     * @return stdClass|null
     * @throws Exception
     */
    public function getProzessActions($id)
    {
        try {
            $this->countVmixActions = 0;
            $this->countPacificActions = 0;
            $this->countDbActions = 0;

            if (empty($id)) {
                throw new Exception('Keine Id vorhanden um Aktion zu halten');
            }

            $stmt = "select * from 
				 $this->table i , $this->mapTable j 
				 where i.$this->pk = j.$this->pk 
				 and j.$this->pk2 = '$id'
				 order by i.RFA_NAME_IN asc";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception(
                    'Fehler beim holen der Aktionen fuer einen Prozess. MSG:: ' . $this->db->errorMsg()
                );
            }

            foreach ($res as $values) {

                if (trim($values['RFA_TYPE']) == 'VMIX') {
                    $this->countVmixActions++;
                }
                if (trim($values['RFA_TYPE']) == 'DB') {
                    $this->countDbActions++;
                }
                if (trim($values['RFA_TYPE']) == 'Pacific') {
                    $this->countPacificActions++;
                }
                $item[$values[$this->pk]] = new ActionsItem();
                foreach ($values as $key => $value) {

                    if ($this->bookingState) {
                        $item[$values[$this->pk]]->bookingState = true;
                    }

                    if ($this->wsInvoiceWithMail) {
                        $item[$values[$this->pk]]->wsInvoiceWithMail = true;
                    }

                    $temp = strtolower($key);
                    $item[$values[$this->pk]]->$temp = $value;
                }
            }
        } catch (Exception $e) {
            throw new Exception($e->getMessage);
        }

        return $item;
    }

    /**
     * @param array $data
     * @param array $values
     * @param int $id
     * @throws Exception
     */
    public function insertProzessInToAction($data, $values, $id)
    {
        try {
            if ($data == null || $data == '' || $id == null || $id == '' || $id <= 0) {
                throw new Exception('Keine Id oder Daten um daten einzufuegen');
            }

            $this->db->BeginTrans();

            $this->deleteProzessFromAction($id);

            if (!is_array($data) || !is_array($values)) {

                $temp['RFA_ID'] = $data;
                $temp['P_ID'] = $id;
                $temp['RFA_VALUES'] = $values;

                $stat = $this->db->autoeexecute($this->mapTable, $temp, 'INSERT');

                if (!$stat || $this->db->errorMsg()) {
                    $this->db->RollBackTrans();
                    throw new Exception("Fehler beim schreiben in die $this->mapTable MSG::" . $this->db->errorMsg());
                }
            } else {

                foreach ($data as $key => $value) {
                    if ($value > 0) {
                        $temp['RFA_ID'] = $value;
                        $temp['RFA_VALUE'] = $values[$key];
                        $temp['P_ID'] = $id;

                        $stat = $this->db->autoexecute($this->mapTable, $temp, 'INSERT');

                        if ($this->db->errorMsg() || !$stat) {
                            $this->db->RollBackTrans();
                            throw new Exception(
                                "Fehler beim schreiben in die $this->mapTable MSG::" . $this->db->errorMsg()
                            );
                        }
                    }
                }
            }

            $this->db->CommitTrans(true);
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param int $id
     * @throws Exception
     */
    private function deleteProzessFromAction($id)
    {
        if (empty($id)) {
            throw new Exception('Keine Id uebermittelt um daten in der Action zuloeschen');
        }

        $stmt = "DELETE FROM $this->mapTable where $this->pk2 = '$id'";

        $this->db->execute($stmt);

        if ($this->db->errorMsg()) {
            throw new Exception('Loeschen der Action ist eine fehelr aufgetreten.  MSG::' . $this->db->errorMsg());
        }
    }

    /**
     * @param Booking_Fullbooking $booking
     * @param ActionsItem $actions
     * @throws Exception
     */
    public function processFinalActions(&$booking, &$actions)
    {

        $this->log->info('Start FinalActions');
        $this->_vmixActions = array();
        $this->_dbActions = array();
        $this->_pacificActions = array();

        try {

            if (is_array($actions)) {
                foreach ($actions as $action) {

                    if (strtolower($action->rfa_type) == 'vmix') {
                        array_push($this->_vmixActions, $action);
                    }
                    if (strtolower($action->rfa_type) == 'db') {
                        array_push($this->_dbActions, $action);
                    }
                    if (strtolower($action->rfa_type) == 'pacific') {
                        array_push($this->_pacificActions, $action);
                    }
                }
            }

            if (count($this->_dbActions) > 0) {
                try {
                    $this->db->BeginTrans();
                    $this->_processDBActions($booking);
                    $this->db->CommitTrans(true);
                } catch (Exception $e) {
                    $this->db->RollBackTrans();
                    throw new Exception($e->getMessage());
                }
            }

            if (count($this->_vmixActions) > 0) {
                try {
                    $this->_processVmixActions($booking);
                } catch (Exception $e) {
                    throw new Exception($e->getMessage());
                }
            }

            if (count($this->_pacificActions) > 0) {
                try {

                    $ret = $booking->getData();
                    $bookData = $ret['coreData']->getData();
                    if ($bookData['vbp_vfly_customer'] == 'KONTI') {
                        $this->_sunnyVflyAction($bookData);
                    }

                } catch (Exception $e) {
                    echo $e->getMessage();
                    throw new Exception($e->getMessage());
                }
            }
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param Booking_Bookingcore $booking
     * @return bool|void
     * @throws Exception
     */
    private function _processVmixActions(&$booking)
    {
        if (!empty($this->_vmixActions)) {
            $today = (new DateTimeImmutable())->format(self::DATETIMEFORMAT);
            $classification = BookingClassificationService::classifyByBookingNumber($booking->vt_bo_id);
            if ($classification->getClient() === 'ULTRA') {
                if ($this->log != null) {
                    $this->log->info(
                        ' das schreiben in Vmix wird nicht durchgefuehrt weil die Buchung 5xxx und 15xxxx ausgeschlossen sind'
                    );
                }
                return true;
            }
            if ($classification->getClient() === 'VFLY') {
                $updateData = [];
                $bookingId = $booking->vt_bo_id;
                /** @var ActionsItem $action */
                foreach ($this->_vmixActions as $action) {
                    switch ($action->rfa_name) {
                        case 'invoice_on':
                            $updateData = $this->getCustCalculationData($updateData, $today);
                            break;
                        case 'bemerkung':
                            // Update clob in method
                            $this->updateInfoFielInformation($bookingId, $action, $today);
                            break;
                        case 'booking_status':
                            $updateData = $this->getInternalBookingCondition($action, $updateData);
                            break;
                        case 'booking_processing':
                            $updateData = $this->getBookingState($action, $updateData);
                            break;
                        case 'inkasso':
                            $updateData = $this->getInkassoData($action, $updateData);
                            break;
                        case 'hotel_confirmed':
                            $updateData = $this->getHotelConfirmedData($updateData);
                            break;
                        case 'transfer_confirmed':
                            $updateData = $this->getTransferConfirmedData($updateData);
                            break;
                        case 'prepayment_on':
                            $updateData = $this->getPrepaymentInforamtion($updateData, $today);
                            break;
                        case 'balancepayment_on':
                            $updateData = $this->getBalancePaymentInformation($updateData, $today);
                            break;
                        case 'rulas_on':
                            $updateData = $this->getTravelDocumentInformation($updateData, $today);
                            break;
                    }
                }

                $this->log->info(
                    'Vfly write to vmix is disabled, write data to database after mapping. '.
                    print_r($updateData,1)
                );

                if (!empty($updateData)) {
                    $this->db->AutoExecute(self::BOOKINGTABLE, $updateData);
                }

            } else {

                $vmixData = [];
                foreach ($this->_vmixActions as $action) {

                    $vmixData[$action->rfa_name] = $action->rfa_value;

                    switch ($action->rfa_name) {
                        case 'invoice_on':
                            $vmixData['invoice_date'] = $today;
                            $vmixData['invoice_who'] = self::UPDATER;
                            break;
                        case 'rulas_on':
                            $vmixData['rulas_date'] = $today;
                            $vmixData['rulas_who'] = self::UPDATER;
                            break;
                        case 'prepayment_on':
                            $vmixData['prepayment_date'] = $today;
                            $vmixData['prepayment_who'] = self::UPDATER;
                            break;
                        case 'balancepayment_on':
                            $vmixData['balancepayment_date'] = $today;
                            $vmixData['balancepayment_who'] = self::UPDATER;
                            break;
                        case 'bemerkung':
                            if ($this->bookingState) {
                                $vmixData[$action->rfa_name] = $action->rfa_value . " (Buchungsstatus) => " . $today . " " . self::UPDATER;
                                break;
                            } elseif ($this->wsInvoiceWithMail) {
                                $vmixData[$action->rfa_name] = $action->rfa_value . " (Webservice) => " . $today . " " . self::UPDATER;
                                break;
                            } else {
                                $vmixData[$action->rfa_name] = $action->rfa_value . " (Rula2)=> " . $today . " " . self::UPDATER;
                            }
                            break;
                    }
                }
                $vt_bo_client = $classification->getClient();

                $data = array(
                    "booking_no" => $booking->vt_bo_id,
                    "VmixData" => $vmixData,
                    "vt_bo_client" => strtolower($vt_bo_client)
                );


                if ($this->log != null) {
                    $this->log->info(' das schreiben in Vmix wird druchgefuehrt: ' . print_r($data, 1));
                }

                $xmix = Xmix_Webservice::getInstance();
                $xmix->setURL('xmixbooks')->send($data);
                $xmix->Finish();
            }
        }
    }

    /**
     * @param Booking_Bookingcore $booking
     * @throws Exception
     */
    private function _processDBActions(&$booking)
    {
        if (empty($this->_dbActions)) {
            return;
        }

        $update = array();

        foreach ($this->_dbActions as $action) {
            list($table, $column) = explode(".", $action->rfa_name);

            if (!array_key_exists($table, $update)) {
                $update[$table] = array();
            }

            $valNew = null;
            $tempVal = null;

            $pattern = "/^(?:[\[]{1}[\{]{1}[a-zA-Z0-9_$,;.:-_\*\+]+[\}]{1}[\]]{1})$/";
            if (preg_match($pattern, $action->rfa_value)) {
                $pattern1 = "/[a-zA-Z_-]+/";
                if (preg_match($pattern1, $action->rfa_value, $valNew)) {
                    $tempVal = $valNew[0];
                    $action->rfa_value = $booking->coreData->$tempVal;
                }
            }

            if ($column == 'RU_BO_ID') {
                $value = $booking->coreData->vt_bo_id;
            } else {
                $value = $this->_parseFieldValue($action->rfa_value);
            }

            if ($column === 'VT_BO_VFLY_INTERNAL_RULA') {
                $value = $booking->coreData->vt_bo_vfly_internal_rula ?
                    $booking->coreData->vt_bo_vfly_internal_rula . "\n" . $value :
                    $value;
            }

            if ($column === 'VT_BO_INTERNALDESCR') {
                $value = $this->_parseFieldValue($action->rfa_value, 1);

                if ($this->bookingState) {
                    $value = "(Buchungsstatus) => " . $value;
                } elseif ($this->wsInvoiceWithMail) {
                    $value = "(Webservice) => " . $value;
                } else {
                    $value = "(Rula2) => " . $value;
                }

                $value = $booking->coreData->vt_bo_internaldescr . "\r\n" . $value;
            }
            /**
             * Special solution needed to send zgmail storno triggered by action or button in vmix
             */
            if ($column == 'VBP_ZG_SEND_BY_RULA2') {

                $stat = $this->makeFinalActionsForZgInDataBase(
                    $action,
                    $booking->coreData->vt_bo_id,
                    new Booking_BookingZgMeldung(),
                    $booking->coreData->mailChannel
                );

                if (!$stat) {
                    $this->log->error('Error by writing Data into ZG Tables.');
                }
            }

            $update[$table][$column] = $value;
        }

        foreach ($update as $tbl => $items) {

            $exec = 'UPDATE';
            $where = '';

            $dbStatusUpdate = array();

            switch ($tbl) {
                case 'VT_RULAS_SEND':
                    $exec = 'INSERT';
                    $dbStatusUpdate['RU_SEND_TIME'] = date(self::DATETIMEFORMAT);
                    $dbStatusUpdate['RU_EMAIL_TO'] = $booking->coreData->getCustomerEmail();
                    break;
                case 'VT_BOOKINGS':
                    $exec = 'UPDATE';
                    $where = 'VT_BO_ID = ' . $booking->vt_bo_id;
                    break;
                case 'VT_RECLAMATION':
                    $exec = 'UPDATE';
                    $where = 'VTR_BO_ID = ' . $booking->vt_bo_id;
                    break;
                case 'VT_RECLAMATION_COUPON':
                    $exec = 'UPDATE';
                    $where = 'VRCP_BO_ID = ' . $booking->vt_bo_id . ' and vrcp_no = ' . $booking->coreData->vrcp_no;
                    break;
                case 'VT_BOOKINGS_PARAMETER':
                    $exec = 'UPDATE';
                    $where = 'VBP_BO_ID = ' . $booking->vt_bo_id;
                    break;
                case 'VT_KUNDEN_MAIL_SEND':
                    $exec = 'INSERT';
                    $dbStatusUpdate['KMS_BO_ID'] = $booking->vt_bo_id;
                    $dbStatusUpdate['KMS_SEND_TIME'] = date(self::DATETIMEFORMAT);
                    $dbStatusUpdate['KMS_EMAIL'] = $booking->coreData->getCustomerEmail();
                    break;
                case 'VT_FLIGHTS_HISTORY':
                    $exec = 'UPDATE';
                    $where = 'vfh_bo_id = ' . $booking->vt_bo_id;
                    $dbStatusUpdate['VFH_FLIGHTRULAM_DATE'] = date(self::DATETIMEFORMAT);
                    break;
                case 'VT_EFT_INVALID':
                    $exec = 'UPDATE';
                    $where = 'vt_bo_id = ' . $booking->vt_bo_id;
                    break;
                case 'VT_CC_GUWID':
                    $exec = 'UPDATE';
                    $where = 'vt_bo_id = ' . $booking->vt_bo_id;
                    break;
                case 'VT_BOOKINGS_CHECKINDATA':
                    $exec = 'UPDATE';
                    $where = 'vbc_bo_id = ' . $booking->vt_bo_id;
                    break;
                case 'VT_BONITATSPRUEFUNG':
                    $exec = 'UPDATE';
                    $where = 'vt_bo_id = ' . $booking->vt_bo_id;
                    break;
                case 'ZAHLUNGSIMPORT_ITEM':
                    $exec = 'UPDATE';
                    $where = 'vt_bo_id = ' . $booking->vt_bo_id . ' and zi_mailsend = 0';
                    $dbStatusUpdate['ZI_MAILSEND_TIME'] = date(self::DATETIMEFORMAT);
                    break;
            }

            foreach ($items as $col => $v) {
                $dbStatusUpdate[$col] = $v;
                if ($this->log != null) {
                    $this->log->info('Folgende Werte in der DB upgdated. ' . $col . '--' . $v);
                }
            }

            $this->log->info('Folgende werte in der DB werden upgedated: DB' . print_r($dbStatusUpdate, 1));

            if ($where) {
                $stat = $this->db->AutoExecute($tbl, $dbStatusUpdate, $exec, $where);
            } else {
                $stat = $this->db->AutoExecute($tbl, $dbStatusUpdate, $exec);
            }

            if (!$stat || $this->db->errorMsg()) {
                throw new Exception("Fehler beim processFinalActions MSG::" . $this->db->errorMsg());
            }
        }
    }

    /**
     * @param string $key
     * @param int $time
     * @return mixed
     * @throws Exception
     */
    private function _parseFieldValue($key, $time = 0)
    {
        $ret = $key;

        if ($time == 1) {
            if (stristr($key, 'today')) {
                $ret = str_replace('today', date('Y-m-d H:i'), $ret);
            }
        } else {
            if (stristr($key, 'today')) {
                $ret = str_replace('today', $this->calculateDate('today'), $ret);
            }

        }

        return $ret;
    }

    /**
     * @param string $date
     * @return bool|false|string
     * @throws Exception
     */
    public function calculateDate($date)
    {
        $obj = new Rulas2_RulesEngine_GenerateSql();
        return $obj->getDateForValue($date);
    }

    /**
     * @return mixed
     */
    public function getCountPacificActions()
    {
        //available after getProzessActions
        return $this->countPacificActions;
    }

    /**
     * @return string
     */
    public function getCountDBActions()
    {
        //available after getProzessActions
        return $this->countDbActions;
    }

    /**
     * @return string
     */
    public function getCountVmixActions()
    {
        //available after getProzessActions
        return $this->countVmixActions;
    }

    /**
     * @param array $booking
     */
    public function _sunnyVflyAction(array $booking)
    {
        $mapTo = array(
            'FZ Mail' =>
                array(
                    'code' => 'FZ',
                    'message' => utf8_decode('Status::Flugzeitenänderungsmail an Kunden am')
                ),
            'USA' =>
                array(
                    'code' => 'US',
                    'message' => utf8_decode('Status::USA Mail an Kunden am')
                ),
            'USA Reminder' =>
                array(
                    'code' => 'UR',
                    'message' => utf8_decode('Status::USA Reminder Mail an Kunden am')
                ),
            'Notfallnummer' =>
                array(
                    'code' => 'KM',
                    'message' => utf8_decode('Status::Notfallnummermail an Kunden am')
                ),
            'Reiseunterlagen' =>
                array(
                    'code' => 'RULA',
                    'message' => utf8_decode('Status::Reiseunterlagen an Kunden am')
                ),
            'Rechnung' =>
                array(
                    'code' => 'RE',
                    'message' => utf8_decode('Status::Rechnung an Kunden am')
                ),
            'Stornorechnung' =>
                array(
                    'code' => 'STORNO',
                    'message' => utf8_decode('Status::Stornorechnung an Kunden am')
                ),
            'Lastschriftinvalid' =>
                array(
                    'code' => 'LI',
                    'message' => utf8_decode('Status::Lastschrift Invalid Mail an')
                ),
            'Zahlungserinnerung AZ' =>
                array(
                    'code' => 'ZERINAZ',
                    'message' => utf8_decode('Status::Zahlungserinnerung Anzahlungsmail an ')
                ),
            'Zahlungserinnerung RZ' =>
                array(
                    'code' => 'ZERINRZ',
                    'message' => utf8_decode('Status::Zahlungserinnerung Testzahlungsmail an ')
                ),
            'Buchungsnotiz' =>
                array(
                    'code' => 'NOTIZ',
                    'message' => utf8_decode('Status::Buchungsnotiz an')
                )

        );

        if (is_array($this->_pacificActions) && count($this->_pacificActions) > 0) {
            foreach ($this->_pacificActions as $value) {
                try {

                    $temp = array(
                        'svpi_bo_id' => $booking['vt_bo_id'],
                        'svpi_from' => $mapTo[$value->rfa_name]['code'],
                        'svpi_time' => date(self::DATETIMEFORMAT),
                        'svpi_message' => $mapTo[$value->rfa_name]['message'] . " " . date('d.m.Y H:i')
                    );
                    $this->insertIntoSunnyPropertyTable($temp);
                } catch (Exception $e) {
                    $this->log->err($e->getMessage());
                }
            }
        }
    }

    /**
     * @param array $data
     * @return bool
     * @throws Exception
     */
    private function insertIntoSunnyPropertyTable(array $data)
    {
        $table = 'SUNNY_VFLY_PROPERTY_ITEMS';

        $stat = $this->db->autoExecute($table, $data, 'INSERT');

        if ($stat && !$this->db->errorMsg()) {
            return true;
        } else {
            throw new Exception('Fehler beim Update der Daten im '
                . __METHOD__ . ' DATA::' . print_r($data, 1) . ' MSG::' . $this->db->errorMsg());
        }
    }

    /**
     * @param ActionsItem $action
     * @param string $bookingno
     * @param Booking_BookingZgMeldung $zgObj
     * @param string $emailChannel
     * @return bool
     */
    private function makeFinalActionsForZgInDataBase($action, $bookingno, $zgObj, $emailChannel = 'incagency')
    {
        $classification = BookingClassificationService::classifyByBookingNumber($bookingno);

        if (empty($classification->getClient())) {
            return false;
        }

        if (
            $action->rfa_value == 1
            && $this->addDataToZgConfs($bookingno, $zgObj, $emailChannel)
            && $this->addDataToZgHistory($bookingno, $zgObj)
        ) {
            return true;
        }

        return false;
    }

    /**
     * @param string $bookingno
     * @param Booking_BookingZgMeldung $obj
     * @param string $emailChannel
     * @return bool
     */
    private function addDataToZgConfs($bookingno, $obj, $emailChannel)
    {
        /**
         * DUMMY Data to trigger storno zg mail for future. vto-app storno zg mail need this data to trigger.
         */
        return $obj->insertIntoZgConfs(
            [
                'VT_BO_ID' => $bookingno,
                'VT_ZGC_NO' => 1,
                'VT_ZGC_TYPE' => $this->getChannelType($emailChannel),
                'VT_ZGC_CREATEDDATE' => date(self::DATETIMEFORMAT),
                'VT_ZGC_INCAGENTMAILS' => 'it@vtours.de',
                'VT_ZGC_CONFIRMED' => 1,
                'VT_ZGC_REJECTED' => 0,
                'VT_ZGC_CONFPROVIDESOF' => 'R2 DUMMY SYSTEM',
                'VT_ZGC_CREATED' => date(self::DATETIMEFORMAT),
                'VT_ZGC_HASH' => md5($bookingno . 'Zielgebiet' . time() . rand())
            ]
        );

    }

    /**
     * @param string $bookingno
     * @param Booking_BookingZgMeldung $obj
     * @return bool
     */
    private function addDataToZgHistory($bookingno, $obj)
    {
        /**
         * DUMMY Data to trigger storno zg mail for future. vto-app storno zg mail need this data to trigger.
         */
        return $obj->insertIntoZgHistory(
            [
                'VTZGE_BO_ID' => $bookingno,
                'VTZGE_DATE' => date(self::DATETIMEFORMAT),
                'VTZGE_TO' => 'it@vtours.de',
                'VTZGE_MA' => 'R2 DUMMY SYSTEM',
                'VTZGE_HTML' => 'DUMMY EINTRAG',
                'VTZGE_TYPE' => 'Zielgebiet',
                'VTZGE_CANCELATION' => 0
            ]
        );

    }

    /**
     * @param string $emailChannel
     * @return string
     */
    private function getChannelType($emailChannel)
    {
        $type = "Hotel";

        if (strtolower($emailChannel) == 'tagency') {
            $type = "Transfer";
        }

        if (strtolower($emailChannel) == 'agency_tagency') {
            $type = "Hotel_Transfer";
        }

        if (strtolower($emailChannel) == 'incagency') {
            $type = "Hotel";
        }

        return $type;
    }

    /**
     * @param array $updateData
     * @param string $today
     * @return array
     */
    private function getCustCalculationData($updateData, $today)
    {
        $updateData['VT_BO_CUSTCALCULATION'] = 1;
        $updateData['VT_BO_CUSTCALCULATIONDATE'] = $today;
        $updateData['VT_BO_CALCULATIONPROVIDESOF'] = 'SYSTEM';

        return $updateData;
    }

    /**
     * @param string $bookingid
     * @param ActionsItem $action
     * @param string $today
     * @return void
     */
    private function updateInfoFielInformation($bookingid, $action, $today)
    {
        if ($this->bookingState) {
            $msg = $action->rfa_value . " (Buchungsstatus) => " . $today ." ". self::UPDATER;
        } elseif ($this->wsInvoiceWithMail) {
            $msg = $action->rfa_value . " (Webservice) => " . $today ." ". self::UPDATER;
        } else {
            $msg = $action->rfa_value . " (Rula2)=> " . $today . " ".self::UPDATER;
        }

        $stmt = "update vt_bookings 
                    set VT_BO_INTERNALDESCR = VT_BO_INTERNALDESCR || CASE WHEN VT_BO_INTERNALDESCR IS NOT NULL  THEN chr(10) else '' END  ||'$msg'
                 where vt_bo_id = :bookingNumber";

        $this->db->Execute(
            $stmt,
            [
                'bookingNumber' => $bookingid
            ]
        );
    }

    /**
     * @param ActionsItem $action
     * @param array $updateData
     * @return array
     */
    private function getInternalBookingCondition($action, $updateData)
    {
        $updateData['VT_BO_INTERNALBOOKINGCONDITION'] = $action->rfa_value;
        return $updateData;
    }

    /**
     * @param ActionsItem $action
     * @param array $updateData
     * @return array
     */
    private function getBookingState($action, $updateData)
    {
        $updateData['VT_BO_INTERNALTREATMENTCONDITI'] = $action->rfa_value;
        return $updateData;
    }

    /**
     * @param array $updateData
     * @param string $today
     * @return array
     */
    private function getPrepaymentInforamtion($updateData, $today)
    {
        $updateData['VT_BO_PREPAYMENT'] = 1;
        $updateData['VT_BO_PREPAYMENTDATE'] = $today;
        $updateData['VT_BO_PREPAYMENTPROVIDESOF'] = 'SYSTEM';
        return $updateData;
    }

    /**
     * @param array $updateData
     * @param string $today
     * @return array
     */
    private function getBalancePaymentInformation($updateData, $today)
    {
        $updateData['VT_BO_BALANCEPAYMENT'] = 1;
        $updateData['VT_BO_BALANCEDATE'] = $today;
        $updateData['VT_BO_BALANCEPROVIDESOF'] = 'SYSTEM';
        return $updateData;
    }

    /**
     * @param array $updateData
     * @param string $today
     * @return array
     */
    private function getTravelDocumentInformation($updateData, $today)
    {
        $updateData['VT_BO_TRAVELDOCUMENTS'] = 1;
        $updateData['VT_BO_TRAVELDOCUMENTSDATE'] = $today;
        $updateData['VT_BO_CUSTDOCPROVIDESOF'] = 'SYSTEM';
        return $updateData;
    }

    /**
     * @param ActionsItem $action
     * @param array $updateData
     * @return array
     */
    private function getInkassoData(ActionsItem $action, array $updateData)
    {
        $updateData['VT_BO_CASHING'] = $action->rfa_value;
        return $updateData;
    }

    /**
     * @param array $updateData
     * @return array
     */
    private function getHotelConfirmedData(array $updateData)
    {
        $updateData['VT_BO_HOTELCONFIRMED'] = 1;
        return $updateData;
    }

    /**
     * @param array $updateData
     * @return array
     */
    private function getTransferConfirmedData(array $updateData)
    {
        $updateData['VT_BO_TRANSFERCONFIRMED'] = 1;
        return $updateData;
    }
}