<?php
namespace Service;

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2021 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

/**
 * Class MobilePhoneCleanerAndValidator
 */
class MobilePhoneCleanerAndValidator
{

    /**
     * @var string[]
     */
    private $germanPhoneCountryCodes = [
        '049',
        '+49',
        '0049',
        '49'
    ];

    /**
     * @var string[]
     */
    private $austrianPhoneCountryCodes = [
        '043',
        '+43',
        '0043',
    ];

    /**
     * @var string[]
     */
    private $switzerlandPhoneCountryCodes = [
        '041',
        '+41',
        '0041',
    ];

    /**
     * @var string
     */
    private $germanCountryCode = '49';

    /**
     * @var string
     */
    private $austrianCountryCode = '43';

    /**
     * @var string
     */
    private $switzerlandCountryCode = '41';


    /**
     * @var string[]
     */
    private $replaceCharactersWithSpace = [
        '-',
        ' ',
        '(',
        ')',
        '/',
        '\\',
        '.',
    ];

    /**
     * `/^(\+49|49|049|0049)?(0)?(15|16|17)+[0-9]{7,12}$/`;
     * @param string $mobileNumber
     * @return bool
     */
    public function isGermanMobilePhoneNumber($mobileNumber)
    {
        $mobileNumber = str_replace($this->replaceCharactersWithSpace, '', $mobileNumber);
        $pattern = '/^(\+49|49|049|0049)?(0)?(15|16|17)+[0-9]{7,12}$/';
        return (bool)preg_match($pattern, $mobileNumber);
    }

    /**
     * @param string $mobileNumber
     * @return string
     */
    public function getCleanGermanNumber($mobileNumber)
    {
            $mobileNumber = str_replace($this->replaceCharactersWithSpace, '', $mobileNumber);
            array_walk(
                $this->germanPhoneCountryCodes, function ($code) use (&$mobileNumber) {

                if (substr($mobileNumber, 0, strlen($code)) === $code) {
                    $mobileNumber = substr_replace($mobileNumber, $this->germanCountryCode, 0, strlen($code));
                }
            });

            if (substr($mobileNumber, 0, 1) === '0') {
                $mobileNumber = substr_replace($mobileNumber, $this->germanCountryCode, 0, 1);
            }

            if (substr($mobileNumber, 0, 2) !== $this->germanCountryCode) {
                $mobileNumber = $this->germanCountryCode . $mobileNumber;
            }

            return $mobileNumber;
    }

    /**
     * @param string $mobileNumber
     * @return string
     */
    public function getCleanAustrianNumber($mobileNumber)
    {

            $mobileNumber = str_replace($this->replaceCharactersWithSpace, '', $mobileNumber);
            array_walk(
                $this->austrianPhoneCountryCodes, function ($code) use (&$mobileNumber) {

                if (substr($mobileNumber, 0, strlen($code)) === $code) {
                    $mobileNumber = substr_replace($mobileNumber, $this->austrianCountryCode, 0, strlen($code));
                }
            });

            if (substr($mobileNumber, 0, 1) === '0') {
                $mobileNumber = substr_replace($mobileNumber, $this->austrianCountryCode, 0, 1);
            }

            if (substr($mobileNumber, 0, 2) !== $this->austrianCountryCode) {
                $mobileNumber = $this->austrianCountryCode . $mobileNumber;
            }

            return $mobileNumber;
    }

    /**
     * @param string $mobileNumber
     * @return string
     */
    public function getCleanSwitzerLandNumber($mobileNumber)
    {
            $mobileNumber = str_replace($this->replaceCharactersWithSpace, '', $mobileNumber);
            array_walk(
                $this->switzerlandPhoneCountryCodes, function ($code) use (&$mobileNumber) {

                if (substr($mobileNumber, 0, strlen($code)) === $code) {
                    $mobileNumber = substr_replace($mobileNumber, $this->switzerlandCountryCode, 0, strlen($code));
                }
            });

            if (substr($mobileNumber, 0, 1) === '0') {
                $mobileNumber = substr_replace($mobileNumber, $this->switzerlandCountryCode, 0, 1);
            }

            if (substr($mobileNumber, 0, 2) !== $this->switzerlandCountryCode) {
                $mobileNumber = $this->switzerlandCountryCode . $mobileNumber;
            }
            return $mobileNumber;

    }

    /**
     * `/^(\+43|43|043|0043)?(0)?(650|651|651|652|659|660|661|663|664|665|666|667|668|669|67|69)+[0-9]{7,12}$/`
     * @param string $mobileNumber
     * @return bool
     */
    public function isAustrianMobilePhoneNumber($mobileNumber)
    {
        $mobileNumber = str_replace($this->replaceCharactersWithSpace, '', $mobileNumber);
        $pattern = '/^(\+43|43|043|0043)?(0)?(650|651|651|652|659|660|661|663|664|665|666|667|668|669|67|69)+[0-9]{7,12}$/';
        return (bool)preg_match($pattern, $mobileNumber);
    }

    /**
     * `/^(\+41|41|041|0041)?(0)?(76|77|78|79)+[0-9]{7,12}$/`;
     * @param string $mobileNumber
     * @return bool
     */
    public function isSwitzerlandMobilePhoneNumber($mobileNumber)
    {
        $mobileNumber = str_replace($this->replaceCharactersWithSpace, '', $mobileNumber);
        $pattern = '/^(\+41|41|041|0041)?(0)?(76|77|78|79)+[0-9]{7,12}$/';
        return (bool)preg_match($pattern, $mobileNumber);
    }

    /**
     * @param  $mobileNumber
     * @return string|bool
     */
    public function checkPhoneNumber($mobileNumber)
    {
        if ($this->isGermanMobilePhoneNumber($mobileNumber)) {
            return $this->getCleanGermanNumber($mobileNumber);
        }

        if ($this->isAustrianMobilePhoneNumber($mobileNumber)) {
            return $this->getCleanAustrianNumber($mobileNumber);
        }

        if($this->isSwitzerlandMobilePhoneNumber($mobileNumber)) {
            return $this->getCleanSwitzerLandNumber($mobileNumber);
        }

        return false;
    }
}
