<?php

namespace Vtours\Rula2Engine\Booking;

/**
 *  LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 *  and is private by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license
 *  is a violation of the license agreement and will be prosecuted by
 *  civil and criminal law.
 *
 * @copyright 2021 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

use Lifestyle\BookingClassification\Service\BookingClassificationService;
use Vtours\Rula2Engine\Rula2Std;

/**
 * Class BookingSql
 */
class BookingSql extends Rula2Std
{
    /**
     * @var Zend_Config_Ini
     */
    private $config;

    /**
     * @var string[]
     */
    protected $convertToLower = [
        'VT_BO_BA_AGEMAIL',
        'VT_BO_CUSTOMER_EMAIL',
        'VT_BO_BA_AGEMAIL_ANZ',
    ];

    public function init()
    {
        parent::init();

        $this->config = new Booking_BookingSqlConfig();
        $this->config->setConfigDataForCoreSelect($this->ini->client);
    }

    /**
     * @param int $vt_bo_id
     * @param string $vonbis
     * @param string $zusatz
     * @return array|bool
     * @throws Exception
     */
    public function getCoreData($vt_bo_id = null, $vonbis = null, $zusatz = null)
    {
        try {
            $this->init();

            $sql = "select ";
            $sql .= $this->config->get_vt_bookings();
            $zusatz->tables['VT_BOOKINGS'] = 0;

            $sql .= $this->config->get_vt_hotels();
            $zusatz->tables['VT_HOTELS'] = 0;

            $sql .= $this->config->get_exchange_rates();
            $zusatz->tables['VT_EXCHANGE_RATES_LIST'] = 0;

            if ($zusatz->tables['VT_FLIGHT_SAVED']) {
                $sql .= $this->config->get_vt_flight_saved();
                $zusatz->tables['VT_FLIGHT_SAVED'] = 0;
            } else {
                $zusatz->tables['VT_FLIGHT_SAVED'] = 1;
            }

            if ($zusatz->tables['VT_HISTORY']) {
                $sql .= $this->config->get_vt_history();
                $zusatz->tables['VT_HISTORY'] = 0;
            } else {
                $zusatz->tables['VT_HISTORY'] = 1;
            }

            if ($zusatz->tables['VT_RECLAMATION']) {
                $sql .= $this->config->get_vt_reclamation();
                $zusatz->tables['VT_RECLAMATION'] = 0;
            } else {
                $zusatz->tables['VT_RECLAMATION'] = 1;
            }

            if ($zusatz->tables['VT_RECLAMATION_COUPON']) {
                $sql .= $this->config->get_vt_reclamation_coupon();
                $zusatz->tables['VT_RECLAMATION_COUPON'] = 0;
            } else {
                $zusatz->tables['VT_RECLAMATION_COUPON'] = 1;
            }

            if ($zusatz->tables['VT_RECLAMATION_REFUND']) {
                $sql .= $this->config->get_vt_reclamation_refund();
                $zusatz->tables['VT_RECLAMATION_REFUND'] = 0;
            } else {
                $zusatz->tables['VT_RECLAMATION_REFUND'] = 1;
            }

            if ($zusatz->tables['VT_RULAS_SEND']) {
                $sql .= $this->config->get_vt_rulas_send();
                $zusatz->tables['VT_RULAS_SEND'] = 0;
            } else {
                $zusatz->tables['VT_RULAS_SEND'] = 1;
            }

            if ($zusatz->tables['VT_BOOKINGS_PARAMETER']) {
                $sql .= $this->config->get_vt_bookings_parameter();
                $zusatz->tables['VT_BOOKINGS_PARAMETER'] = 0;
            } else {
                $zusatz->tables['VT_BOOKINGS_PARAMETER'] = 1;
            }

            if ($zusatz->tables['VT_FLIGHTS_HISTORY']) {
                $sql .= $this->config->get_vt_flights_history();
                $zusatz->tables['VT_FLIGHTS_HISTORY'] = 0;
            } else {
                $zusatz->tables['VT_FLIGHTS_HISTORY'] = 1;
            }

            if ($zusatz->tables['VT_EFT_INVALID']) {
                $sql .= $this->config->get_vt_eft_invalid();
                $zusatz->tables['VT_EFT_INVALID'] = 0;
            } else {
                $zusatz->tables['VT_EFT_INVALID'] = 1;
            }

            if ($zusatz->tables['VT_CC_GUWID']) {
                $sql .= $this->config->get_vt_cc_guwid();
                $zusatz->tables['VT_CC_GUWID'] = 0;
            } else {
                $zusatz->tables['VT_CC_GUWID'] = 1;
            }

            if ($zusatz->tables['VT_FLIGHTS']) {
                $sql .= $this->config->get_vt_flights_first();
                $zusatz->tables['VT_FLIGHTS'] = 0;
            } else {
                $zusatz->tables['VT_FLIGHTS'] = 1;
            }

            if ($zusatz->tables['VT_DUNNINGS']) {
                $sql .= $this->config->get_vt_dunnings();
                $zusatz->tables['VT_DUNNINGS'] = 0;
            } else {
                $zusatz->tables['VT_DUNNINGS'] = 1;
            }

            if ($zusatz->tables['VT_BOOKINGS_CHECKINDATA']) {
                $sql .= $this->config->get_vt_bookings_checkin();
                $zusatz->tables['VT_BOOKINGS_CHECKINDATA'] = 0;
            } else {
                $zusatz->tables['VT_BOOKINGS_CHECKINDATA'] = 1;
            }

            if ($zusatz->tables['VT_BONITATSPRUEFUNG']) {
                $sql .= $this->config->get_boni_data();
                $zusatz->tables['VT_BONITATSPRUEFUNG'] = 0;
            } else {
                $zusatz->tables['VT_BONITATSPRUEFUNG'] = 1;
            }

            if ($zusatz->tables['ZAHLUNGSIMPORT_ITEM']) {
                $sql .= $this->config->get_zahlungs_import();
                $zusatz->tables['ZAHLUNGSIMPORT_ITEM'] = 0;
            } else {
                $zusatz->tables['ZAHLUNGSIMPORT_ITEM'] = 1;
            }


            $sql = substr($sql, 0, -1);
            $sql .= " from vt_bookings b";

            if ($vonbis !== null && $zusatz->sql == null) {
                $sql .= " 
                LEFT JOIN VT_HOTELS h ON HT_BO_ID = b.VT_BO_ID AND (HT_HOTEL_NO = 1 OR HT_HOTEL_NO is null)
                LEFT JOIN VT_EXCHANGE_RATES_LIST ER ON ER_BO_ID = b.VT_BO_ID
                LEFT JOIN AIRPORT_CODES ac on ac.LC = h.HT_DESTAIRPORT
                LEFT JOIN AIRPORT_CODES ac1 on ac1.LC = b.vt_bo_destination
			    LEFT JOIN VT_PACKAGETOURS p on (b.vt_bo_id = p.vt_bo_id and p.vt_pt_no = 1)
			    WHERE to_char(b.vt_bo_beginn, 'YYYY-MM-DD')>= '" . $vonbis['von'] . "'
			    AND to_char(b.vt_bo_beginn, 'YYYY-MM-DD') <= '" . $vonbis['bis'] . "'";
            } elseif ($vt_bo_id == null && $zusatz->sql == null) {
                $sql .= " 
                LEFT JOIN VT_HOTELS h ON HT_BO_ID = b.VT_BO_ID AND (HT_HOTEL_NO = 1 OR HT_HOTEL_NO is null)
                LEFT JOIN VT_EXCHANGE_RATES_LIST ER ON ER_BO_ID = b.VT_BO_ID
			    LEFT JOIN AIRPORT_CODES ac on ac.LC = h.HT_DESTAIRPORT
			    LEFT JOIN AIRPORT_CODES ac1 on ac1.LC = b.vt_bo_destination
			    LEFT JOIN VT_PACKAGETOURS p on (b.vt_bo_id = p.vt_bo_id and p.vt_pt_no = 1)
			    where	to_char(b.vt_bo_beginn, 'YYYY-MM-DD') <= to_char(sysdate + 21, 'YYYY-MM-DD')
			    and to_char(b.vt_bo_beginn, 'YYYY-MM-DD') >= to_char(sysdate, 'YYYY-MM-DD')
			    and b.VT_BO_BALANCEPAYMENT = 1
			    and b.vt_bo_storno = 0";
            } elseif ($vt_bo_id !== null && $zusatz->sql == null) {
                $sql .= " 
                LEFT JOIN VT_HOTELS h ON HT_BO_ID = b.VT_BO_ID AND (HT_HOTEL_NO = 1 OR HT_HOTEL_NO is null)
                LEFT JOIN VT_EXCHANGE_RATES_LIST ER ON ER_BO_ID = b.VT_BO_ID
                LEFT JOIN AIRPORT_CODES ac on ac.LC = h.HT_DESTAIRPORT
                LEFT JOIN AIRPORT_CODES ac1 on ac1.LC = b.vt_bo_destination
			    LEFT JOIN VT_PACKAGETOURS p on (b.vt_bo_id = p.vt_bo_id and p.vt_pt_no = 1)
			    where b.VT_BO_ID = '$vt_bo_id'";
            } elseif ($zusatz->sql != null) {
                $sql .= $zusatz->joins . " where " . $zusatz->sql . " ";

                foreach ($zusatz->special as $value) {
                    $sql .= $value . " ";
                }
            }

            if ($zusatz->sql == null) {
                $res = $this->db->getRow($sql);
            } else {
                $res = $this->db->getAll($sql);
            }

            if (!class_exists('DB', false)) {
                $this->log->info($sql);
            }
            if ($this->db->errorMsg()) {
                throw new Exception($this->db->errorMsg());
            }

        } catch (Exception $e) {
            throw new Exception('Fehler beim holen der BookingCoreData:: MSG::' .
                $e->getMessage() . ' SQL::' . $sql);
        }

        $data = $zusatz->sql ? $this->cleanArray($res) : $this->cleanArray($res, MB_CASE_TITLE, true);

        $data['config'] = $zusatz;

        return $data;
    }

    /**
     * @param Booking_Bookingcore $data
     * @return bool
     * @throws Exception
     */
    public function getHotelData(Booking_Bookingcore $data)
    {
        $this->init();

        if (strtolower($data->getHotelCode()) == 'manuell') {
            $teilSql = "LEFT JOIN VT_HOTEL_ADDRESS_INDI ON HTA_BOID = " . $data->vt_bo_id;
        } else {
            $teilSql = "LEFT JOIN VT_HOTEL_ADDRESS ON HT_HOTELCODE = HTA_HOTELCODE ";
        }

        if ($data->vt_bo_bookingtype != 'P') {
            $stmt = "SELECT
                        VT_BO_ID,
                        HT_HOTELNAME,
                        HT_EXTERNALPROZESSNO,
                        HTA_STREET,
                        HTA_ZIP,
                        HTA_CITY,
                        HTA_COUNTRY,
                        HTA_FAX,
                        HTA_PHONE,
                        HTA_EMAIL,
                        HTA_URLADDRESS,
                        HT_HOTEL_NO,
                        HT_BRANDNAME,
                        HT_ROOMCODE,
                        HT_ASSINGLEROOM,
                        HT_HOTELLOCCODE,
                        HT_HOTELLOCATION,
                        HT_ALLOCATION,
                        TO_CHAR(HT_BEGINN,'DD.MM.YYYY') as HT_BEGINN,
                        TO_CHAR(HT_END,'DD.MM.YYYY') as HT_END,
                        TO_CHAR(HT_END - HT_BEGINN) as DURATION,
                        HT_INCAGENT,
                        HT_TRANSFERAGENT,
                        HT_TRANSFERAGENTNAME,
                        HT_TRANSFERTYPE,
                        HT_HOTELCODE,
                        HT_ROOMTYPE||' ('||trim(substr(ht_crsroomcode,0,3))||') ' as HT_ROOMTYPE,
                        HT_CRSROOMCODE,
                        HT_ROOMVIEW,
                        HT_ROOMACCOMODATION,
                        HT_BOOKINFO,
                        HT_INCAGENTNAME,
                        HT_MEALTYPE,
                        HT_TRANSFERTYPE,
                        VT_BO_DESTINATION,
                        HT_CATEGORY,
                        VT_BO_HOTELPRICE,
                        HT_BOOKEDCURRENCY,
                        HT_ROOMDESCRIPTION, 
                        (ROUND(VT_BO_HOTELPRICE / (SELECT CASE 
                                                      WHEN ER_CONVERSIONRATE IS NOT NULL 
                                                      THEN ER_CONVERSIONRATE 
                                                   ELSE 1 END as ER_CONVERSIONRATE
                                              FROM VT_EXCHANGE_RATES_LIST 
                                              WHERE ER_BO_ID = VT_BO_ID 
                                              AND TRIM(ER_EXCHANGECURRENCYCODE) = TRIM(HT_BOOKEDCURRENCY)),2)) as VT_BO_HOTELPRICE_FREMD
                    FROM VT_BOOKINGS
                    LEFT JOIN VT_HOTELS ON VT_BO_ID = HT_BO_ID
                    $teilSql
                    WHERE VT_BO_ID = '" . $data->vt_bo_id . "'
                    AND VT_BO_ID = HT_BO_ID
                    ORDER BY HT_HOTEL_NO";

            $res = $this->db->getAll($stmt); //Koennen mehrere Hotelzimmer sein

            foreach ($res as &$row) {
                $utils = new HtCategoryUtil ($row['HT_CATEGORY']);
                $row['HT_CATEGORY'] = $utils->getAsStars();
            }

            if (count($res) > 1) {
                $res[0]['anzahlZimmer'] = count($res);
            }
            if ($this->db->errorMsg()) {
                return false;
            }
        } else {
            // latest vt_bo_bookingtype with 'P' from 10.2014
            return false;
        }
        $res = $this->cleanArray($res, MB_CASE_UPPER);

        $res[0]['dataForInvoice'] = $this->makeHotelDataForInvoice($res);

        return $res;
    }

    /**
     * @param int $vt_bo_id
     * @return Booking_Pricecorrections|bool
     * @throws Exception
     */
    public function getPriceCorrections($vt_bo_id)
    {
        $this->init();

        if (empty($vt_bo_id)) {
            throw new Exception('Keine Buchungsnummer erhalten');
        }
        $stmt = "select * 
                  from vt_pricecorrections 
                 where vt_bo_id = $vt_bo_id";

        $res = $this->db->getAll($stmt);

        if (count($res) > 0) {
            foreach ($res as $key => $values) {
                if ($values['VT_PRICE_DESCRIPTION']) {
                    $values['VT_PRICE_DESCRIPTION'] =
                        mb_convert_case($values['VT_PRICE_DESCRIPTION'], MB_CASE_UPPER, "UTF-8");
                }
                $obj[$key] = new Booking_Pricecorrections($values);
                $obj[$key]->initValues();
            }
        } else {
            return false;
        }

        return $obj;
    }

    /**
     * @param Booking_Bookingcore $data
     * @return bool
     */
    public function getParticipantsData(Booking_Bookingcore $data)
    {
        $this->init();

        $statBooker = false;

        $stmt = "SELECT
                    CASE WHEN LOWER(TRIM(PA_FIRSTNAME)) = "
            . $this->db->qstr(strtolower(trim($data->getCustomerFirstname())))
            . " AND LOWER(TRIM(PA_NAME)) = " . $this->db->qstr(strtolower(trim($data->getCustomerName()))) . "
                        THEN '1'
                        ELSE '0'
                    END as Booker,
                    PA_NAME,
                    PA_FIRSTNAME,
                    CASE WHEN UPPER(PA_ADDRESS) = 'D'
                        THEN 'Frau'
                        WHEN UPPER(PA_ADDRESS) = 'H'
                        THEN 'Herr'
                        WHEN UPPER(PA_ADDRESS) = 'K'
                        THEN 'CHD'
                        WHEN UPPER(PA_ADDRESS) = 'KIND'
                        THEN 'CHD'
                        WHEN UPPER(PA_ADDRESS) = 'B'
                        THEN 'BABY'
                        ELSE PA_ADDRESS
                    END as PA_ADDRESS,
                    PA_AGE,
                    TO_CHAR(VT_BO_BEGINN,'DD.MM.YYYY') as VT_BO_BEGINN,
                    VT_BO_DESTINATION,
                    PA_PARTICIPANT_NO
                    FROM VT_PARTICIPANTS , VT_BOOKINGS
                    WHERE PA_BO_ID = '" . $data->vt_bo_id . "'
                    AND VT_BO_ID = PA_BO_ID ORDER BY PA_PARTICIPANT_NO ASC";

        $res = $this->db->getAll($stmt);

        foreach ($res as $index => $val) {
            foreach ($val as $key => $value) {
                $res[$index][$key] = str_replace(array('ö', 'ü', 'ä'), array('Ö', 'Ü', 'Ä'),
                    $value);
            }
            if ($val['BOOKER'] == 1) {
                $statBooker = true;
            }
            if ($val['PA_ADDRESS'] == 'CHD') {
                $data->bookingHasChildren = 1;
            }
            if ($val['PA_ADDRESS'] == 'BABY') {
                $data->bookingHasBabies = 1;
            }
        }
        if (!$statBooker) {
            $res[0]['BOOKER'] = 1;
        }
        if ($this->db->errorMsg()) {
            return false;
        } else {
            return $res;
        }
    }

    /**
     * @param Booking_Bookingcore $data
     * @return bool
     */
    public function getSavedFlights(Booking_Bookingcore $data)
    {

        if ($data->vt_bo_bookingtype != 'P') {
            $stmt = "SELECT
			VT_FLI_SV_NUMBER,
			VT_FLI_SV_KEY,
			VT_FLI_SV_STORNOFLAG,
			VT_FLI_SV_TICKETTYPE,
			VT_FLI_SV_DESTINATION,
			(SELECT FLIGHTTYPE FROM VT_FLIGHT_FLIGHTTYPE_TICKET WHERE AIRLINE = VT_FLI_SV_AIRLINE) AS flighttype,
			(SELECT ac2.LC || ',' || ac2.airport FROM airport_codes ac2 WHERE vt_fli_sv_departure = ac2.LC AND ROWNUM = 1) AS vt_fli_sv_dep,
			(SELECT ac2.LC || ',' || ac2.airport FROM airport_codes ac2 WHERE vt_fli_sv_destination = ac2.LC AND ROWNUM = 1) AS vt_fli_sv_des,
			VT_FLI_SV_DEPARTURE,
			(SELECT DISTINCT(AIRPORT) FROM AIRPORT_CODES WHERE LC = VT_FLI_SV_DESTINATION) AS VT_FLI_SV_DESTINATION_T,
			(SELECT DISTINCT(AIRPORT) FROM AIRPORT_CODES WHERE LC = VT_FLI_SV_DEPARTURE) AS VT_FLI_SV_DEPARTURE_T,
			TO_CHAR(VT_FLI_SV_BEGINN,'DD.MM.YYYY') AS VT_FLI_SV_BEGINN,
			TO_CHAR(VT_FLI_SV_BEGINN,'YYYY-MM-DD') AS VT_FLI_SB_BEGINN_SORT,
			TO_CHAR(VT_FLI_SV_END,'DD.MM.YYYY') AS VT_FLI_SV_END,
			VT_FLI_SV_BOOKINGCODE,
			VT_FLI_SV_AIRLINE,
			VT_FLI_SV_PROCESSNO,
            VT_FLI_SV_PROVIDER,
            VT_FLI_SV_PROVIDER_2,
			VT_FLI_SV_KEY,
			CONCAT(CONCAT( CASE WHEN VT_FLI_SV_AIRLINE = 'VTO' THEN VT_FLI_SV_FLIGHTCARRIER ELSE VT_FLI_SV_AIRLINE END ,'/'),
			CASE WHEN VT_FLI_SV_AIRLINE != 'VTO'
                                THEN (SELECT AIRLINENAME FROM VT_AIRLINES WHERE trim(UPPER(AIRLINECODE)) = trim(UPPER(VT_FLI_SV_AIRLINE)))
				ELSE (SELECT AIRLINENAME FROM VT_AIRLINES WHERE trim(UPPER(AIRLINECODE)) = trim(UPPER(VT_FLI_SV_FLIGHTCARRIER)))
                        END) AS AIRLINECODEANDNAME,
			(SELECT fli_flightairline FROM vt_flights WHERE fli_bo_id = vt_bo_id AND fli_flightseqment_no = 1) AS airline,
			VT_FLI_SV_PROCESSNO,
			(SELECT DISTINCT(1)
				FROM vt_flights f1
				WHERE f1.fli_bo_id = vt_bo_id
				AND  SUBSTR(f1.FLI_FLIGHTDESTINATION , 0, REGEXP_INSTR(f1.FLI_FLIGHTDESTINATION,'[,. ]') -1)
				IN (SELECT ACA_LC FROM AIRPORT_CODES_AMERIKA)
		    ) AS is_Amerika,
			CASE WHEN VT_FLI_SV_TICKETTYPE = 'TOD'
  			THEN (
                                SELECT
                                CONCAT(CONCAT(CONCAT(CONCAT(RULA_SCHALTERINFO,'\n'),CONCAT(RULA_FLUGHAFEN,'\n')),CONCAT(RULA_STANDORT,'\n')),RULA_TELEFON)
                                FROM RULA_TICKETONDEMAND_INFO
                                WHERE LOWER(RULA_ZIELGEBIET) = LOWER(TRIM(VT_FLI_SV_DEPARTURE))
                                AND LOWER(RULA_AIRLINE) = LOWER(VT_FLI_SV_AIRLINE)
                            )
                            ELSE ''
 			END AS VT_FLI_SV_SCHALTERINFO,
			VT_BO_INTERNALTREATMENTCONDITI,
			VT_FLI_SV_STORNOFLAG
			FROM VT_BOOKINGS,VT_FLIGHT_SAVED
			WHERE VT_FLI_SV_ID = '" . $data->vt_bo_id . "'
			AND VT_BO_ID = VT_FLI_SV_ID
			AND ( (VT_FLI_SV_STORNOFLAG = 0 AND VT_BO_STORNO = 0) OR (VT_BO_STORNO = 1 AND (VT_FLI_SV_STORNOFLAG = 0 OR VT_FLI_SV_STORNOFLAG = 1)))
                        ORDER BY VT_FLI_SB_BEGINN_SORT";


            $res = $this->db->getAll($stmt);
            if ($this->db->errorMsg()) {
                return false;
            } else {
                return $res;
            }
        } else {
            $stmt = " SELECT
			1 as VT_FLI_SV_NUMBER,
			flighttype,
			CASE WHEN TICKETTYPE IS NULL
                                THEN 'NOTIX'
				ELSE TICKETTYPE
			END as VT_FLI_SV_TICKETTYPE,
			p.VT_PT_DESTINATION as VT_FLI_SV_DESTINATION,
			(select ac2.LC || ',' || ac2.airport from airport_codes ac2 where p.vt_pt_departure = ac2.LC and ROWNUM = 1) as vt_fli_sv_dep,
			(select ac2.LC || ',' || ac2.airport from airport_codes ac2 where p.vt_pt_destination = ac2.LC and ROWNUM = 1) as vt_fli_sv_des,
			p.VT_PT_DEPARTURE as VT_FLI_SV_DEPARTURE,
			TO_CHAR(p.VT_PT_BEGINN,'DD.MM.YYYY') as VT_FLI_SV_BEGINN,
			TO_CHAR(p.VT_PT_END,'DD.MM.YYYY') as VT_FLI_SV_END,
			'' as VT_FLI_SV_BOOKINGCODE,
			p.VT_PT_AIRLINECODE as VT_FLI_SV_AIRLINE,
			p.VT_PT_FLIGHTPROZESSNO as VT_FLI_SV_PROCESSNO,
			b.VT_BO_INTERNALTREATMENTCONDITI,
			(select DISTINCT(1)
                from vt_flights f1
                where f1.fli_bo_id = b.vt_bo_id
                and  SUBSTR(f1.FLI_FLIGHTDESTINATION , 0, REGEXP_INSTR(f1.FLI_FLIGHTDESTINATION,'[,. ]') -1)
                IN (SELECT ACA_LC FROM AIRPORT_CODES_AMERIKA)
            ) as is_Amerika
            FROM VT_BOOKINGS b
            LEFT JOIN VT_PACKAGETOURS p ON p.VT_BO_ID = b.VT_BO_ID
            LEFT OUTER JOIN VT_FLIGHT_FLIGHTTYPE_TICKET vf ON UPPER(AIRLINE) = UPPER(p.VT_PT_AIRLINECODE)
            WHERE p.VT_BO_ID = '$data->vt_bo_id'
            AND p.VT_PT_NO = 1";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                return false;
            } else {
                return $res;
            }
        }
    }

    /**
     * @param Booking_Bookingcore $data
     * @return bool
     */
    public function getFlighttimes(Booking_Bookingcore $data)
    {
        $stmt = "SELECT
                    FLI_CABINCLASS,
                    TO_CHAR(FLI_FLIGHTBEGINN,'DD.MM.YYYY') AS FLI_FLIGHTBEGINN,
                    TO_CHAR(FLI_FLIGHTBEGINN,'YYYY-MM-DD') AS FLI_FLIGHTBEGINN_ORDER,
                    FLI_FLIGHTDEPARTURE,
                    FLI_FLIGHTDESTINATION,
                    SUBSTR(FLI_FLIGHTDEPARTURE,0,REGEXP_INSTR(FLI_FLIGHTDEPARTURE,'[,; ]')-1)||' ('||TRIM(SUBSTR(FLI_FLIGHTDEPARTURE,REGEXP_INSTR(FLI_FLIGHTDEPARTURE,'[,; ]')+1))||')' AS FLI_DEP_EXT,
                    SUBSTR(FLI_FLIGHTDESTINATION,0,REGEXP_INSTR(FLI_FLIGHTDESTINATION,'[,; ]')-1)||' ('||TRIM(SUBSTR(FLI_FLIGHTDESTINATION,REGEXP_INSTR(FLI_FLIGHTDESTINATION,'[,; ]')+1))||')' AS FLI_DEST_EXT,
                    TRIM(SUBSTR(FLI_FLIGHTDEPARTURE,REGEXP_INSTR(FLI_FLIGHTDEPARTURE,'[,; ]')+1)) AS FLI_DEP_EXT_I,
                    TRIM(SUBSTR(FLI_FLIGHTDESTINATION,REGEXP_INSTR(FLI_FLIGHTDESTINATION,'[,; ]')+1)) AS FLI_DEST_EXT_I,
                    FLI_FLIGHTSTARTTIME,
                    FLI_FLIGHTLANDINGTIME,
                    CASE WHEN replace(FLI_FLIGHTLANDINGTIME,':','') < replace(FLI_FLIGHTSTARTTIME,':','')
                        THEN '1'
                        ELSE '0'
                    END AS Nachtflug,
                    CASE WHEN REGEXP_LIKE(TO_CHAR(FLI_FLIGHTNUMBER),'[a-zA-Z]')
			  THEN FLI_FLIGHTNUMBER
			  ELSE TRIM(SUBSTR(FLI_FLIGHTAIRLINE,INSTR(FLI_FLIGHTAIRLINE,',')+1) || FLI_FLIGHTNUMBER)
                    END AS FLI_FLIGHTNUMBER,
                    REGEXP_REPLACE(FLI_FLIGHTAIRLINE,'[;,]','-') AS FLI_FLIGHTAIRLINE,
                    (SELECT convert(FLI_FLIGHTTYPE,'us7ascii','utf8') FROM dual) AS FLI_FLIGHTTYPE,
                    CASE
                        WHEN LENGTH(TRIM(SUBSTR(FLI_FLIGHTAIRLINE,REGEXP_INSTR(FLI_FLIGHTAIRLINE,'[,]')+1))) = '2'
                        	THEN
                                (SELECT AIRLINENAME
                                 FROM VT_AIRLINES
                                 WHERE AIRLINECODE = TRIM(SUBSTR(FLI_FLIGHTAIRLINE,REGEXP_INSTR(FLI_FLIGHTAIRLINE,'[,]')+1)))
                                 WHEN LENGTH(TRIM(SUBSTR(FLI_FLIGHTAIRLINE,REGEXP_INSTR(FLI_FLIGHTAIRLINE,'[,]')+1))) = '3'
                             THEN
                                (SELECT AIRLINENAME
                                  FROM VT_AIRLINES
                                  WHERE AIRLINECODE = TRIM(SUBSTR(FLI_FLIGHTAIRLINE,REGEXP_INSTR(FLI_FLIGHTAIRLINE,'[,]')+1)))
                        ELSE FLI_FLIGHTAIRLINE
                    END AS FLI_FLIGHTAIRLINENAME,
                    TRIM(SUBSTR(FLI_FLIGHTAIRLINE,REGEXP_INSTR(FLI_FLIGHTAIRLINE,'[,]')+1)) AS FLI_FLIGHT_AIRLINECODE,
                    fli_flight_gepaeck
                    FROM VT_FLIGHTS
                    WHERE FLI_BO_ID = '" . $data->vt_bo_id . "'
                    AND FLI_STORNOFLAG = 0
                    ORDER BY FLI_FLIGHTBEGINN_ORDER, FLI_FLIGHTSTARTTIME";


        $res = $this->db->getALL($stmt);
        if ($this->db->errorMsg()) {
            return false;
        } else {
            return $res;
        }
    }

    /**
     * @param Booking_Bookingcore $data
     * @return bool
     */
    public function getFlighttimesFromHistory(Booking_Bookingcore $data)
    {

        $stmt = "SELECT
		CASE WHEN TO_CHAR(vfh.VFH_FLIGHTBEGINN,'YYYY-MM-DD') BETWEEN TO_CHAR(vfs.VT_FLI_SV_BEGINN -1,'YYYY-MM-DD') AND TO_CHAR(vfs.VT_FLI_SV_BEGINN +1,'YYYY-MM-DD')
                    THEN TO_CHAR('H')
                    WHEN TO_CHAR(vfh.VFH_FLIGHTBEGINN,'YYYY-MM-DD') BETWEEN TO_CHAR(vfs.VT_FLI_SV_END -1,'YYYY-MM-DD') AND TO_CHAR(vfs.VT_FLI_SV_END+1,'YYYY-MM-DD')
                    THEN TO_CHAR('R')
                    ELSE ''
                END AS VFH_FLIGHTTYPE,
		TO_CHAR(vfh.VFH_FLIGHTBEGINN,'DD.MM.YYYY') AS VFH_FLIGHTBEGINN,
		TO_CHAR(vfh.VFH_FLIGHTBEGINN,'YYYY-MM-DD') AS VFH_FLIGHTBEGINN_ORDER,
  		vfh.VFH_FLIGHTDEPARTURE,
  		vfh.VFH_FLIGHTDESTINATION,
		SUBSTR(vfh.VFH_FLIGHTDEPARTURE,0,REGEXP_INSTR(vfh.VFH_FLIGHTDEPARTURE,'[,; ]')-1)||' ('||TRIM(SUBSTR(vfh.VFH_FLIGHTDEPARTURE,REGEXP_INSTR(vfh.VFH_FLIGHTDEPARTURE,'[,; ]')+1))||')' AS VFH_DEP_EXT,
		SUBSTR(vfh.VFH_FLIGHTDESTINATION,0,REGEXP_INSTR(vfh.VFH_FLIGHTDESTINATION,'[,; ]')-1)||' ('||TRIM(SUBSTR(vfh.VFH_FLIGHTDESTINATION,REGEXP_INSTR(vfh.VFH_FLIGHTDESTINATION,'[,; ]')+1))||')' AS VFH_DEST_EXT,
  		TRIM(SUBSTR(vfh.VFH_FLIGHTDEPARTURE,REGEXP_INSTR(vfh.VFH_FLIGHTDEPARTURE,'[,; ]')+1)) AS VFH_DEP_EXT_I,
		TRIM(SUBSTR(vfh.VFH_FLIGHTDESTINATION,REGEXP_INSTR(vfh.VFH_FLIGHTDESTINATION,'[,; ]')+1)) AS VFH_DEST_EXT_I,
		vfh.VFH_FLIGHTSTARTTIME,
 		vfh.VFH_FLIGHTLANDINGTIME,
		CASE WHEN replace(vfh.VFH_FLIGHTLANDINGTIME,':','') < replace(vfh.VFH_FLIGHTSTARTTIME,':','')
                    THEN '1'
                    ELSE '0'
		END AS Nachtflug,
  		CASE WHEN REGEXP_LIKE(TO_CHAR(vfh.VFH_FLIGHTNUMBER),'[a-zA-Z]')
                    THEN vfh.VFH_FLIGHTNUMBER
                    ELSE TRIM(SUBSTR(vfh.VFH_FLIGHTAIRLINE,INSTR(vfh.VFH_FLIGHTAIRLINE,',')+1) || vfh.VFH_FLIGHTNUMBER)
                END AS VFH_FLIGHTNUMBER,
  		REGEXP_REPLACE(vfh.VFH_FLIGHTAIRLINE,'[;,]','-') AS VFH_FLIGHTAIRLINE,
  		CASE
                    WHEN LENGTH(TRIM(SUBSTR(vfh.VFH_FLIGHTAIRLINE,REGEXP_INSTR(vfh.VFH_FLIGHTAIRLINE,'[,]')+1))) = '2'
                    THEN
                        (SELECT TRIM(AIRLINECODE)||', '||AIRLINENAME
                         FROM VT_AIRLINES
                         WHERE AIRLINECODE = TRIM(SUBSTR(vfh.VFH_FLIGHTAIRLINE,REGEXP_INSTR(vfh.VFH_FLIGHTAIRLINE,'[,]')+1)))
                         WHEN LENGTH(TRIM(SUBSTR(vfh.VFH_FLIGHTAIRLINE,REGEXP_INSTR(vfh.VFH_FLIGHTAIRLINE,'[,]')+1))) = '3'
                    THEN
                        (SELECT TRIM(AIRLINECODE)||','||AIRLINENAME
                         FROM VT_AIRLINES
                         WHERE AIRLINECODE = TRIM(SUBSTR(vfh.VFH_FLIGHTAIRLINE,REGEXP_INSTR(vfh.VFH_FLIGHTAIRLINE,'[,]')+1)))
                    ELSE VFH_FLIGHTAIRLINE
                END AS VFH_FLIGHTAIRLINENAME,
                TRIM(SUBSTR(vfh.VFH_FLIGHTAIRLINE,REGEXP_INSTR(vfh.VFH_FLIGHTAIRLINE,'[,]')+1)) AS VFH_FLIGHT_AIRLINECODE
		FROM VT_FLIGHTS_HISTORY vfh,VT_FLIGHT_SAVED vfs
		WHERE vfh.VFH_BO_ID = '" . $data->vt_bo_id . "'
		AND vfs.VT_FLI_SV_ID = vfh.VFH_BO_ID
		AND vfs.VT_FLI_SV_STORNOFLAG = 0
		AND vfs.VT_FLI_SV_NUMBER = (
                        SELECT MIN(VT_FLI_SV_NUMBER)
                        FROM VT_FLIGHT_SAVED vfs2
                        WHERE vfs2.VT_FLI_SV_ID = vfs.VT_FLI_SV_ID
                        AND vfs2.VT_FLI_SV_STORNOFLAG = 0
                )
                AND vfh.VFH_FLIGHTINSERT  = (SELECT max(vfh1.VFH_FLIGHTINSERT)
		FROM VT_FLIGHTS_HISTORY vfh1
		WHERE vfh1.VFH_BO_ID = vfh.VFH_BO_ID
                AND vfh1.VFH_STORNOFLAG = 0
		AND vfh1.VFH_FLIGHTSEQ = vfh.VFH_FLIGHTSEQ )
		ORDER BY VFH_FLIGHTBEGINN_ORDER, vfh.VFH_FLIGHTSTARTTIME";
        $res = $this->db->getALL($stmt);

        if ($this->db->errorMsg()) {
            return false;
        } else {
            return $res;
        }
    }

    /**
     * @param Booking_Bookingcore $data
     * @return bool
     */
    public function getBookingMasterData(Booking_Bookingcore $data)
    {
        $stmt = "SELECT
                    b.VT_BO_ID,
                    TO_CHAR(b.VT_BO_CREATED,'DD.MM.YYYY') as VT_BO_CREATED,
                    b.VT_BO_INTERNALTREATMENTCONDITI,
                    b.VT_BO_BOOKINGTYPE,
                    b.VT_BO_DESTINATION,
                    CASE WHEN HT_HOTELLOCCODE IS NULL AND VT_PT_HOTELLOCATIONCODE IS NULL
                    THEN 'STD'
                    WHEN HT_HOTELLOCCODE = '' AND VT_PT_HOTELLOCATIONCODE  = ''
                    THEN 'STD'
                    WHEN HT_HOTELLOCCODE IS NOT NULL OR HT_HOTELLOCCODE != ''
                    THEN HT_HOTELLOCCODE
                    WHEN VT_PT_HOTELLOCATIONCODE IS NOT NULL OR VT_PT_HOTELLOCATIONCODE != ''
                    THEN VT_PT_HOTELLOCATIONCODE
                    END as HT_HOTELLOCCODE,
                    CASE WHEN HT_HOTELCODE IS NULL AND VT_PT_PRODUCTCODE IS NULL
                    THEN 'STD'
                    WHEN HT_HOTELCODE = '' AND VT_PT_PRODUCTCODE IS NULL
                    THEN 'STD'
                    WHEN HT_HOTELCODE IS NOT NULL OR HT_HOTELCODE != ''
                    THEN HT_HOTELCODE
                    WHEN VT_PT_PRODUCTCODE IS NOT NULL OR VT_PT_PRODUCTCODE != ''
                    THEN VT_PT_PRODUCTCODE
                    ELSE HT_HOTELCODE END as HT_HOTELCODE,
                    AC.LAND as VT_BO_COUNTRY
                    FROM VT_BOOKINGS b
                    LEFT JOIN VT_HOTELS ON HT_BO_ID = b.VT_BO_ID AND HT_HOTEL_NO = 1
                    LEFT JOIN VT_PACKAGETOURS vp ON vp.VT_BO_ID = b.VT_BO_ID
                    LEFT JOIN AIRPORT_CODES AC ON AC.LC=B.VT_BO_DESTINATION
                    WHERE b.VT_BO_ID = '$data->vt_bo_id'";


        $res = $this->db->getRow($stmt);

        if ($this->db->errorMsg()) {
            return false;
        } else {
            return $res;
        }
    }

    /**
     * @param Booking_Bookingcore $core
     * @param string $client
     * @return array
     */
    public function getAirportInfos(Booking_Bookingcore $core, $client = 'vto')
    {
        $airportcode = $core->vt_bo_destination;

        $airportInfo = array();
        if ($client == 'xfly') {
            $sql = "
			SELECT *   FROM  VT_AIRPORT_INFOS ai,  VT_FLIGHT_SAVED f
			WHERE ai.AIRPORT_CODE=f.VT_FLI_SV_DESTINATION
			AND f.vt_fli_sv_id=" . $core->vt_bo_id . "
			AND f.VT_FLI_SV_STORNOFLAG = 0
			ORDER BY f.VT_FLI_SV_NUMBER DESC";
        } else {
            $sql = "SELECT * FROM VT_AIRPORT_INFOS WHERE AIRPORT_CODE='" . $airportcode . "'";
        }

        $res = $this->db->getAll($sql);
        if ($res) {
            $airportInfo['AI_CODE'] = $res[0]['AIRPORT_CODE'];
            $airportInfo['AI_NAME'] = $res[0]['AIRPORT_NAME'];
            $airportInfo['AI_INFOS'] = $res[0]['INFOS'];
        }
        return $airportInfo;
    }

    /**
     * @param Booking_Bookingcore $obj
     * @return bool|void
     * @throws Exception
     */
    public function getTransferAgencies(Booking_Bookingcore $obj)
    {
        try {
            $transAgency = $obj->getTransferCode();

            if ($transAgency == null || $transAgency == '') {
                return;
            }
            $stmt = "SELECT *
			FROM RULA_TRANSFER_AGENCIES
			WHERE lower(RFI_KUERZEL) = '" . strtolower($transAgency) . "'
			AND ROWNUM = 1";

            $res = $this->db->getAll($stmt);

            if (count($res) > 1) {
                throw new Exception('Kein eindeutiger Datensatz Transfer');
            } elseif (count($res) <= 0) {
                return false;
            }
            return ($res);
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param string $mode
     * @param atring $agency
     * @return string
     */
    private function getStmtForgetHotelIncomingAgencies($mode, $agency)
    {

        return "SELECT *
                    FROM RULA_INCOMING_AGENCIES
                    WHERE lower(RIA_KUERZEL) = '" . strtolower($agency) . "'
                    and ria_type = '$mode'";
    }

    /**
     * @param Booking_Bookingcore $obj
     * @return array
     */
    public function getIncomingAgenciesForZG(Booking_Bookingcore $obj)
    {

        try {
            if ($obj->getTransferCode()) {
                $transfer = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies('transfer',
                    $obj->getTransferCode()));

                if ($this->db->errorMsg()) {
                    throw new Exception($this->db->errorMsg());
                }
                if (count($transfer) > 1) {
                    throw new Exception('Zuviele Eintraege erhalten fuer die ZG Transfer');
                }
            }

            if ($obj->getIncomingAgentur()) {
                $hotel = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies('incoming',
                    $obj->getIncomingAgentur()));

                if ($this->db->errorMsg()) {
                    throw new Exception($this->db->errorMsg());
                }
                if (count($transfer) > 1) {
                    throw new Exception('Zuviele Eintraege erhalten fuer die ZG Hotel');
                }
            }

            $newObj = array();

            if (is_array($transfer[0]) && count($transfer) == 1) {
                foreach ($transfer[0] as $key => $value) {
                    if (!is_int($key)) {
                        $temp = strtolower($key);
                        if ($temp == 'ria_e_mail') {
                            $newObj['transfer'][$temp] = str_replace(' ', '', $value);
                        } else {
                            $newObj['transfer'][$temp] = $value;
                        }
                    }
                }
            }

            if (is_array($hotel[0]) && count($hotel) == 1) {
                foreach ($hotel[0] as $key => $value) {
                    if (!is_int($key)) {
                        $temp = strtolower($key);
                        if ($temp == 'ria_e_mail') {
                            $newObj['incoming'][$temp] = str_replace(' ', '', $value);
                        } else {
                            $newObj['incoming'][$temp] = $value;
                        }
                    }
                }
            }
        } catch (Exception $e) {
            $this->log->error('Fehler beim holen der ZG Daten::'.$e->getMessage());
        }

        return $newObj;
    }

    /**
     * @param Booking_Bookingcore $obj
     * @return array|bool
     * @throws Exception
     */
    public function getHotelIncomingAgencies(Booking_Bookingcore $obj)
    {

        try {
            $data = array();

            $mode = null;
            $agency = null;

            $ausnahme = $this->isZgAnException($obj->getZielgebiet());

            if (in_array(strtoupper($obj->vt_bo_bookingtype), array('X', 'Y', 'C'))
                && $obj->getTransferCode() != $obj->getIncomingAgentur()
                && $obj->getTransferCode() != null && !$ausnahme
            ) {
                $agency = $obj->getTransferCode();
                $mode = 'transfer';
            } else {
                $agency = $obj->getIncomingAgentur();
                $mode = 'incoming';
            }

            $res = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies($mode, $agency));

            if ($mode == 'incoming' && empty($res)) {
                $mode = 'transfer';
                $agency = $obj->getTransferCode();
                $res = null;
                $res = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies($mode, $agency));
            } elseif ($mode == 'transfer' && empty($res)) {
                $mode = 'incoming';
                $agency = $obj->getIncomingAgentur();
                $res = null;
                $res = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies($mode, $agency));
            }

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage. MSG::' . $this->db->errorMsg());
            }
            if (count($res) > 1) {
                throw new Exception('Kein eindeutiger Datensatz IncAgency');
            } elseif (empty($res)) {
                return true;
            }

            foreach ($res[0] as $key => $value) {
                $temp = strtolower($key);
                $data[0][$temp] = $value;
            }

            return $data;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param string $vt_bo_agencyno
     * @return string
     */
    public function getLogopoolImage($vt_bo_agencyno)
    {
        $pic = '';

        $sql = sprintf("SELECT vt_ag_kettenzugehoerigkeit FROM vt_agencies WHERE vt_ag_agenturnr = '%d'",
            $vt_bo_agencyno);
        $res = $this->db->getOne($sql);

        if ($res == null || $res == '' || substr($res, 0, 1) == '.') {
            $res = $vt_bo_agencyno;
        }

        if ($res && $res != 'RSB') {
            $name = '';
            if (!class_exists('DB', false)) {
                $name = $this->ini->logopoolUrl . str_replace(" ", "_", $res) . '.jpg';
            } else {
                $name = $this->ini->VTOURS_WS_URL . $this->ini->VTOURS_WS_LOGOPATH . '.jpg';
            }
            if (is_callable($name)) {
                $name = $this->ini->rula2InvoiceLogo;
                $pic = '<img alt="" src="' . $name . '" />';
            } else {
                $pic = '<img alt="" src="' . $this->logo . '" />';
            }
        } else {
            $pic = '<img alt="" src="' . $this->logo . '" />';
        }

        return $pic;
    }

    /**
     * @param int $vt_bo_id
     * @return bool
     */
    public function getCountRoom($vt_bo_id)
    {
        $sql = "SELECT COUNT(HT_ALLOCATION) as cnt FROM VT_HOTELS WHERE HT_BO_ID = $vt_bo_id";
        if (!$res = $this->db->getOne($sql)) {
            return false;
        }
        return $res;
    }

    /**
     * @param array $res
     * @param int $mode
     * @param bool $zusatz
     * @return array
     */
    public function cleanArray($res, $mode = MB_CASE_TITLE, $zusatz = false)
    {
        $data = array();

        if (is_array($res) && !$zusatz) {
            foreach ($res as $row => $array) {
                if (is_array($array)) {
                    foreach ($array as $key => $value) {
                        if (!is_numeric($key)) {
                            if ($key == 'VT_VMIX_TXT' || $key == 'VT_BO_BA_AGNAME') {
                                $data[$row][$key] = $value;
                            } elseif (in_array(strtoupper($key), $this->convertToLower)) {
                                $data[$row][$key] = mb_convert_case($value, MB_CASE_LOWER, 'UTF-8');
                            } else {
                                $data[$row][$key] = mb_convert_case($value, $mode, 'UTF-8');
                            }
                        }
                    }
                }
            }
        } else {
            foreach ($res as $key => $value) {
                if (!is_numeric($key)) {
                    if ($key == 'VT_VMIX_TXT' || $key == 'VT_BO_BA_AGNAME') {
                        $data[$key] = $value;
                    } elseif (in_array(strtoupper(trim($key)), $this->convertToLower)) {
                        $data[$key] = mb_convert_case($value, MB_CASE_LOWER, 'UTF-8');
                    } else {
                        $data[$key] = mb_convert_case($value, $mode, 'UTF-8');
                    }
                }
            }
        }

        return $data;
    }

    /**
     * @param string $zg
     * @return bool
     * @throws Exception
     */
    public function isZgAnException($zg)
    {
        $res = null;

        if (empty($zg)) {
            return false;
        }
        try {
            $stmt = "select 1
			from airport_codes
			where land in ('Thailand')
			and lc = '$zg'";

            $res = $this->db->getOne($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen von Ausnahmen. MSG::' . $this->db->errorMsg());
            }

            return !empty($res);

        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                Zend_Registry::get('log')->error($e->getMessage());
            } else {
                throw new Exception($e->getMessage());
            }
        }
    }

    /**
     * @param int $vt_bo_id
     * @return bool|stdClass|void
     * @throws Exception
     */
    public function getBankData($vt_bo_id)
    {

        if ($this->client != 'vto' && substr($this->client, 0, 4) != 'vtoi') {
            return;
        }
        $std = new stdClass();

        // muss ich leider so machen , weil der Salt probleme macht in der config.main.xml (NB)
        if (class_exists('DB', false)) {
            $salt_iban = \Config::stepIn()->cfg('SALT_IBAN');
            $salt_bic = \Config::stepIn()->cfg('SALT_BIC');

            if (is_dir('/data1/www/')) {
                include_once('/data1/www/vtours-accounting/library/Toolbox/Decrypt/Handle.php');
                include_once('/data1/WWW/vtours-accounting/library/Toolbox/Encrypt/Handle.php');
            } else {
                include_once('/data1/vtours-accounting/library/Toolbox/Decrypt/Handle.php');
                include_once('/data1/vtours-accounting/library/Toolbox/Encrypt/Handle.php');
            }
        } else {
            $salt_iban = $this->ini->bankdata->salt_iban;
            $salt_bic = $this->ini->bankdata->salt_bic;
        }

        $classification = BookingClassificationService::classifyByBookingNumber($vt_bo_id);

        if (!empty($classification->getClient())) {
            try {
                $stmt = "select iban,bic
						 from vt_eft_temp where vt_bo_id = $vt_bo_id ";

                $res = $this->db->getAll($stmt);

                if ($this->db->errorMsg()) {
                    throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__);
                }

                if ($res[0]['IBAN'] == null || $res[0]['IBAN'] == '' || $res[0]['BIC'] == null || $res[0]['BIC'] == '') {
                    if (!class_exists('DB', false)) {
                        throw new Exception('Keine Bankdaten vorhanden fuer Vorgang ' . $vt_bo_id);
                    }
                }


                $tempObj = new Toolbox_Decrypt_Handle();

                $tempObj->setSalt($salt_iban);
                $tempiban = $tempObj->decryptIbanOrBicFromEft($res[0]['IBAN']);

                $s = 0;
                for ($i = 0; $i < strlen($tempiban); $i++) {
                    if ($s < 4) {
                        $std->iban .= $tempiban[$i];
                        $s++;
                        continue;
                    } else {
                        $std->iban .= " " . $tempiban[$i];
                        $s = 1;
                        continue;
                    }
                }

                $tempObj->setSalt($salt_bic);
                $std->bic = $tempObj->decryptIbanOrBicFromEft($res[0]['BIC']);
            } catch (Exception $e) {
                if (!class_exists('DB', false)) {
                    Zend_Registry::get('log')->error($e->getMessage());
                } else {
                    throw new Exception($e->getMessage());
                }
                return false;
            }
        }

        return $std;
    }

    /**
     * @param array $data
     * @return stdClass
     */
    public function makeHotelDataForInvoice($data)
    {
        $temp = array();

        foreach ($data as $number => $values) {
            $checkStr = str_replace(' ', '',
                strtoupper($values['HT_MEALTYPE']) .
                strtoupper($values['HT_ROOMTYPE']) .
                strtoupper($values['HT_ROOMACCOMODATION']));

            if (!$temp[$checkStr] instanceof stdClass) {
                $temp[$checkStr] = new stdClass();
                $temp[$checkStr]->valueFromNumber = $number;
                $temp[$checkStr]->countValue = 1;
            } else {
                $temp[$checkStr]->countValue++;
            }
        }

        $t = 0;
        $obj = array();

        foreach ($temp as $value) {
            $obj[$t] = new stdClass();
            $obj[$t]->HT_MEALTYPE = $data[$value->valueFromNumber]['HT_MEALTYPE'];
            $obj[$t]->HT_ROOMTYPE = $data[$value->valueFromNumber]['HT_ROOMTYPE'];
            $obj[$t]->HT_ROOMACCOMODATION = $data[$value->valueFromNumber]['HT_ROOMACCOMODATION'];
            $obj[$t]->HT_ROOMVIEW = $data[$value->valueFromNumber]['HT_ROOMVIEW'];
            $obj[$t]->HT_ASSINGLEROOM = $data[$value->valueFromNumber]['HT_ASSINGLEROOM'];

            $obj[$t]->countValue = $value->countValue;
            $t++;
        }

        return $obj;
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function getHistoryData(Booking_Bookingcore $data)
    {

        try {
            $stmt = "select
                        to_char(vh.HST_INSERT_TIMESTAMP,'dd.mm.YYYY') as HST_INSERTTIMESTAMP
			from vt_history vh
                        where vh.hst_bo_id = $data->vt_bo_id
			and (vh.HST_INSERT_TIMESTAMP IS NULL OR vh.HST_INSERT_TIMESTAMP = (
				SELECT MAX(vh2.HST_INSERT_TIMESTAMP)
				FROM VT_HISTORY vh2
                            WHERE vh2.HST_BO_ID = vh.HST_BO_ID )
                        )";
            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im '
                    . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			HistoryDaten nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function getReclamationData(Booking_Bookingcore $data)
    {

        try {
            $stmt = "select
			vtr.vtr_email_send,
			vtr.vtr_state,
			vtr.vtr_email,
			vtr.vtr_confirm,
			vtr.vtr_insert_by_name,
			vtr.vtr_insert_by_fname,
			to_char(vtr.vtr_eingang_date,'dd.mm.YYYY') as vtr_eingang_date
			from vt_reclamation vtr
                        where vtr.vtr_bo_id = $data->vt_bo_id ";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }
            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			ReclamationsDaten nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     * @throws Exception
     */
    public function getBoenitaetsData(Booking_Bookingcore $data)
    {

        try {

            $stmt = "select
                      score_ampel_id,
                      score_wert,
                      score_ampel_value,
                      vt_edit,
                      to_char(vt_edit_date,'YYYY-MM-DD') as vt_edit_date,
                      boniver_checked,
                      to_char(boniver_checkdate,'YYYY-MM-DD') as boniver_checkdate,
                      vt_vmix_txt
                     from vt_bonitatspruefung vp
                     where vp.vt_bo_id = $data->vt_bo_id ";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Error in ' . __METHOD__ . ' ERR :: ' . $this->db->errorMsg());
            }

            if (count($res) > 0) {
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }

        } catch (Exception $e) {
            if (!class_exists(DB)) {
                $this->log->err($e->getMessage());
            }
            throw $e;
        }

    }

    /**
     * @param array $data
     * @throws Exception
     */
    public function getZahlungsImportData($data)
    {
        try {

            $stmt = "select
                      zi_mailsend,
                      betrag,
                      inserttime
                     from zahlungsimport_item zi
                     where zi.vt_bo_id = $data->vt_bo_id ";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Error in ' . __METHOD__ . ' ERR :: ' . $this->db->errorMsg());
            }

            if (count($res) > 0) {
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }

        } catch (Exception $e) {
            if (!class_exists(DB)) {
                $this->log->err($e->getMessage());
            }
            throw $e;
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */

    public function getReclamationCouponData(Booking_Bookingcore $data)
    {

        try {
            $stmt = "select
			          vrcp.vrcp_id,
			          vrcp.vrcp_hash,
			          vrcp.vrcp_exported,
			          vrcp.vrcp_no,
			          vrcp.vrcp_amount,
			          vrcp.vrcp_send_via_mail
                    from vt_reclamation_coupon vrcp where vrcp.vrcp_bo_id = $data->vt_bo_id
                    and vrcp.vrcp_no = (select max(vrcp1.vrcp_no)
                    from vt_reclamation_coupon vrcp1
                    where vrcp1.vrcp_bo_id = vrcp.vrcp_bo_id)";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			ReclamationsCouponDaten nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    if ($temp == 'vrcp_amount') {
                        $data->$temp = number_format($value, 2, ',', '.');
                    } else {
                        $data->$temp = $value;
                    }
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function vtFirstFlightsData(Booking_Bookingcore $data)
    {
        try {
            $stmt = "select "
                . "trim(substr(fli_flightairline,INSTR(fli_flightairline,',')+1))"
                . "from vt_flights "
                . "where fli_bo_id = $data->vt_bo_id "
                . "and fli_flightseqment_no = 1 ";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler bei der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			vtFirstFlightsData nachtraeglich geholt.');
                }

                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }

            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function getReclamationRefundData(Booking_Bookingcore $data)
    {
        try {
            $stmt = "select
			vrr.vrr_recipient_fname,
			vrr.vrr_recipient_name
			from vt_reclamation_refund vrr where vrr.vrr_bo_id = $data->vt_bo_id
			and vrr.vrr_no = (select max(vrcp1.vrcp_no)
                    from vt_reclamation_coupon vrcp1
                    where vrcp1.vrcp_bo_id = vrr.vrr_bo_id)";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			ReclamationsRefundDaten nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function getBookingParameter(Booking_Bookingcore $data)
    {
        try {
            $stmt = "select
			  vbp.vbp_prepayment_mail,
			  vbp.vbp_zahlschein_az,
			  vbp.vbp_zahlschein_rz,
			  vbp.vbp_vfly_customer,
			  vbp.vbp_inform_flight_time_change,
			  vbp.vbp_inform_flight_time_ch_date
            from vt_bookings_parameter vbp
            where vbp.vbp_bo_id = $data->vt_bo_id";


            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			BookingsParameter nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function vftInvalidData(Booking_Bookingcore $data)
    {
        try {
            $stmt = "select
			to_char(vei.created,'YYYY-MM-DD') as created,
			vei.mail_send
			from vt_eft_invalid vei where vei.vt_bo_id = $data->vt_bo_id
			and (vei.CREATED IS NULL OR vei.CREATED = (
                            SELECT (MAX(vei2.CREATED))
                            FROM VT_EFT_INVALID vei2
                            WHERE vei2.VT_BO_ID = vei.VT_BO_ID ))";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			vftInvalidDaten nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function vtcGuWidData(Booking_Bookingcore $data)
    {
        try {
            $stmt = "select
			vcc.cc_amount,
			vcc.cc_booking_type,
			vcc.cc_process,
			vcc.info_mail_send
			from vt_cc_guwid vcc where vcc.vt_bo_id = $data->vt_bo_id
			and (vcc.INSERT_TIME IS NULL OR vcc.INSERT_TIME = (
				SELECT (MAX(vcc2.INSERT_TIME))
					FROM VT_CC_GUWID vcc2 WHERE vcc2.VT_BO_ID = vcc.VT_BO_ID
			))";

            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			vtcGuWidDaten nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $data
     */
    public function vtDunnings(Booking_Bookingcore $data)
    {
        try {
            $stmt = "select
			vdd.dun_id
			from vt_dunnings vdd where vdd.vt_bo_id = $data->vt_bo_id
			and (vdd.dun_timestamp IS NULL OR vdd.dun_timestamp = (
				SELECT (MAX(vdd1.dun_timestamp))
					FROM vt_dunnings vdd1 WHERE vdd1.VT_BO_ID = vdd.VT_BO_ID
			))";
            $res = $this->db->getRow($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler in der DB Abfrage im ' . __METHOD__ . ' MSG::' . $this->db->errorMsg());
            }

            if (count($res) > 0 && is_array($res)) {
                if (!class_exists('DB', false)) {
                    $this->log->info('			vtdunnings nachtraeglich geholt.');
                }
                foreach ($res as $key => $value) {
                    $temp = strtolower($key);
                    $data->$temp = $value;
                }
            }
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }

    /**
     * @param Booking_Bookingcore $coreData
     * @return array|void
     * @throws Exception
     */
    public function getUltraTransferData($coreData)
    {
        $stmt = "SELECT vt_booking_transfer.*,
                  to_char(TRANSFER_THERE_DATE,'DD.MM.YYYY') AS TRANSFER_THERE_DATE,
                  to_char(TRANSFER_THERE_TIME ,'HH24:MI') AS TRANSFER_THERE_TIME,
                  to_char(TRANSFER_THERE_DATE,'YYYY-MM-DD') AS TRANSFER_THERE_DATE_SORT
                 FROM vt_booking_transfer 
                 WHERE booking_id =:booking_id
                 AND transfer_deleted = 0 
                 ORDER BY TRANSFER_THERE_DATE_SORT ASC";

        $res = $this->db->getAll($stmt, array('booking_id' => $coreData->vt_bo_id));

        if ($this->db->errorMsg()) {
            throw new Exception('Fehler bim holen der Daten im '
                . __METHOD__ . ' SQLERR::' . $this->db->errorMsg());
        }

        if (empty($res)) {
            return;
        }

        $temp = array();
        foreach ($res as $key => $value) {

            foreach ($value as $keyT => $data) {
                if (!preg_match('/^[0-9]+$/', $keyT)) {
                    $temp[$key][$keyT] = $data;
                }
                if ($keyT == 'AGENCY_ID') {
                    $resInc = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies('transfer', $data));

                    if (!$resInc) {
                        $resInc = $this->db->getAll($this->getStmtForgetHotelIncomingAgencies('incoming', $data));
                    }

                    if (is_array($resInc) && !empty($resInc)) {
                        foreach ($resInc[0] as $keyInc => $valueInc) {
                            if (!preg_match('/^[0-9]+$/', $keyInc)) {
                                $temp[$key][$keyInc] = $valueInc;
                            }
                        }
                    }
                }
            }

            $extra = $this->getExtraData($res[$key]['TRANSFER_ID']);
            $luggage = $this->getLuggageData($res[$key]['TRANSFER_ID']);

            if (count($extra) > 0) {
                $temp[$key]['extra'] = $extra;
            }
            if (count($luggage) > 0) {
                $temp[$key]['luggage'] = $luggage;
            }
        }
        return $temp;
    }

    /**
     * @param string $transferId
     * @throws Exception
     */
    private function getExtraData($transferId)
    {
        if (!$transferId) {
            return;
        }
        $stmt = "SELECT * 
                  FROM vt_booking_extras 
                  WHERE transfer_id =:transferid 
                  AND extras_deleted = 0";

        $res = $this->db->getAll($stmt, array('transferid' => $transferId));

        if ($this->db->errorMsg()) {
            throw new Exception('Fehler bei Abfrage im ' . __METHOD__ . ' SQLERR:: ' . $this->db->errorMsg());
        }

        return $res;
    }

    /**
     * @param string $transferId
     * @throws Exception
     */
    private function getLuggageData($transferId)
    {
        if (!$transferId) {
            return;
        }
        $stmt = "SELECT * 
                  FROM vt_booking_luggage 
                  WHERE transfer_id =:transferid
                  AND luggage_deleted = 0";
        $res = $this->db->getAll($stmt, array('transferid' => $transferId));

        if ($this->db->errorMsg()) {
            throw new Exception('Fehler bei Abfrage im ' . __METHOD__ . ' SQLERR:: ' . $this->db->errorMsg());
        }
        return $res;
    }

    /**
     * @param int $id
     * @return array|false
     */
    public  function getPayCodeLinks($id)
    {
        $classification = BookingClassificationService::classifyByBookingNumber($id);

        try {
            if (empty($classification->getClient())) {
                $this->log->err('ID nicht erhalten im ' . __METHOD__);
            }

            $stmt = "select payment_type, paycode_url
                 from vt_payment_paycode 
                 where bookings_id = $id";
            $res = $this->db->getAll($stmt);

            if ($res && count($res) > 0) {
                foreach ($res as $value) {
                    if ($value['PAYMENT_TYPE'] == 'PREPAYMENT') {
                        $return['AZ'] = $value['PAYCODE_URL'];
                    } elseif ($value['PAYMENT_TYPE'] == 'BALANCEPAYMENT') {
                        $return['RZ'] = $value['PAYCODE_URL'];
                    } elseif ($value['PAYMENT_TYPE'] == 'CANCELLATION') {
                        $return['ST'] = $value['PAYCODE_URL'];
                    }
                }
            } else {
                $stmt = "select 
                  Zahlungart, 
                  Paycode_url
                 from vt_paycode 
                 where vt_bo_id = $id
                 and aktiv = 1 ";
                $res = $this->db->getAll($stmt);
                if ($res && count($res) > 0) {
                    foreach ($res as $value) {
                        if ($value['ZAHLUNGART'] == 'AZ') {
                            $return['AZ'] = $value['PAYCODE_URL'];
                        } elseif ($value['ZAHLUNGART'] == 'RZ') {
                            $return['RZ'] = $value['PAYCODE_URL'];
                        } elseif ($value['ZAHLUNGART'] == 'ST') {
                            $return['ST'] = $value['PAYCODE_URL'];
                        }
                    }
                } else {
                    return false;
                }
            }

            return $return;
        } catch (Exception $e) {
            $this->log->err($e->getMessage());
        }
    }

    /**
     * @param string $flight_processno
     * @return string
     */
    public function getRyanairEmail($flight_processno)
    {
        $ryanair_email = '';
        $sql = 'SELECT ryanair_email FROM ryanair_emails WHERE flight_processno = :flight_processno';
        try {
            $ryanair_email = $this->db->GetOne($sql, array('flight_processno' => $flight_processno));
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
        return $ryanair_email;
    }

    /**
     * @param string $flight_processno
     * @param string $ryanair_email
     */
    public function persistRyanairEmail($flight_processno, $ryanair_email)
    {
        $sql = 'INSERT INTO ryanair_emails (flight_processno, ryanair_email) VALUES (:flight_processno, :ryanair_email)';
        try {
            $this->db->Execute($sql, array('flight_processno' => $flight_processno, 'ryanair_email' => $ryanair_email));
        } catch (Exception $e) {
            if (!class_exists('DB', false)) {
                $this->log->err($e->getMessage());
            }
            echo $e->getMessage();
        }
    }
}
