<?php

namespace Vtours\Rula2Engine;

/**
 * *
 *  LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 *  and is private by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license
 *  is a violation of the license agreement and will be prosecuted by
 *  civil and criminal law.
 *
 * @copyright 2019 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 *
 */

use Exception;
use Lifestyle\BookingClassification\Service\BookingClassificationService;
use Vtours\Rula2Engine\Booking\BookingCore;
use Vtours\Rula2Engine\Booking\FullBooking;
use Vtours\Rula2Engine\Booking\Pricecorrections;
use Vtours\Rula2Engine\Booking\BookingSavedFlight;
use Vtours\Rula2Engine\Booking\BookingFlightSegment;
use Smarty;

class Template extends Rula2Std
{
    const PRICE_CORRECTION_WATER_PLANE = 'WASSERFLUGZEUG';

    const MYTHOS_MIGRATION_VFLY_DATE = '13.11.2021';

    /**
     * @var string
     */
    protected $table;

    /**
     * @var string
     */
    protected $table2;

    /**
     * @var string
     */
    protected $pk;

    /**
     * @var string
     */
    protected $pk2;

    /**
     * @var string
     */
    protected $smartyTemplate = '';

    /**
     * @var bool
     */
    protected $renderCSS = true;

    /**
     * @var null
     */
    protected $glossar = null;

    /**
     * @var string[]
     */
    protected $searchvars = array('rt_name', 'rt_titel', 'rt_dokutyp');

    /**
     * @var ActionsItem
     */
    protected $actItem;

    /**
     * @var int
     */
    public $zaehler = 0;

    /**
     * @var string
     */
    private $rendered = null;

    /**
     * @var bool
     */
    public $aLotOfHoteloucher = false;

    /**
     * @var string[]
     */
    private $dateMonthGerman = [
        '01' => 'Januar',
        '02' => 'Februar',
        '03' => 'März',
        '04' => 'April',
        '05' => 'Mai',
        '06' => 'Juni',
        '07' => 'Juli',
        '08' => 'August',
        '09' => 'September',
        '10' => 'Oktober',
        '11' => 'November',
        '12' => 'Dezember'
    ];

    /**
     * Template constructor.
     * @throws Exception
     */
    public function __construct()
    {
        $this->table = "RULA2_TEMPLATE";
        $this->table2 = "RULA2_PROZESS_TO_TEMPLATE";
        $this->pk = "RT_ID";
        $this->pk2 = "T_ID";
        parent::init();
    }

    /**
     * @param int $cms
     * @param int $sort
     * @return array
     * @throws Exception
     */
    public function getAllTemplatesForList($cms = 1, $sort = 0)
    {
        try {

            if ($cms == 1) {
                $sql = $this->getWhereForSql($sort);

                $anzPages = $this->countElements($sql);

                if ($sort->actPage == 0 || $sort->actPage == 1 || $sort->actPage == null || $sort == null || $sort == '' || $sort == 0) {
                    $von = 1;
                    $bis = $von + self::$anzElemente - 1;
                } else {
                    $bis = ($sort->actPage) * self::$anzElemente;
                    $von = $bis - self::$anzElemente + 1;
                }

                $stmt = " select * from ( select $this->table.*,RANK() OVER (ORDER BY $sort->field||RT_ID $sort->direction) as element 
						  from $this->table " . $sql . " )
						  where element between $von AND $bis";

            } else {
                $stmt = " select * from $this->table order by rt_name";
            }

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen der Templates fuer die Liste');
            }
            foreach ($res as $key => $value) {
                $res[$key] = new TemplateItem($value);
            }
            $res['anz'] = $anzPages;

            return $res;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param int $pid
     * @return array
     * @throws Exception
     */
    public function getTemplatesForProcess($pid)
    {
        $obj = array();
        try {

            if ($pid == null || $pid == '' || $pid <= 0) {
                throw new Exception('Keine Prozessnummer vorhanden um Prozess abzuholen');
            }

            $stmt = "select * from $this->table i,$this->table2 j
					where i.$this->pk = j.$this->pk2
					and j.P_ID = " . $pid;

            $res = $this->db->getAll($stmt);

            foreach ($res as $value) {
                $temp = new TemplateItem($value);
                $obj[$temp->rt_id] = $temp;
            }
            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen der Templates zu einer P_ID MSG::' . $this->db->errorMsg());
            }
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }

        return $obj;
    }

    /**
     * @param int $tid
     * @return TemplateItem
     * @throws Exception
     */
    public function getTemplateById($tid)
    {
        if ($tid == null || $tid == '' || $tid <= 0) {
            throw new Exception('Keine Templateid vorhanden um Template abzuholen');
        }

        $stmt = "select * from $this->table where $this->pk = " . $tid;

        $res = $this->db->getAll($stmt);

        $res = new TemplateItem($res[0]);

        if ($this->db->errorMsg()) {
            throw new Exception('Fehler in DB Abfrage um Template zuholen. MSG::' . $this->db->errorMsg());
        }

        return $res;
    }

    /**
     * @param array $data
     * @param int $tid
     * @return bool
     * @throws Exception
     */
    public function updateTemplate($data, $tid)
    {
        if (!isset($data['rt_dontset_border'])) {
            $data['rt_dontset_border'] = 0;
        }

        if ($tid == null || $tid == '' || $tid <= 0 || $data == null || $data == '') {
            throw new Exception('Keine Templateid/Daten vorhanden um Template upzudaten');
        }

        $html = $data['RT_HTML'];
        unset($data['RT_HTML']);

        $this->updateBlob($html, $tid);

        $stat = $this->db->autoexecute($this->table, $data, 'UPDATE', 'RT_ID = ' . $tid);

        if ($this->db->errorMsg() || !$stat) {
            throw new Exception('Fehler beim Update der Template Daten. MSG:: ' . $this->db->errorMsg());
        }

        return true;
    }

    /**
     * @param array $data
     * @return bool
     * @throws Exception
     */
    public function insertTemplate($data)
    {
        if ($data == null || $data == '') {
            throw new Exception('Keine Daten vorhanden um Template einzufuegen');
        }

        $html = $data['RT_HTML'];
        $html = str_replace('\"', '', $html);
        unset($data['RT_HTML']);

        $stat = $this->db->autoExecute($this->table, $data);
        if (!$stat || $this->db->errorMsg()) {
            throw new Exception('Fehler beim Insert der Template Dateien.MSG::' . $this->db->errorMsg());
        }

        $id = $this->db->getOne("SELECT MAX($this->pk) from $this->table");

        if ($id > 0 || $this->db->errorMsg() == null) {
            $this->updateBlob($html, $id);
        } else {
            throw new Exception('Fehler beim holen der letzten ID im Templates' . $this->db->errorMsg());
        }

        return true;
    }

    /**
     * @param int $tid
     * @return bool
     * @throws Exception
     */
    public function deleteTemplate($tid)
    {
        if ($tid == null || $tid == '' || $tid <= 0) {
            throw new Exception('Keine Templateid vorhanden um Template zuloeschen');
        }

        $stmt = "DELETE FROM $this->table where $this->pk = " . $tid;

        $this->db->execute($stmt);

        if ($this->db->errorMsg()) {
            throw new Exception('Fehler beim Template loechen.  MSG ' . $this->db->errorMsg());
        }

        return true;
    }

    /**
     * @param int $tid
     * @param int $pid
     * @return bool
     * @throws Exception
     */
    public function addTemplatesToProzess($tid, $pid)
    {
        try {
            $stat = false;

            if ($tid == null || $tid == '' || $tid <= 0) {
                throw new Exception('Keine Templateid vorhanden um Template zum Prozess hinzufuegen');
            }

            $this->db->BeginTrans();

            if (!is_array($tid)) {

                $data['T_ID'] = $tid;
                $data['P_ID'] = $pid;

                if ($this->deleteTemplatesFromProcess($pid)) {

                    $stat = $this->db->autoExecute($this->table2, $data, 'INSERT');
                }

                if (!$stat || $this->db->errorMsg()) {
                    $this->db->RollBackTrans();
                    throw new Exception('Fehler beim erzeugen der Mapping Tabelle. MSG_ID :: ' . $this->db->errorMsg());
                }
            } else {

                if ($this->deleteTemplatesFromProcess($pid)) {
                    foreach ($tid as $value) {
                        if ($value > 0) {
                            $data['T_ID'] = $value;
                            $data['P_ID'] = $pid;

                            $stat = $this->db->autoExecute($this->table2, $data, 'INSERT');

                            if (!$stat || $this->db->errorMsg()) {
                                $this->db->RollBackTrans();
                                throw new Exception('Fehler beim erzeugen der Mapping Tabelle. MSG_ID :: ' . $this->db->errorMsg());
                            }
                        }
                    }
                }
            }
        } catch (Exception $e) {
            $this->db->RollBackTrans();
            throw new Exception($e->getMessage());
        }

        $this->db->commitTrans(true);

        return true;
    }

    /**
     * @param int $pid
     * @return bool
     * @throws Exception
     */
    public function deleteTemplatesFromProcess($pid)
    {
        if ($pid == '' || $pid == null || $pid <= 0) {
            throw new Exception('Fehler beim loeschen er Mapping Tabelle.Keine Prozessnummer');
        }

        $stmt = "DELETE FROM $this->table2 where P_ID = " . $pid;

        $this->db->execute($stmt);

        if ($this->db->errorMsg()) {
            throw new Exception('Fehler beim loeschen der Mapping Tabelle.MSG::' . $this->db->errorMsg());
        }

        return true;
    }

    /**
     * @param string $html
     * @param int $id
     * @throws Exception
     */
    private function updateBlob($html, $id)
    {
        try {
            $this->db->execute("INSERT INTO $this->table (RT_HTML) VALUES (empty_blob())");
            $this->db->updateBlob($this->table, "RT_HTML", $html, "$this->pk = " . $id);
        } catch (Exception $e) {
            throw new Exception('Fehler beim Updaten der RT_HTML Spalte' . $this->db->errorMsg());
        }
    }

    /**
     * @param bool $display
     * @return bool|string
     */
    public function tinyCss($display = false)
    {
        return file_get_contents(dirname(__FILE__) . "/Booking/Templates/tiny.css");
    }

    /**
     * @param string $param
     * @param bool string $withCSS
     * @param null string $from
     */
    public function setSmartyTemplate($param, $withCSS = true, $from = null)
    {

        try {

            if ($from == 'hotelvoucher' || $from == 'var' || $from == 'flugvoucher') {
                $t = $this->getTemplateById((int)$param);
                $this->smartyTemplate = trim($t->rt_html);
                return;
            }

            $this->renderCSS = $withCSS ? true : false;

            $header = "";
            $footer = "";
            $headerGes = "";
            $footerGes = "";

            if ((int)$this->actItem->rt_header > 0) {
                $header = $this->getHeadersTemplatesById($this->actItem->rt_header)->rt_html;
            }
            if ((int)$this->actItem->rt_footer > 0) {
                $footer = $this->getFootersTemplatesById($this->actItem->rt_footer)->rt_html;
            }

            if (is_numeric($param)) {
                $t = $this->getTemplateById((int)$param);

                if (!stristr($t->rt_html, '<html>')) {
                    $headerGes = $this->getHtmlHead($this->actItem->rt_dontset_border) . $header;
                    $footerGes = $footer . $this->getHtmlFooter();

                    /**
                     * raender setzen jetzt noch fix. Koennte man in Zukunft dynamischer machen.
                     * Eingabe des Abstands ueber GUI Nur fuer Dokumente die sie selber erstellen (Faengt immer mit <p>
                     * oder <div> tag an ohne style Angaben)
                     * Dokumente von Life-style sind in den Anfangstags style Angaben
                     * Keine schoene Loesung , aber das haette man bei der Entwicklung beruechsichtigen muessen
                     */

                    if ((substr($t->rt_html, 0, 3) == '<p>'
                            || substr($t->rt_html, 0, 5) == '<div>')
                        && trim($t->rt_html) != '<p>[{$fli_flightticket}]</p>' && $t->rt_emailtyp == 'PDF') {

                        $t->rt_html = "<div style=\" padding:0px 25px\">" . $t->rt_html . "</div>";

                        $this->smartyTemplate = $headerGes . trim($t->rt_html) . $footerGes;
                    } else {
                        $this->smartyTemplate = $headerGes . trim($t->rt_html) . $footerGes;
                    }
                } else {
                    $headerGes = $header;
                    $footerGes = $footer;
                    $this->smartyTemplate = $headerGes . $t->rt_html . $footerGes;
                }
            } else {
                $this->smartyTemplate = stripslashes($headerGes . $param . $footerGes);
            }
        } catch (Exception $e) {
            echo $e->getMessage();
        }
    }

    /**
     * @param Fullbooking $booking
     * @param Smarty $view
     * @param string $from
     * @return mixed
     * @throws Exception
     */
    public function render($booking, &$view, $from = 'accounting')
    {
        try {

            $classification = BookingClassificationService::classifyByBookingNumber($booking->vt_bo_id);
            $this->aLotOfHoteloucher = false;
            $view->assign("client", $this->client);
            $hotelDataVoucher = $booking->hotelDataForVoucher;

            $tmpId = explode(',', $this->ini->hotelvoucherId);

            if (in_array($this->actItem->rt_id, $tmpId) && $hotelDataVoucher[0]->anzahlzimmer > 1) {

                $anzZimmer = (int)$hotelDataVoucher[0]->anzahlzimmer - 1;
                $this->aLotOfHoteloucher = true;
                $hotelDataVoucher[$this->zaehler]->ht_allocation = str_replace(';', ',',
                    $hotelDataVoucher[$this->zaehler]->ht_allocation);
                $participantsToHotelRoom = explode(',', $hotelDataVoucher[$this->zaehler]->ht_allocation);
                $participantsTemp = array();
                foreach ($participantsToHotelRoom as $participants) {
                    $participantsTemp[(int)$participants - 1] = $booking->participants[(int)$participants - 1];
                }
            }

            $this->glossar = new Glossar;
            $view->register_resource('string',
                array('string_get_template', 'string_get_timestamp', 'string_get_secure', 'string_get_trusted'));
            $std_delimiter = $this->glossar->getStandardDelimiter();
            $delimiter = $from == 'cms' ? $this->glossar->getCmsDelimiter() : $this->glossar->getAccountingDelimiter();

            $template = $this->_prepareTemplate($this->smartyTemplate, $view, $from);

            $found = array($std_delimiter['left'], $std_delimiter['right']);
            $replace = array($delimiter['left'], $delimiter['right']);

            if (is_file(dirname(__FILE__) . "/Booking/Templates/bx_mail_participants.tpl.htm")) {
                $ptpl = file_get_contents(dirname(__FILE__) . "/Booking/Templates/bx_mail_participants.tpl.htm");
                $participants_tpl = str_replace($found, $replace, $ptpl);
                $view->clear_all_assign();
            }

            $view->assign("client", $this->client);
            if ($this->aLotOfHoteloucher) {
                $view->assign('participants', $participantsTemp);
                $participants = $view->fetch("string:$participants_tpl");
            } else {
                $view->assign('participants', $booking->participants);
                $participants = $view->fetch("string:$participants_tpl");
            }

            $this->setManuellFlags($booking, $view);

            $view->assign($booking->coreData->getData());

            if (is_file(dirname(__FILE__) . "/Booking/Templates/bx_booking_hotel_invoice.tpl.htm") &&
                !empty($hotelDataVoucher)) {
                $view->assign('hotelsForInvoice', $hotelDataVoucher);

                list($d, $m, $y) = $temp = explode('.', $booking->coreData->vt_bo_created);
                $tempCreated = mktime(0, 0, 0, $m, $d, $y);
                $toCreated = mktime(0, 0, 0, '05', '12', '2014');

                if ($tempCreated <= $toCreated && $classification->getCustomer() === 'LINIE') {
                    $view->assign('oldBooking', 1);
                }

                $tempCreated = mktime(0, 0, 0, $m, $d, $y);
                $toCreated = mktime(0, 0, 0, '07', '22', '2014');

                if ($tempCreated <= $toCreated && $classification->getCustomer() === 'LINIE') {
                    $view->assign('oldBooking2', 1);
                }

                $ptplHotelInvoice = file_get_contents(dirname(__FILE__) . "/Booking/Templates/bx_booking_hotel_invoice.tpl.htm");
                $hotel_Invoice_tpl = str_replace($found, $replace, $ptplHotelInvoice);
                $hotelBookingInvoice = $view->fetch("string:$hotel_Invoice_tpl");
            }

            $participantsHotel = $this->definePassagierDataForHotelTemplate($view, $found, $replace);

            $view->assign('participantsZg', $this->definePassagierDataForZgTemplate($view, $found, $replace));

            if (is_file(dirname(__FILE__) . "/Booking/Templates/bx_mail_flighttimes_invoice.tpl.htm")) {

                $view->assign('vt_fli_sv_departure_t', $booking->savedFlight[0]->vt_fli_sv_departure_t);
                $view->assign('vt_fli_sv_destination_t', $booking->savedFlight[0]->vt_fli_sv_destination_t);
                $view->assign('vt_fli_sv_destination_t', $booking->savedFlight[0]->vt_fli_sv_destination_t);

                $flugTabExt = file_get_contents(dirname(__FILE__) . "/Booking/Templates/bx_mail_flighttimes_invoice.tpl.htm");
                $flugTabExtA = str_replace($found, $replace, $flugTabExt);
            }

            /** @var Pricecorrections[] $priceCorrections */
            $priceCorrections = empty($booking->priceCorrection) ? [] : $booking->priceCorrection;

            $view->assign('pricecorrection', $priceCorrections);

            if (is_file(dirname(__FILE__) . "/Booking/Templates/pricecorrection.tpl.htm") && count($priceCorrections) > 0) {
                $priceCorrectionTpl = file_get_contents(dirname(__FILE__) . "/Booking/Templates/pricecorrection.tpl.htm");
                $priceCorrectionTplA = str_replace($found, $replace, $priceCorrectionTpl);
            }

            foreach ($priceCorrections as $priceCorrection) {
                if (static::PRICE_CORRECTION_WATER_PLANE === $priceCorrection->getVtPriceCorrectionType()) {
                    $view->assign('vt_waterplane_price_description', $priceCorrection->getVtPriceDescription());
                    $view->assign('vt_waterplane_price_price',
                        number_format($priceCorrection->getVtPricePrice(), 2, ',', ''));
                    break;
                }
            }
            // ----- Flighttimes
            if (is_file(dirname(__FILE__) . "/Booking/Templates/bx_mail_flighttimes.tpl.htm")) {
                $ftpl = file_get_contents(dirname(__FILE__) . "/Booking/Templates/bx_mail_flighttimes.tpl.htm");
            }

            if (is_file(dirname(__FILE__) .
                "/Booking/Templates/bx_mail_participants_usa_mail.tpl.htm")) {

                $mpusa = file_get_contents(dirname(__FILE__) . "/Booking/Templates/bx_mail_participants_usa_mail.tpl.htm");
                $mpusa = str_replace($found, $replace, $mpusa);
                $view->assign('mpusa_tpl', $view->fetch("string:$mpusa"));
            }

            if (is_file(dirname(__FILE__) . "/Booking/Templates/bx_mail_participants_usa_mail_en.tpl.htm")) {
                $mpusa_en = file_get_contents(dirname(__FILE__) . "/Booking/Templates/bx_mail_participants_usa_mail_en.tpl.htm");
                $mpusa_en = str_replace($found, $replace, $mpusa_en);
                $view->assign('mpusa_tpl_en', $view->fetch("string:$mpusa_en"));
            }

            if (count($booking->flighttimesHistory) > 0) {
                $ftplH = file_get_contents(dirname(__FILE__) .
                    "/Booking/Templates/bx_mail_flighttimes_history.tpl.htm");

                $flightH_tpl = str_replace($found, $replace, $ftplH);
                $view->assign('flighttimesH', $booking->flighttimesHistory);
                $flighttimesH = $view->fetch("string:$flightH_tpl");
            }

            if (strtolower($booking->coreData->vt_bo_client) == 'ultra') {
                $this->initUltraData($view, $booking, $found, $replace);
            }

            /**
             * PAYCODE LINKS
             */
            if ($booking->coreData->payCodeLinks['AZ']) {
                $view->assign('az_paycodelink', $booking->coreData->payCodeLinks['AZ']);
            }
            if ($booking->coreData->payCodeLinks['RZ']) {
                $view->assign('rz_paycodelink', $booking->coreData->payCodeLinks['RZ']);
            }
            if ($booking->coreData->payCodeLinks['ST']) {
                $view->assign('st_paycodelink', $booking->coreData->payCodeLinks['ST']);
            }

            $flight_tpl = str_replace($found, $replace, $ftpl);
            $view->assign('flighttimes', $booking->flighttimes);
            $flighttimes = $view->fetch("string:$flight_tpl");
            $view->assign('priceCorTab', $view->fetch("string:$priceCorrectionTplA"));
            $view->assign('fli_flighttimesExt', trim($view->fetch("string:$flugTabExtA")));
            $view->assign('hotelsBookingInvoice', $hotelBookingInvoice);
            $view->assign('fli_flighttimes', $flighttimes);
            $view->assign('count_flighttimes', count($flighttimes));
            $view->assign('vt_bo_reiseteilnehmer', $participants);
            $view->assign('count_reiseteilnehmer', count($booking->participants));
            $view->assign('count_adults', $booking->getAdultsOfBookings());
            $view->assign('count_childs', $booking->getChildsOfBookings());
            $view->assign('count_infants', $booking->getInfantsOfBookings());
            $view->assign('participantsHotel', $participantsHotel);
            $view->assign('hotelprice_fremd', $booking->coreData->getHotelPriceForeignCurrency());
            $view->assign('hotelprice_fremdwaehrung', $booking->coreData->getExchangeCurrency());
            $view->assign('sicherungsschein',
                'http://wsvto.vtours.de/img/sicherungsschein_' . $this->ini->client . '.gif');
            $dateToday2 = date('d') . '. ' . $this->dateMonthGerman[date('m')] . ' ' . date('Y');
            $view->assign('dateToday2', $dateToday2);

            $view->assign('trustpilot_link',
                $this->generateTrustPilotLink($booking->vt_bo_id,
                    $booking->coreData,
                    strtolower($booking->coreData->vt_bo_client))
            );

            if (count($booking->flighttimesHistory) > 0) {
                $view->assign('fli_flighttimesH', $flighttimesH);
                $view->assign('count_flighttimes', count($flighttimesH));
            }

            if ($this->aLotOfHoteloucher) {
                $view->assign($hotelDataVoucher[$this->zaehler]->getData());
            } else {
                $view->assign($booking->hotelData->getData());
            }

            $view->assign($booking->customerData->getData());
            $view->assign($booking->getHotelIncomingData());
            $view->assign('zginc', $booking->getZgIncomingData());

            $view->assign('ryanair_email', $booking->coreData->ryanair_email);

            if ($this->client == 'xlow') {
                $tmpXC = $booking->getXlowContactDataForTemplate();

                if (is_array($tmpXC) && count($tmpXC) >= 1) {
                    foreach ($tmpXC as $keyXC => $valueXC) {
                        $view->assign($keyXC, $valueXC);
                    }
                }
            }
            $view->assign("client", $this->client);
            if ((int)$booking->coreData->vt_bo_storno == 1) {
                $view->assign('storno_due_date', date('d.m.Y', mktime(0, 0, 0, date('m'), date('d') + 7, date('Y'))));
            }
            if ($booking->coreData->vt_bo_prepayment_due) {
                $view->assign('vt_bo_prepayment_due2',
                    $this->addBankdaysToDate($booking->coreData->vt_bo_prepayment_due, 7, 'd.m.Y'));
            }

            if ($booking->coreData->vt_bo_balancepayment_due) {
                $view->assign('vt_bo_balancepayment_due2',
                    $this->addBankdaysToDate($booking->coreData->vt_bo_balancepayment_due, 4, 'd.m.Y'));
            }

            $view->assign('is_vfly', $booking->coreData->is_vfly);
            $view->assign('displayServiceChargeSeparately', (strtotime($booking->coreData->vt_bo_created) >= strtotime(self::MYTHOS_MIGRATION_VFLY_DATE)) ? 1 : 0 );

            if ($booking->coreData->is_vfly == 1) {
                if ($booking->coreData->vt_bo_custcalculationdate == null || $booking->coreData->vt_bo_custcalculationdate == '') {
                    $view->assign('vt_bo_balancepayment_due2', $this->addBankdaysToDate(date('d.m.Y'), 4, 'd.m.Y'));
                } else {
                    $view->assign('vt_bo_balancepayment_due2',
                        $this->addBankdaysToDate($booking->coreData->vt_bo_custcalculationdate, 4, 'd.m.Y'));
                }
            }

            $temp = $booking->flighttimes;

            $pfad = null;
            $client = $this->client;

            $savedFlights = array();
            foreach ($booking->savedFlight as $obj) {
                $savedFlights[] = $obj;
            }

            $flugVoucherDiff = 0;

            if (count($savedFlights) == 2
                && $this->checkAndSetFlightTimesForFlightSeqment($savedFlights, $temp)) {
                $flugVoucherDiff = 1;
            }


            if (!empty($savedFlights)) {
                $view->assign('savedFlights', $savedFlights);

                $pfad = $this->ini->templates->flugvoucher->$client;

                if (is_file(dirname(__FILE__) . $pfad)) {
                    $ftpl = file_get_contents(dirname(__FILE__) . $pfad);
                }

                $view->assign('flugVoucherDiff', $flugVoucherDiff);
                $view->assign('date_today', date('d.m.Y'));
                $view->assign('airlinecodeandname', str_replace("/", " - ", $booking->getAirlineCodeAndName()));
                $view->assign('airline', $temp[0]->fli_flightairlinename);
                $view->assign('airlinecode', $temp[0]->fli_flight_airlinecode);
                $view->assign('vt_fli_sv_airline', $savedFlights[0]->vt_fli_sv_airline);
                $view->assign('vt_fli_sv_processno', $savedFlights[0]->vt_fli_sv_processno);
                $pageHeader = (count($savedFlights) * 2 + 1);
                $view->assign('pag', $pageHeader);
                $tickets_tpl = str_replace($found, $replace, $ftpl);
                $flighttckets = $view->fetch("string:$tickets_tpl");
                $view->assign('fli_flightticket', $flighttckets);
                $view->assign('hotelDataVoucher', $hotelDataVoucher);
                $tickets_tpl = str_replace($found, $replace, $ftpl);
            }

            $this->assignHotelDataAsArray($view, $booking);

            if ($this->aLotOfHoteloucher && $from != "cms") {
                if ((int)$this->zaehler < (int)$anzZimmer) {
                    $this->rendered .= $view->fetch("string:$template");
                    (int)$this->zaehler++;
                    $this->render($booking, $view, $from);
                    $this->zaehler--;
                    if ($this->zaehler == 0) {
                        $render = $this->rendered;
                        $this->rendered = null;
                        $this->renderCSS = 1;
                        $this->aLotOfHoteloucher = false;
                        return $this->getHtmlHead() . $render . $this->getHtmlFooter();
                    } else {
                        return;
                    }
                } elseif ((int)$this->zaehler == (int)$anzZimmer) {
                    $this->rendered .= $view->fetch("string:$template");
                    return;
                }
            } else {
                $rendered = $view->fetch("string:$template");
            }
            return $rendered;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param string $tpl
     * @param string $view
     * @param string $from
     * @return string|string[]
     * @throws Exception
     */
    private function _prepareTemplate($tpl, &$view, $from)
    {


        $this->glossar->setLdelim('[{');
        $this->glossar->setRdelim('}]');
        $std_delimiter = $this->glossar->getStandardDelimiter();
        $delimiter = $from == 'cms' ? $this->glossar->getCmsDelimiter() : $this->glossar->getAccountingDelimiter();

        $left = str_replace("[", "\[", $std_delimiter['left']);
        $right = str_replace("]", "\]", $std_delimiter['right']);

        $pattern_inline = '/(' . $left . '(?P<inlines>\$tpl_[a-z0-9_]*)' . $right . ')/';
        preg_match_all($pattern_inline, $tpl, $in);

        foreach ($in['inlines'] as $inliner) {
            list($t, $tid) = explode("_", $inliner);
            $inline_tpl = $this->getTemplateById($tid);
            $back = $this->_prepareTemplate($inline_tpl->rt_html, $view, $from);

            $tpl = str_replace($std_delimiter['left'] . $inliner . $std_delimiter['right'], $back, $tpl);
        }

        $pattern_opt = '/(' . $left . '(?P<options>\$option_[a-z0-9_]*(\.html)?|(\.text)?)' . $right . ')/';
        preg_match_all($pattern_opt, $tpl, $o);

        if (!empty($o['options'])) {
            $ois = array();
            foreach ($o['options'] as $opt) {
                list($item, $oid) = explode("_", array_shift(explode(".", $opt)));
                $ois[] = $oid;
            }

            $oO = new Options();
            $allOptions = $oO->getAllOptions();

            if ($from == 'cms') {
                foreach ($allOptions as $opt) {
                    if (in_array($opt->ro_id, $ois)) {
                        $tpl = str_replace($std_delimiter['left'] . '$option_' . $opt->ro_id . ".html" . $std_delimiter['right'],
                            "PLATZHALTER " . strtoupper($opt->ro_name), $tpl
                        );
                        $tpl = str_replace($std_delimiter['left'] . '$option_' . $opt->ro_id . ".text" . $std_delimiter['right'],
                            "PLATZHALTER " . strtoupper($opt->ro_name), $tpl
                        );
                        $tpl = str_replace($std_delimiter['left'] . '$option_' . $opt->ro_id . $std_delimiter['right'],
                            "PLATZHALTER " . strtoupper($opt->ro_name), $tpl
                        );
                    }
                }
            }
        }


        $pattern = '/(' . $left . '(?P<variables>\$[a-z0-9_\.]*)' . $right . ')/';

        preg_match_all($pattern, $tpl, $m);
        $found = array();
        $replace = array();

        foreach ($m['variables'] as $v) {
            $found[] = $std_delimiter['left'] . $v . $std_delimiter['right'];
            $replace[] = $delimiter['left'] . $v . $delimiter['right'];
        }

        $template = str_replace($found, $replace, $tpl);
        $template = str_replace($std_delimiter['left'], $delimiter['left'], $template);
        $template = str_replace($std_delimiter['right'], $delimiter['right'], $template);

        return $template;
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getHeadersTemplates()
    {

        try {
            $stmt = "select * from $this->table where RT_DOKUTYP = 'HEADER'";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen aller Headers Templates. MSG::' . $this->db->errorMsg());
            }

            foreach ($res as $key => $values) {
                $data[$key] = new TemplateItem($values);
            }

            return $data;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param int $id
     * @return TemplateItem
     * @throws Exception
     */
    public function getHeadersTemplatesById($id)
    {

        try {
            $stmt = "select * from $this->table where RT_DOKUTYP = 'HEADER' and $this->pk = $id";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen des Header Templates.ID::' . $id . ' MSG::' . $this->db->errorMsg());
            }

            return new TemplateItem($res[0]);

        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @return array
     * @throws Exception
     */
    public function getFootersTemplates()
    {

        try {

            $stmt = "select * from $this->table where RT_DOKUTYP = 'FOOTER'";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen aller Foote3r Templates. MSG::' . $this->db->errorMsg());
            }

            foreach ($res as $key => $values) {
                $data[$key] = new TemplateItem($values);
            }

            return $data;
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param int $id
     * @return TemplateItem
     * @throws Exception
     */
    public function getFootersTemplatesById($id)
    {

        try {

            $stmt = "select * from $this->table where RT_DOKUTYP = 'FOOTER' and $this->pk = '$id'";

            $res = $this->db->getAll($stmt);

            if ($this->db->errorMsg()) {
                throw new Exception('Fehler beim holen des Header Templates.ID::' . $id . ' MSG::' . $this->db->errorMsg());
            }

            return new TemplateItem($res[0]);

        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }

    /**
     * @param TemplateItem $obj
     * @throws Exception
     */
    public function initializeItem($obj)
    {
        if ($obj instanceof TemplateItem) {
            $this->actItem = $obj;
        } else {
            throw new Exception('Kein Item Objekt im Initialize');
        }
    }

    /**
     *
     */
    public function unInitializeItem()
    {


        unset($this->actItem);
    }

    /**
     * @param int $noBorder
     * @return string
     */
    public function getHtmlHead($noBorder = 0)
    {

        $pdfHead = '<pd4ml:page.header>
					<div style="margin-bottom: 20px;">
					<table style="font-family: Arial; font-size: 9pt; font-weight: bold; color: #9c9c9c; width: 100%;">
					<tbody>
						<tr>
							<td class="tright">[{$logopool_logo}]</td>
						</tr>
					</tbody>
					</table>
					</div>
					</pd4ml:page.header>';
        if ($this->client == 'vto') {
            $pdfFoot = ' <pd4ml:page.footer>
					 <div><img src="http://vtours.de/res/footer_rula2.jpg" height="130px" /></div>
					 </pd4ml:page.footer>';
        }
        if ($this->client == 'vtoidach') {
            $pdfFoot = ' <pd4ml:page.footer>
					 <div><img src="https://vtours.com/templates/tpl/src/img/footer_vtoi.jpg"/></div>
					 </pd4ml:page.footer>';
        }
        if ((int)$this->actItem->rt_header == 1) {
            $str = $pdfHead;
        }
        if ((int)$this->actItem->rt_footer == 2) {
            $str .= $pdfFoot;
        }

        $str .= '<!DOCTYPE html>
				<html>
				<head>
				<meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
				<title>Dokument</title>';

        if ($this->renderCSS) {
            $css = $this->tinyCss();
            $str .= "<!--[{literal}]--><style type=\"text/css\"><!--" . $css . "--></style><!--[{/literal}]-->";
        }

        if ($noBorder == 1) {
            $str .= '</head><body style="margin:0;padding:0;">';
        } else {
            $str .= '</head><body>';
        }

        return $str;
    }

    /**
     * @return string
     */
    public function getHtmlFooter()
    {
        return "</body></html>";
    }

    /**
     * @param array $flightSaved
     * @param null|array  $flighttimes
     * @return bool|void
     */
    public function checkAndSetFlightTimesForFlightSeqment(&$flightSaved, $flighttimes = null)
    {
        $stat = true;

        foreach ($flightSaved as $item) {
            if ($item instanceof BookingSavedFlight) {

                $res = $item->getDataAsObject();
                $beginn = mktime($res->vt_fli_sv_beginn);
                $temp = explode('.', $res->vt_fli_sv_beginn);
                $beginn_1 = mktime(date('d.m.Y', mktime(0, 0, 0, $temp[1], $temp[0] + 1, $temp[2])));
                $end = mktime($res->vt_fli_sv_end);

                if (!($beginn == $end || $beginn_1 == $end)) {
                    $stat = false;
                } else {
                    if ($flighttimes == null || count($flighttimes) <= 1) {
                        $stat = false;
                    } else {
                        $erg = $this->getFlightItem($beginn, $beginn_1, $flighttimes);
                        if (count($erg) <= 0) {
                            $stat = false;
                        }
                    }
                }

                if (!$stat) {
                    continue;
                }

                $item->flugzeit = $erg;
            }
        }

        return ($stat);
    }

    /**
     * @param string $date1
     * @param string $date2
     * @param array $flighttimes
     * @return array|false|void
     */
    public function getFlightItem($date1, $date2, $flighttimes)
    {


        $dataBack = array();
        $resf = null;

        foreach ($flighttimes as $fItem) {

            if ($fItem instanceof BookingFlightSegment) {
                $resf = $fItem->getDataAsObject();
                if (mktime($resf->fli_flightbeginn) == $date1 || mktime($resf->fli_flightbeginn) == $date2) {
                    array_push($dataBack, $resf);
                }
            } else {
                return;
            }
        }

        if ($dataBack == null || count($dataBack) == 0) {
            return false;
        } else {
            return $dataBack;
        }
    }

    /**
     * @param string $date
     * @param int $addDays
     * @param string $format
     * @return false|string
     * @throws Exception
     */
    private function addBankdaysToDate($date, $addDays = 2, $format = 'Y-m-d')
    {

        if (preg_match('/^([0-3]{1}[0-9]{1}[.\-;:,]{1}[0-1]{1}[0-9]{1}[.\-;:,]{1}[0-9]{4})$/', $date)) {

            $givenFormat = 'd.m.Y';

            $date = str_replace(array(';,;:'), '.', $date);

            list($day, $month, $year) = explode('.', $date);
        } elseif (preg_match('/^([0-9]{4}[.\-;:,]{1}[0-1]{1}[0-9]{1}[.\-;:,]{1}[0-3]{1}[0-9]{1})$/', $date)) {

            $givenFormat = 'Y-m-d';

            $date = str_replace(array(';,;:'), '-', $date);

            list($year, $month, $day) = explode('-', $date);
        } else {
            throw new Exception('no valid Format given in ' . __METHOD__);
        }

        if ($format == null || $format == '') {
            $format = $givenFormat;
        }

        if (!checkdate($month, $day, $year)) {
            throw new Exception("no valid Date given in " . __METHOD__);
        }

        $i = 0;
        while ($addDays > 0) {

            $tempDayName = date('D', mktime(0, 0, 0, $month, $day + $i, $year));
            $tempNewDate = date($format, mktime(0, 0, 0, $month, $day + $i, $year));

            $i++;
            if (strtolower($tempDayName) != 'sun' && strtolower($tempDayName) != 'sat') {
                $addDays--;
            }
        }

        return $tempNewDate;
    }

    /**
     * @param array $booking
     * @param Smarty $view
     * @return bool
     */
    private function setManuellFlags($booking, $view)
    {

        $time = $booking->coreData->vt_bo_created;

        list($day, $month, $year) = explode('.', $time);

        if ($booking->coreData->vt_fli_sv_provider == 3000007 &&
            (substr(trim($booking->coreData->airline), -2) == 'XQ' || substr(trim($booking->coreData->airline),
                    -2) == 'XG') &&
            mktime(0, 0, 0, $month, $day, $year) < mktime(0, 0, 0, 11, 01, 2015)
        ) {
            $booking->coreData->procNoNotExport = 1;
            $view->assign('procNoNotExport', 1);
        }

        return true;
    }

    /**
     * @param Smarty $view
     * @param array $booking
     * @param array $found
     * @param array $replace
     * @return bool
     */
    private function initUltraData(&$view, &$booking, $found, $replace)
    {

        if ($this->log) {
            $view->assign('language_ut_de', $this->ini->ultra->language_ut_de);
            $view->assign('language_ut_en', $this->ini->ultra->language_ut_en);
            $view->assign('logo_en', $this->ini->ultra->logo_en);
            $view->assign('logo_de', $this->ini->ultra->logo_de);
        } else {
            $de = \Config::stepIn()->cfg('ut_language_ut_de');
            $en = \Config::stepIn()->cfg('ut_language_ut_en');
            $view->assign('language_ut_de', $de);
            $view->assign('language_ut_en', $en);
            $view->assign('logo_en', \Config::stepIn()->cfg('ut_logo_en'));
            $view->assign('logo_de', \Config::stepIn()->cfg('ut_logo_de'));
        }

        $transferData = $booking->getUtTransferData();
        $view->assign('uttransfers', $transferData);

        if (count($transferData) > 0) {
            try {
                //Rechnung
                $view->assign('transfers', $transferData);
                $utTransfers = file_get_contents(dirname(__FILE__)
                    . "/Booking/Templates/Ultra/ut_transfers_invoice.tpl.htm");

                $utTransfers_tpl = str_replace($found, $replace, $utTransfers);
                $transferTemp = $view->fetch("string:$utTransfers_tpl");
                $view->assign('transferDetails', $transferTemp);

                //Voucher
                $view->assign('vt_bo_customer_lang', 'de');
                $lang = $booking->coreData->vt_bo_customer_lang;
                if (!empty($lang)) {
                    $view->assign('vt_bo_customer_lang', strtolower($lang));
                }
                $utTransfers = file_get_contents(dirname(__FILE__)
                    . "/Booking/Templates/Ultra/ut_transfers_voucher.tpl.htm");

                $utTransfers_tpl = str_replace($found, $replace, $utTransfers);
                $transferTemp = $view->fetch("string:$utTransfers_tpl");
                $view->assign('transferDetailsVoucher', $transferTemp);
            } catch (Exception $e) {
                return false;
            }

            return true;
        }
    }

    /**
     * @param Smarty $view
     * @param FullBooking $booking
     */
    private function assignHotelDataAsArray($view, $booking)
    {
        $view->assign('hotelArr', $booking->getHotelArray());
    }

    /**
     * @param Smarty $view
     * @param string|array $found
     * @param string|array $replace
     * @return mixed
     */
    private function definePassagierDataForHotelTemplate($view, $found, $replace)
    {

        if (is_file(dirname(__FILE__) . "/Booking/Templates/hotelvoucher_participants_tabelle.htm.tpl")) {
            $ptplHotelVoucher = file_get_contents(dirname(__FILE__) . "/Booking/Templates/hotelvoucher_participants_tabelle.htm.tpl");
            $participants_Hotel_tpl = str_replace($found, $replace, $ptplHotelVoucher);
            return $view->fetch("string:$participants_Hotel_tpl");
        }

    }

    /**
     * @param Smarty $view
     * @param string|array$found
     * @param string|array$replace
     * @return false|string|void
     */
    private function definePassagierDataForZgTemplate($view, $found, $replace)
    {

        if (is_file(dirname(__FILE__) . "/Booking/Templates/zgmeldung_participants_tabelle.htm.tpl")) {
            $ptplZgInfo = file_get_contents(dirname(__FILE__) . "/Booking/Templates/zgmeldung_participants_tabelle.htm.tpl");
            $participants_zg_tpl = str_replace($found, $replace, $ptplZgInfo);
            return $view->fetch("string:$participants_zg_tpl");
        }

    }

    /**
     * @param string $boId
     * @param BookingCore $coreData
     * @param string $client
     * @return string
     */
    private function generateTrustPilotLink($boId, BookingCore $coreData, $client)
    {
        $trustPilotDomain = ($client === 'ultra') ? 'urlaubstransfers.de' : 'vtours.com';
        $encrypt_key = ($client === 'ultra') ? 'rGoLWoWvRbWWCi+H+Not02rOJBAaX1kOFLxiPTnwlUE=' : 'EF75t/86FwUo+O2dQAPy1AJsgFttOB8vNS4tPo/zyOU=';
        $auth_key = ($client === 'ultra') ? 'GaKOdue1VTXh5uOmfTWmpSPBxOlvdMAYi3MDtenws3U=' : 'xEt3st77XSIaZcySFV+BVW9Hh1W2eWkAwbv06Hj9p4s=';

        $customerDataJson = [
            'email' => $coreData->getCustomerEmail(),
            'name' => $coreData->getCustomerFirstname() . ' ' . $coreData->getCustomerName(),
            'ref' => $boId
        ];

        $payload = $this->encryptPayload(json_encode($customerDataJson), base64_decode($encrypt_key), base64_decode($auth_key));

        return 'https://de.trustpilot.com/evaluate-bgl/' . $trustPilotDomain . '?p=' . $payload;
    }

    /**
     * @param string|array $payload
     * @param string $encrypt_key
     * @param string $auth_key
     * @return string
     */
    private function encryptPayload($payload, $encrypt_key, $auth_key)
    {
        // Generate an Initialization Vector (IV) according to the block size (128 bits)
        $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length('AES-128-CBC'));

        //Encrypting the JSON with the encryptkey and IV with AES-CBC with key size of 256 bits, openssl_encrypt uses PKCS7 padding as default
        $payload_encrypted = openssl_encrypt($payload, 'AES-256-CBC', $encrypt_key, OPENSSL_RAW_DATA, $iv);

        //Create a signature of the ciphertext.
        $HMAC = hash_hmac('sha256', ($iv . $payload_encrypted), $auth_key, true);

        //Now base64-encode the IV + ciphertext + HMAC:
        $base64_payload = base64_encode(($iv . $payload_encrypted . $HMAC));

        return urlencode($base64_payload);
    }
}

/**
 * @param string $query
 * @param string $tpl_source
 * @param Smarty $smarty_obj
 * @return bool
 */
function string_get_template($query, &$tpl_source, &$smarty_obj)
{
    $tpl_source = $query;
    return true;
}

/**
 * @param string $tpl_name
 * @param string $tpl_timestamp
 * @param Smarty $smarty_obj
 * @return bool
 */
function string_get_timestamp($tpl_name, &$tpl_timestamp, &$smarty_obj)
{


    // do database call here to populate $tpl_timestamp.
    $tpl_timestamp = time();
    return true;
}

/**
 * @param string $tpl_name
 * @param Smarty $smarty_obj
 * @return bool
 */
function string_get_secure($tpl_name, &$smarty_obj)
{


    // assume all templates are secure
    return true;
}

/**
 * @param string $tpl_name
 * @param Smarty $smarty_obj
 */
function string_get_trusted($tpl_name, &$smarty_obj){}
