<?php

namespace Vtours\Rula2Engine\Booking;

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2021 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

use Vtours\Rula2Engine\Extendablemagic;
use Vtours\Rula2Engine\Service\MobilePhoneCleanerAndValidator;
use Exception;


/**
 * Class BookingCore
 */
class BookingCore extends Extendablemagic
{

    /**
     * @var string[]
     */
    private $erdeKenner = ['E', 'B'];

    /**
     * @var string[]
     */
    private $pauschKenner = ['X', 'Y', 'C'];

    /**
     * @var string
     */
    public $er_exchangecurrencycode;

    /**
     * @var float
     */
    public $er_conversionrate;

    /**
     * @var float
     */
    public $vt_bo_hotelprice;

    /**
     * @return bool
     */
    public function isErde()
    {
        return (in_array($this->vt_bo_bookingtype, $this->erdeKenner));
    }

    /**
     * @return bool
     */
    public function isFlightOnly()
    {
        return ($this->vt_bo_bookingtype == 'F');
    }

    /**
     * @return bool
     */
    public function isHotelAndFlight()
    {
        return (in_array($this->vt_bo_bookingtype, $this->pauschKenner));
    }

    /**
     * @return bool
     */
    public function isPausch()
    {
        return ($this->vt_bo_bookingtype == 'P');
    }

    /**
     * @return bool
     */
    public function hasHotel()
    {
        return ($this->isErde() || $this->isHotelAndFlight());
    }

    /**
     * @return false|string
     * @throws Exception
     */
    public function getTravelYear()
    {
        return substr($this->getBeginnDate(), 0, 4);
    }

    /**
     * @return mixed|string
     * @throws Exception
     */
    public function getBeginnDate()
    {
        if ($this->isErde()) {
            return $this->ht_beginn;
        }
        if ($this->isFlightOnly()) {
            return $this->vt_fli_sv_beginn;
        }
        if ($this->vt_bo_beginn) {
            return $this->vt_bo_beginn;
        }

        throw new Exception('Beginn date for ' . $this->vt_bo_id . ' not set!');
    }

    /**
     * @return mixed|string
     * @throws Exception
     */
    public function getZielgebiet()
    {
        if ($this->vt_bo_destination) {
            return $this->vt_bo_destination;
        }
        throw new Exception('Zielgebiet (' . $this->vt_bo_destination . ') not set!');
    }

    /**
     * @return mixed|string
     * @throws Exception
     */
    public function getLocationCode()
    {
        if ($this->hasHotel()) {
            return $this->ht_hotelloccode;
        }

        if ($this->isPausch()) {
            return $this->vt_pt_hotellocationcode;
        }

        if ($this->isFlightOnly()) {
            return 'Nur Flug';
        }

        throw new Exception ('Location code not set!');
    }

    /**
     * @return mixed|string|void
     * @throws Exception
     */
    public function getHotelCode()
    {

        if ($this->hasHotel()) {
            return $this->ht_hotelcode;
        }

        if ($this->isPausch()) {
            return $this->vt_pt_productcode;
        }

        if ($this->isFlightOnly()) {
            return 'Nur Flug';
        }

        if (!$this->hasHotel() && strtoupper($this->vt_bo_bookingtype) == 'TO') {
            return;
        }

        throw new Exception ($this->vt_bo_id . ': Hotel code not set! (' . $this->vt_bo_bookingtype . ')');
    }

    /**
     * @return mixed|string
     */
    public function getCustomerEmail()
    {
        return $this->vt_bo_customer_email;
    }

    /**
     * @return mixed|string
     */
    public function getBalancePayment()
    {
        return $this->vt_bo_balancepayment;
    }

    /**
     * @return string
     */
    public function getInternalTreatmentConditi()
    {
        return trim(strtolower($this->vt_bo_internaltreatmentconditi));

    }

    /**
     * @return mixed|string
     */
    public function getVtFlightsSeqments()
    {
        return $this->vt_fli_flight_seqments;
    }

    /**
     * @return mixed|string
     */
    public function getCustomerName()
    {
        return $this->vt_bo_customer_name;
    }

    /**
     * @return mixed|string
     */
    public function getCustomerFirstname()
    {
        return $this->vt_bo_customer_firstname;
    }

    /**
     * @return mixed|string
     */
    public function getTransferCode()
    {
        return $this->ht_transferagent;
    }

    /**
     * @return bool
     */
    public function isRick()
    {
        return strtoupper($this->vt_bo_cashing) == 'RIK' ? true : false;
    }

    /**
     * @return mixed|string
     */
    public function getIncomingAgentur()
    {
        return $this->ht_incagent;
    }

    /**
     * @return mixed|string
     */
    public function getBookingType()
    {
        return $this->vt_bo_bookingtype;
    }

    /**
     * @return bool|string
     */
    public function getExchangeRate()
    {
        if (!(float)$this->er_conversionrate) {
            return 1;
        }
        return (float)$this->er_conversionrate;
    }

    /**
     * @return float
     */
    public function getExchangeCurrency()
    {
        if (!$this->er_exchangecurrencycode) {
            return false;
        }
        return strtoupper($this->er_exchangecurrencycode);
    }

    /**
     * @return float
     */
    public function getHotelPrice()
    {
        return (float)$this->vt_bo_hotelprice;
    }

    /**
     * @return string
     */
    public function getHotelPriceForeignCurrency()
    {
        return number_format(
            $this->getHotelPrice() / $this->getExchangeRate(),
            2,
            '.',
            ''
        );
    }

    /**
     * @param string $type
     * @param string|null $preSetNumber
     * @return bool|string
     */
    public function getMobileNumber($type = 'smsclient', $preSetNumber = null)
    {
        //Prio hat das Feld Telefonnummer. Wenn es eine valide Handynummer ist, wird diese herangezogen
        //Ist das Feld mit einer Festnetznummer gefüllt oder leer wird das Feld Mobil auf eine verarbeitbare Nr. geprüft
        //do not know which the client got issued in case of fields so keep it empty

        $customerNumber = (null !== $preSetNumber) ? $preSetNumber : $this->vt_bo_customer_telephone;
        $customerMobileNumber = (null !== $preSetNumber) ? $preSetNumber : $this->vt_bo_customer_mobile;

        //init things
        $numberCheck = new MobilePhoneCleanerAndValidator();
        $phoneNumber = $numberCheck->checkPhoneNumber(($type === 'smsclient') ? $customerNumber : '');
        $mobileNumber = $numberCheck->checkPhoneNumber(($type === 'smsclient') ? $customerMobileNumber : '');

        return (false !== $phoneNumber) ? $phoneNumber : $mobileNumber;
    }


}
