<?php

namespace TraveltainmentApi\WebService;

use Vtours\TTXmlWebService\Model\Response\Hotel;
use Vtours\TTXmlWebService\Model\Response\Meal;
use Vtours\TTXmlWebService\Model\Response\Offer;
use Vtours\TTXmlWebService\Model\Response\Region;
use Vtours\TTXmlWebService\Model\Response\SearchEngineHotelListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineRegionTreeRS;
use Vtours\TTXmlWebService\Model\Response\WeatherResponse\YearOverview;
use Vtours\TTXmlWebService\Service\TravelTainmentService;

class TraveltainmentApiService
{

    /**
     * @var TravelTainmentService $traveltainmentService
     */
    private $traveltainmentService;

    public function __construct()
    {
        $this->traveltainmentService = new TravelTainmentService();
    }

    public function getOfferByHotelCode(array $travelDuration, array $travelPeriod, $mealType, $traveller, $hotelCode, $hotelCodeType, $offerType)
    {
        /**
         * @var $searchEngineOfferListResponse SearchEngineOfferListRS
         */
        $searchEngineOfferListResponse = $this->traveltainmentService->getAllHotelOffersByGiata($travelDuration, $travelPeriod, $mealType, $traveller,
            $hotelCode, $hotelCodeType, $offerType);

        if ($searchEngineOfferListResponse->getInfoMessage() == null) {

            if ($searchEngineOfferListResponse->getResultSet()->getTotalNumberOfResults() != 0 && $searchEngineOfferListResponse->isSuccess()) {
                $objectData = [];
                $objectData['request'] = $this->getRequiredRequestData($objectData, $travelPeriod, $traveller);
                $offer = $searchEngineOfferListResponse->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $searchEngineOfferListResponse->getHotel() ?: $searchEngineOfferListResponse->getResultSet()->getListItem()[0]->getHotel();
                $region = $searchEngineOfferListResponse->getRegion() ?: $searchEngineOfferListResponse->getResultSet()->getListItem()[0]->getRegion();
                $objectData = $this->createObjectData($objectData, $offer, $hotel, $region, $offerType, 'offer');

                return $objectData;
            } else {
                return [];
            }
        } else {
            //infomessage is only set, if tt returns an error
            //maybe add info for pimcore-user
            return [];
        }
    }


    public function getOfferByRegionCode(array $travelDuration, array $travelPeriod, $mealType, $traveller, $regionCode, $offerType)
    {
        /**
         * @var $searchEngineHotelListResponse SearchEngineHotelListRS
         */
        $searchEngineHotelListResponse = $this->traveltainmentService->getBestHotelOffersByRegion($travelDuration, $travelPeriod, $mealType,
            $traveller, $regionCode, $offerType);

        if ($searchEngineHotelListResponse->getInfoMessage() == null) {

            if ($searchEngineHotelListResponse->getResultSet()->getTotalNumberOfResults() != 0 && $searchEngineHotelListResponse->isSuccess()) {
                $objectData = [];
                $objectData['request'] = $this->getRequiredRequestData($objectData, $travelPeriod, $traveller);

                $offer = $searchEngineHotelListResponse->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $searchEngineHotelListResponse->getResultSet()->getListItem()[0]->getHotel();
                $region = $searchEngineHotelListResponse->getRegion() ?: $searchEngineHotelListResponse->getResultSet()->getListItem()[0]->getRegion();
                $objectData = $this->createObjectData($objectData, $offer, $hotel, $region, $offerType, 'hotel');

                return $objectData;
            } else {
                return [];
            }
        } else {
            //infomessage is only set, if tt returns an error
            //maybe add info for pimcore-user
            return [];
        }
    }


    public function getCompleteRegionTree()
    {
        /**
         * @var $regionTreeResponse SearchEngineRegionTreeRS
         */
        $regionTreeResponse = $this->traveltainmentService->getRegionTree();
        if ($regionTreeResponse->isSuccess() == true) {
            return $regionTreeResponse;
        } else {
            return new \Exception('no regions available from TT');
        }
    }


    public function createObjectData(array $objectData, Offer $offer, Hotel $hotel, Region $region, $offerType, $requestType)
    {
        $objectData['hotel'] = $this->getHotelData($hotel, $objectData);
        $objectData['region'] = $this->getRegionData($region, $objectData);
        $objectData['offer'] = $this->getOfferData($offer, $objectData, $offerType);
        $objectData['deeplink'] = $this->createDeeplink($objectData, $requestType);

        return $objectData;
    }

    public function getHotelData(Hotel $hotel, array $objectData)
    {
        if ($hotel != null) {
            $objectData = [
                'hotelName'    => $hotel->getObjectName(),
                'hotelCity'    => $hotel->getCity()->getCity(),
                'hotelStars'   => $hotel->getCategory()->getStars(),
                'hotelPicture' => $hotel->getObjectImages()->getLargeImage(),
                'hotelIff'     => $hotel->getObjectId(),
                'hotelGiata'   => $hotel->getAccomIds()[0]
            ];
        }

        return $objectData;
    }

    public function getRegionData(Region $region, $objectData)
    {
        /** @var $clima YearOverview */
        $regionClima = '';
        if ($region->getWeatherForRegion() != null) {
            foreach ($region->getWeatherForRegion()->getClimaData()->getYearOverview() as $clima) {
                if ($clima->getClimaDataType() == 'MaxTemperature') {
                    $regionClima = $clima->getValueOfMonth()->getValueOfMonth();
                }
            }
        }

        if ($region != null) {
            $objectData = [
                'hotelCountry'  => $region->getCountry(),
                'regionClima'   => $regionClima,
                'topRegionId'   => $region->getTopRegionId(),
                'destinationId' => $region->getDestinationId()
            ];
        }

        return $objectData;
    }

    public function getOfferData(Offer $offer, $objectData, $offerType)
    {
        if ($offer != null) {
            $objectData = [
                'offerID'             => $offer->getOfferId(),
                'offerMinPrice'       => $offer->getPriceInformation()->getOriginalPrice()->getOriginalPrice(),
                'offerPriceCurrency'  => $offer->getPriceInformation()->getOriginalPrice()->getCurrency(),
                'offerWithFlight'     => false,
                'offerWithTransfer'   => false,
                'offerTravelDuration' => $offer->getTravelDuration()
            ];
            if ($offerType == 'Hotel') {
                /** @var Meal $meal */
                $meal = $offer->getAccommodation()->getMeal();
                $objectData['offerMealType'] = $meal->getMeal();
                $objectData['offerRoomType'] = $offer->getAccommodation()->getRoom()->getRoom();
            } else {
                if ($offerType == 'Package') {
                    $objectData['offerWithFlight'] = true;
                    $objectData['offerMealType'] = $offer->getPackage()->getAccommodation()->getMeal()->getMeal();
                    $objectData['offerRoomType'] = $offer->getPackage()->getAccommodation()->getRoom()->getRoom();

                    $journeyAttributes = $offer->getSpecialJourneyAttributes();
                    if ($journeyAttributes != null && (strpos($journeyAttributes, 'TRANSFER')) !== false) {
                        $objectData['offerWithTransfer'] = true;
                    }
                }
            }
        }

        return $objectData;
    }

    public function getRequiredRequestData(array $objectData, array $travelPeriod, $traveller)
    {
        $objectData = [
            'departure' => $travelPeriod[0],
            'return'    => $travelPeriod[1],
            'adults'    => $traveller
        ];

        return $objectData;

    }

    public function createDeeplink(array $objectData, $requestType)
    {
        $deeplinkData = [];
        if ($requestType == 'Package') {
            $deeplinkData['prefix'] = 'http://urlaub.vtours.de/' . $requestType . '?';
        } else {
            $deeplinkData['prefix'] = 'http://' . strtolower($requestType) . '.vtours.de/' . $requestType . '?';
        }

        $ddate = $objectData['request']['departure'];
        $ddate = $ddate->format('Y-m-d');
        $rdate = $objectData['request']['return'];
        $rdate = $rdate->format('Y-m-d');
        $aid = $objectData['hotel']['hotelIff'];
        $aidgiata = $objectData['hotel']['hotelGiata'];
        $adult = $objectData['request']['adults'];
        $cyid = $objectData['region']['destinationId'];
        $rid = $objectData['region']['topRegionId'];
        $dur = $objectData['offer']['offerTravelDuration'];

        $parameters = ('ddate=' . $ddate . '&' . 'rdate=' . $rdate . '&' . 'adult=' . $adult . '&' . 'cyid=' . $cyid . '&' . 'rid=' . $rid . '&' . 'dur=' . $dur);

        if ($aid != null) {
            $deeplinkData['parameters'] = $parameters . '&aid=' . $aid;
        } else {
            $deeplinkData['parameters'] = $parameters . '&aidgiata=' . $aidgiata;
        }

        $deeplinkData['deeplink'] = ($deeplinkData['prefix'] . $deeplinkData['parameters']);

        return $deeplinkData;
    }
}
