<?php

namespace TraveltainmentApi\WebService;

use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\Traveller;
use Vtours\TTXmlWebService\Model\Response\Hotel;
use Vtours\TTXmlWebService\Model\Response\Meal;
use Vtours\TTXmlWebService\Model\Response\Offer;
use Vtours\TTXmlWebService\Model\Response\Region;
use Vtours\TTXmlWebService\Model\Response\SearchEngineHotelListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineRegionTreeRS;
use Vtours\TTXmlWebService\Model\Response\WeatherResponse\YearOverview;
use Vtours\TTXmlWebService\Service\TravelTainmentService;

class TraveltainmentApiService
{

    /**
     * @var TravelTainmentService $traveltainmentService
     */
    private $traveltainmentService;

    /**
     * @var array $offerData
     */
    public $offerData;


    public function __construct()
    {
        $this->traveltainmentService = new TravelTainmentService();
        $this->offerData['adult'] = 0;
        $this->offerData['child'] = [];
        $this->offerData['travellers'];
    }

    public function getOfferByHotelCode(
        array $travelDuration,
        array $travelPeriod,
        $mealType,
        array $travellers,
        array $pricing,
        array $hotelAttributes,
        array $specialJourneyAttributes,
        $brand,
        $departureAirport,
        $hotelCode,
        $hotelCodeType,
        $offerType
    ) {
        $travellerList = $this->prepareTravellerInformation($travellers);
        $hotelAttributeList = $this->prepareHotelAttributes($hotelAttributes);
        $specialJourneyAttributeList = $this->prepareSpecialJourneyAttributes($specialJourneyAttributes);
        /**
         * @var $searchEngineOfferListResponse SearchEngineOfferListRS
         */
        $searchEngineOfferListResponse = $this->traveltainmentService->getAllHotelOffersByHotelCode($travelDuration, $travelPeriod, $pricing,
            $hotelAttributeList, $mealType,
            $travellerList, $specialJourneyAttributeList, $brand, $departureAirport,
            $hotelCode, $hotelCodeType, $offerType);

        if ($searchEngineOfferListResponse->getInfoMessage() == null) {

            if ($searchEngineOfferListResponse->getResultSet()->getTotalNumberOfResults() != 0 && $searchEngineOfferListResponse->isSuccess()) {
                $this->offerData['request'] = $this->getRequiredRequestData($travelPeriod, $hotelAttributes, $specialJourneyAttributes,
                    $departureAirport, $brand, $pricing);
                $offer = $searchEngineOfferListResponse->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $searchEngineOfferListResponse->getHotel() ?: $searchEngineOfferListResponse->getResultSet()->getListItem()[0]->getHotel();
                $region = $searchEngineOfferListResponse->getRegion() ?: $searchEngineOfferListResponse->getResultSet()->getListItem()[0]->getRegion();
                $this->createObjectData($offer, $hotel, $region, $offerType, 'HotelCode');

                return $this->offerData;
            } else {
                return [];
            }
        } else {
            //infomessage is only set, if tt returns an error
            //maybe add info for pimcore-user
            return [];
        }
    }

    public function prepareTravellerInformation(array $travellers)
    {
        $travellerList = [];
        $this->offerData['travellers'] = count($travellers);

        foreach ($travellers as $traveller) {
            if ($traveller != null || $traveller != 0) {
                $travellerList[] = new Traveller($traveller);
                if ($traveller > 16) {
                    $this->offerData['adult'] = $this->offerData['adult'] + 1;

                } else {
                    if ($traveller <= 16) {
                        $this->offerData['child'][] = $traveller;
                    }
                }

            }
        }
        if (!empty(array_filter($this->offerData['child']))) {
            $this->offerData['child'] = implode(',', $this->offerData['child']);
        }

        return $travellerList;
    }

    public function prepareHotelAttributes(array $hotelAttributes)
    {
        $hotelAttributeList = [];
        if ($hotelAttributes['SANDYBEACH'] == 1) {
            $hotelAttributeList[] = 'SANDYBEACH';
        }
        if ($hotelAttributes['FAMILYFRIENDLY'] == 1) {
            $hotelAttributeList[] = 'FAMILYHOLIDAY';
            $hotelAttributeList[] = 'CHILDFRIENDLY';

        }
        if ($hotelAttributes['ADULTSONLY'] == 1) {
            $hotelAttributeList[] = 'ADULTSONLY';
        }

        return $hotelAttributeList;
    }

    public function prepareSpecialJourneyAttributes(array $specialJourneyAttributes)
    {
        $specialJourneyAttributeList = [];
        if ($specialJourneyAttributes['TRANSFER'] == 1) {
            $specialJourneyAttributeList[] = 'TRANSFER';
        }

        return $specialJourneyAttributeList;
    }

    public function getOfferByRegionCode(
        array $travelDuration,
        array $travelPeriod,
        $mealType,
        array $travellers,
        array $pricing,
        array $hotelAttributes,
        array $specialJourneyAttributes,
        $brand,
        $departureAirport,
        $regionCode,
        $offerType
    ) {

        $travellerList = $this->prepareTravellerInformation($travellers);
        $hotelAttributeList = $this->prepareHotelAttributes($hotelAttributes);
        $specialJourneyAttributeList = $this->prepareSpecialJourneyAttributes($specialJourneyAttributes);
        /**
         * @var $searchEngineHotelListResponse SearchEngineHotelListRS
         */
        $searchEngineHotelListResponse = $this->traveltainmentService->getBestHotelOffersByRegion($travelDuration, $travelPeriod, $pricing,
            $hotelAttributeList, $mealType,
            $travellerList, $specialJourneyAttributeList, $brand, $departureAirport, $regionCode, $offerType);

        if ($searchEngineHotelListResponse->getInfoMessage() == null) {

            if ($searchEngineHotelListResponse->getResultSet()->getTotalNumberOfResults() != 0 && $searchEngineHotelListResponse->isSuccess()) {

                $this->offerData['request'] = $this->getRequiredRequestData($travelPeriod, $hotelAttributes, $specialJourneyAttributes,
                    $departureAirport, $brand, $pricing);

                $offer = $searchEngineHotelListResponse->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $searchEngineHotelListResponse->getResultSet()->getListItem()[0]->getHotel();
                $region = $searchEngineHotelListResponse->getRegion() ? $searchEngineHotelListResponse->getRegion() : $searchEngineHotelListResponse->getResultSet()->getListItem()[0]->getRegion();

                $this->createObjectData($offer, $hotel, $region, $offerType, 'TopRegionId');

                return $this->offerData;
            } else {
                return [];
            }
        } else {
            //infomessage is only set, if tt returns an error
            //maybe add info for pimcore-user
            return [];
        }
    }


    public function getCompleteRegionTree()
    {
        /**
         * @var $regionTreeResponse SearchEngineRegionTreeRS
         */
        $regionTreeResponse = $this->traveltainmentService->getRegionTree();
        if ($regionTreeResponse->isSuccess() == true) {
            return $regionTreeResponse;
        } else {
            return new \Exception('no regions available from TT');
        }
    }


    public function createObjectData(
        Offer $offer,
        Hotel $hotel,
        Region $region,
        $offerType,
        $codeType
    ) {
        $this->offerData['hotel'] = $this->getHotelData($hotel);
        $this->offerData['region'] = $this->getRegionData($region);
        $this->offerData['offer'] = $this->getOfferData($offer, $offerType);
        $this->offerData['deeplink'] = $this->createDeeplink($offerType, $codeType);
    }

    public function getHotelData(Hotel $hotel)
    {
        if ($hotel != null) {
            return [
                'hotelName'    => $hotel->getObjectName(),
                'hotelCity'    => $hotel->getCity()->getCity(),
                'hotelStars'   => $hotel->getCategory()->getStars(),
                'hotelPicture' => $hotel->getObjectImages()->getLargeImage(),
                'hotelIff'     => $hotel->getObjectId(),
                'hotelGiata'   => $hotel->getAccomIds()[0]
            ];
        }
    }

    public function getRegionData(Region $region)
    {
        /** @var $clima YearOverview */
        $regionClima = '';
        if ($region->getWeatherForRegion() != null) {
            foreach ($region->getWeatherForRegion()->getClimaData()->getYearOverview() as $clima) {
                if ($clima->getClimaDataType() == 'MaxTemperature') {
                    $regionClima = $clima->getValueOfMonth()->getValueOfMonth();
                }
            }
        }

        if ($region != null) {
            return [
                'hotelCountry'  => $region->getCountry(),
                'regionClima'   => $regionClima,
                'topRegionId'   => $region->getTopRegionId(),
                'destinationId' => $region->getDestinationId()
            ];
        }
    }

    public function getOfferData(Offer $offer, $offerType)
    {
        $offerData = [];
        if ($offer != null) {
            $offerData = [
                'offerID'             => $offer->getOfferId(),
                'offerMinPrice'       => $offer->getPriceInformation()->getOriginalPrice()->getOriginalPrice(),
                'offerPriceCurrency'  => $offer->getPriceInformation()->getOriginalPrice()->getCurrency(),
                'offerWithFlight'     => false,
                'offerWithTransfer'   => false,
                'offerTravelDuration' => $offer->getTravelDuration()
            ];
            if ($offerType == 'Hotel') {
                /** @var Meal $meal */
                $meal = $offer->getAccommodation()->getMeal();
                $offerData['offerMealType'] = $meal->getMeal();
                $offerData['offerRoomType'] = $offer->getAccommodation()->getRoom()->getRoom();
            } else {
                if ($offerType == 'Package') {
                    $offerData['offerWithFlight'] = true;
                    $offerData['offerMealType'] = $offer->getPackage()->getAccommodation()->getMeal()->getMeal();
                    $offerData['offerRoomType'] = $offer->getPackage()->getAccommodation()->getRoom()->getRoom();
                    if ($offer->getPackage()->getFlight()) {
                        $offerData['offerDepartureAirport'] = $offer->getPackage()->getFlight()->getOutBoundFlightSegments()->getDepartureAirport();
                    }

                    $journeyAttributes = $offer->getSpecialJourneyAttributes();
                    if ($journeyAttributes != null && (strpos($journeyAttributes, 'TRANSFER')) !== false) {
                        $offerData['offerWithTransfer'] = true;
                    }
                }
            }
        }

        return $offerData;
    }

    public function getRequiredRequestData(
        array $travelPeriod,
        array $hotelAttributes,
        array $specialJourneyAttributes,
        $departureAirport,
        $brand,
        array $pricing
    ) {
        $objectData = [
            'adult'            => $this->offerData['adult'],
            'child'            => $this->offerData['child'],
            'departure'        => $travelPeriod[0],
            'return'           => $travelPeriod[1],
            'adultsonly'       => $hotelAttributes['ADULTSONLY'],
            'familyfriendly'   => $hotelAttributes['FAMILYFRIENDLY'],
            'beach'            => $hotelAttributes['SANDYBEACH'],
            'transfer'         => $specialJourneyAttributes['TRANSFER'],
            'departureAirport' => $departureAirport,
            'brand'            => $brand,
            'pricing'          => $pricing
        ];

        return $objectData;

    }

    public function createDeeplink($offerType, $codeType)
    {
        $deeplinkData = [];
        $prefix = '';
        $parameters = '';
        if ($offerType == 'Package') {
            $prefix = 'http://urlaub.vtours.de/';
            $parameters = $parameters . 'ibe=package';

            //departure-airport
            $depap = null;
            $depapReq = $this->offerData['request']['departureAirport'];
            $depapOffer = $this->offerData['offer']['offerDepartureAirport'];
            if ($depapOffer != null && $depapReq == null) {
                $depap = $depapOffer->getIataCode();
            }
            if ($depapReq != null && $depapOffer == null) {
                $depap = $depapReq;
            } elseif ($depapOffer != null && $depapReq != null) {
                $depap = $depapReq;
            }
            if ($depap != null) {
                $parameters = $parameters . '&depap=' . $depap;
            }

        } else {
            if ($offerType == 'Hotel') {
                $prefix = 'http://hotel.vtours.de/';
                $parameters = $parameters . 'ibe=hotel';

                //hotelstars
                $stars = $this->offerData['hotel']['hotelStars'];
                $parameters = $parameters . '&stars=' . $stars;
            }
        }
        if ($codeType == 'TopRegionId') {
            $prefix = $prefix . 'hotel?';
        } else {
            $prefix = $prefix . 'offer?';
            //hotelcode
            $aid = $this->offerData['hotel']['hotelIff'];
            $aidgiata = $this->offerData['hotel']['hotelGiata'];
            if ($aid != null) {
                $parameters = $parameters . '&aid=' . $aid;
            } else {
                $parameters = $parameters . '&aidgiata=' . $aidgiata;
            }
            $cyid = $this->offerData['region']['destinationId'];
            $parameters = $parameters . '&cyid=' . $cyid;
        }

        //traveldates and -duration
        $ddate = $this->offerData['request']['departure'];
        $ddate = $ddate->format('Y-m-d');
        $parameters = $parameters . '&ddate=' . $ddate;

        $rdate = $this->offerData['request']['return'];
        $rdate = $rdate->format('Y-m-d');
        $parameters = $parameters . '&rdate=' . $rdate;

        $dur = $this->offerData['offer']['offerTravelDuration'];
        $parameters = $parameters . '&dur=' . $dur;


        //travellers
        $adult = $this->offerData['request']['adult'];
        $child = $this->offerData['request']['child'];
        if ($adult != null || $adult != 0) {
            $parameters = $parameters . '&adult=' . $adult;
        }
        if ($child != null || !empty(array_filter($child))) {
            $parameters = $parameters . '&child=' . $child;
        }
        //region
        $rid = $this->offerData['region']['topRegionId'];
        $parameters = $parameters . '&rid=' . $rid;

        //hotelattributes
        if ($this->offerData['request']['adultsonly'] == 1) {
            $parameters = $parameters . '&trgrp=7';
        }
        if ($this->offerData['request']['familyfriendly'] == 1) {
            $parameters = $parameters . '&hotat=27';
        }
        if ($this->offerData['request']['beach'] == 1) {
            $parameters = $parameters . '&beach=2';
        }

        $priceMin = $this->offerData['request']['pricing'][0];
        $priceMax = $this->offerData['request']['pricing'][1];

        if ($priceMin != null && $priceMax != null) {
            $parameters = $parameters . '&price=' . $priceMin . ',' . $priceMax;
        }
        if ($priceMin != null && $priceMax == null) {
            $parameters = $parameters . '&price=' . $priceMin;
        }
        if ($priceMin == null && $priceMax != null) {
            ;
            $parameters = $parameters . '&price=0,' . $priceMax;
        }

        //transfer
        $trans = $this->offerData['offer']['offerWithTransfer'];
        if ($trans == true) {
            $parameters = $parameters . '&trans=1';
        } else {
            $parameters = $parameters . '&trans=2';
        }

        //brand
        $brand = $this->offerData['request']['brand'];
        $parameters = $parameters . '&brand=' . $brand;

        $deeplinkData['prefix'] = $prefix;
        $deeplinkData['parameters'] = $parameters;
        $deeplinkData['deeplink'] = ($deeplinkData['prefix'] . $deeplinkData['parameters']);

        return $deeplinkData;
    }
}
