<?php

namespace TraveltainmentApi\Action;

use Carbon\Carbon;
use Pimcore\API\Plugin\Exception;
use Pimcore\Model\Element\ValidationException;
use Pimcore\Model\Object\VtoursDatenAngebot;
use TraveltainmentApi\Model\DatenAngebotModel;
use TraveltainmentApi\WebService\TraveltainmentApiService;
use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;

/**
 * Class UpdateOffer
 *
 * @package TraveltainmentApi\Action
 */
class UpdateOffer
{
    /**
     * @var TraveltainmentApiService
     */
    private $traveltainmentApiService;
    private $datenAngebotModel;

    /**
     * Update constructor.
     */
    public function __construct()
    {
        $this->traveltainmentApiService = new TraveltainmentApiService();
        $this->datenAngebotModel = new DatenAngebotModel();
    }

    /**
     * @param VtoursDatenAngebot $object
     *
     * @throws ValidationException
     */
    public function updateObject(VtoursDatenAngebot $object, $factory)
    {
        $requestModel = $this->mapRequestModel($object);

        try {
            if ($requestModel->getCodetype() != 'TopRegionID') {
                /**
                 * @var \Vtours\TTXmlWebService\Model\DatenAngebotResponseModel $response
                 */
                $response = $this->traveltainmentApiService->getOfferByHotelCode(
                    $requestModel,
                    $factory->offerList()
                );
            } else {
                /**
                 * @var \Vtours\TTXmlWebService\Model\DatenAngebotResponseModel $response
                 */
                $response = $this->traveltainmentApiService->getOfferByRegionCode(
                    $requestModel,
                    $factory->hotelList()
                );
            }
            if ($response != []) {
                $this->datenAngebotModel->mapObject($object, $response);
            } else {
                throw new ValidationException('no offers available for this request data', 400);
            }
        } catch (Exception $e) {
            throw new ValidationException($e->getMessage(), 400);
        }
    }

    /**
     * @param VtoursDatenAngebot $object
     *
     * @return DatenAngebotRequestModel
     * @throws ValidationException
     */
    public function mapRequestModel(VtoursDatenAngebot $object)
    {
        $requestModel = new DatenAngebotRequestModel();
        $requestModel = $this->mapRequestHotelAttributes($object, $requestModel);
        $requestModel = $this->mapRequestSpecialJounreyAttributes($object, $requestModel);

        $code = $object->getCode();
        $codeType = $object->getCodeType();
        $offerType = $object->getOfferType();

        $travelPeriodStart = $this->handleTravelDate($object);
        $object->setTravelPeriodStart($travelPeriodStart);

        $travelPeriodEnd = $object->getTravelPeriodEnd();

        $brand = $object->getBrand();

        $travelDurationMin = $object->getTravelDurationMin();
        $travelDurationMax = $object->getTravelDurationMax();

        $mealType = $object->getMealType();
        $roomType = $object->getRoomType();

        $departureAirport = $object->getDepartureAirportSelect();

        $requestModel->setCode($code)
            ->setCodetype($codeType)
            ->setOffertype($offerType)
            ->setTraveldateStart($travelPeriodStart)
            ->setTraveldateEnd($travelPeriodEnd)
            ->setBrand($brand)
            ->setTraveldurationMin($travelDurationMin)
            ->setTraveldurationMax($travelDurationMax)
            ->setMealtype($mealType)
            ->setRoomtype($roomType)
            ->setDepartureAirport($departureAirport);;

        $travellers = null;
        if ($object->getTravellers() != 0 || $object->getTravellers() != null) {
            foreach ($object->getTravellers() as $traveller) {
                $travellers[] = $traveller['age']->getData();
            }
            $requestModel->setTravellers($travellers);
        }
        if (empty($travellers)) {
            throw new ValidationException('missing mandytory field \'Reisende\'', 400);
        }

        $requestModel = $this->mapRequestPrice($object, $requestModel);

        return $requestModel;
    }

    /**
     * @param VtoursDatenAngebot $object
     *
     * @return Carbon
     */
    public function handleTravelDate(VtoursDatenAngebot $object)
    {
        $dateAfterToday = $object->getTravelPeriodStart()->greaterThanOrEqualTo(new Carbon(strtotime(time())));
        if (!$dateAfterToday) {
            return new Carbon(date("d.m.Y", strtotime("+1 days")));
        }

        return $object->getTravelPeriodStart();

    }

    /**
     * @param VtoursDatenAngebot       $object
     * @param DatenAngebotRequestModel $requestModel
     *
     * @return DatenAngebotRequestModel
     */
    public function mapRequestPrice(VtoursDatenAngebot $object, DatenAngebotRequestModel $requestModel)
    {

        $priceMin = $object->getPriceMin();
        if ($priceMin != null) {
            $priceMin = $priceMin->getValue();
            $requestModel->setPriceMin($priceMin);
        }
        $priceMax = $object->getPriceMax();
        if ($priceMax != null) {
            $priceMax = $priceMax->getValue();
            $requestModel->setPriceMax($priceMax);
        }

        return $requestModel;
    }

    /**
     * @param VtoursDatenAngebot       $object
     * @param DatenAngebotRequestModel $requestModel
     *
     * @return DatenAngebotRequestModel
     */
    public function mapRequestHotelAttributes(VtoursDatenAngebot $object, DatenAngebotRequestModel $requestModel)
    {
        $hotelAttributes = [];
        if ($object->getIncludeSandybeach() != null) {
            $hotelAttributes[] = 'SANDYBEACH';
        }
        if ($object->getIncludeAdultsonly() != null) {
            $hotelAttributes[] = 'ADULTSONLY';
        }
        if ($object->getIncludeFamilyfriendly() != null) {
            $hotelAttributeList[] = 'FAMILYHOLIDAY';
            $hotelAttributeList[] = 'CHILDFRIENDLY';
        }
        $requestModel->setHotelAttributes($hotelAttributes);

        return $requestModel;
    }

    /**
     * @param VtoursDatenAngebot       $object
     * @param DatenAngebotRequestModel $requestModel
     *
     * @return DatenAngebotRequestModel
     */
    public function mapRequestSpecialJounreyAttributes(VtoursDatenAngebot $object, DatenAngebotRequestModel $requestModel)
    {
        $specialJourneyAttributes = [];
        if ($object->getIncludeTransfer() != null) {
            $specialJourneyAttributes[] = 'TRANSFER';
        }
        $requestModel->setSpecialJourneyAttributes($specialJourneyAttributes);

        return $requestModel;
    }
}
