<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 */

namespace TraveltainmentApi\Helper;

use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\DatenAngebotResponseModel;

/**
 * Class TraveltainmentDeeplinkHelper
 *
 * @package TraveltainmentApi\Helper
 */
class TraveltainmentDeeplinkHelper
{
    const AGE_OF_ADULT = 16;

    /**
     * create deeplink
     *
     * @param DatenAngebotResponseModel $response
     * @param DatenAngebotRequestModel  $request
     * @param                           $offerType
     * @param                           $codeType
     *
     * @return DatenAngebotResponseModel
     */
    public function createDeeplink(
        DatenAngebotResponseModel $response,
        DatenAngebotRequestModel $request,
        $offerType,
        $codeType
    ) {
        $prefix = '//booking.vtours.com/';
        $parameters = '';
        if ($offerType == 'Package') {
            $parameters = $parameters . 'ibe=package';

            //departure-airport
            $depap = null;
            $depapReq = $request->getDepartureAirport();
            $depapOffer = $response->getOfferDepartureAirport();
            if ($depapOffer != null && $depapReq == null) {
                $depap = $depapOffer;
            }

            if ($depapReq != null && $depapOffer == null) {
                $depap = $depapReq;
            } elseif ($depapOffer != null && $depapReq != null) {
                $depap = $depapReq;
            }
            if ($depap != null) {
                $parameters = $parameters . '&depap=' . $depap;
            }
        } else {
            if ($offerType == 'Hotel') {
                $parameters = $parameters . 'ibe=hotel';

                //hotelstars
                $stars = $response->getHotelStars();
                $parameters = $parameters . '&stars=' . $stars;
            }
        }
        if ($codeType == 'TopRegionId') {
            $prefix = $prefix . 'hotel?';
        } else {
            $prefix = $prefix . 'offer?';
            //hotelcode
            $aid = $response->getHotelIff();
            $aidgiata = $response->getHotelGiata();
            if ($aid != null) {
                $parameters = $parameters . '&aid=' . $aid;
            } else {
                $parameters = $parameters . '&aidgiata=' . $aidgiata;
            }
            $cyid = $response->getDestinationId();
            $parameters = $parameters . '&cyid=' . $cyid;
        }

        //traveldates and -duration
        $ddate = $request->getTraveldateStart();
        $ddate = $ddate->format('Y-m-d');
        $parameters = $parameters . '&ddate=' . $ddate;

        $rdate = $request->getTraveldateEnd();
        $rdate = $rdate->format('Y-m-d');
        $parameters = $parameters . '&rdate=' . $rdate;

        $dur = $response->getOfferTravelDuration();
        $parameters = $parameters . '&dur=' . $dur;


        //travellers
        $travellerList = $this->prepareTravellerList($request->getTravellers());
        $adult = $travellerList['adult'];
        $child = $travellerList['child'];
        if ($adult != null || $adult != 0) {
            $parameters = $parameters . '&adult=' . $adult;
        }
        if ($child != null || !empty(array_filter($child))) {
            $parameters = $parameters . '&child=' . $child;
        }
        //region
        $rid = $response->getHotelRegionId();
        $parameters = $parameters . '&rid=' . $rid;

        //hotelattributes
        if (in_array('ADULTSONLY', $request->getHotelAttributes())) {
            $parameters = $parameters . '&trgrp=7';
        }
        if ((in_array('FAMILYHOLIDAY', $request->getHotelAttributes())) || (in_array('CHILDFRIENDLY', $request->getHotelAttributes()))) {
            $parameters = $parameters . '&hotat=27';
        }

        if (in_array('SANDYBEACH', $request->getHotelAttributes())) {
            $parameters = $parameters . '&beach=2';
        }

        $priceMin = $request->getPriceMin();
        $priceMax = $request->getPriceMax();

        if ($priceMin != null && $priceMax != null) {
            $parameters = $parameters . '&price=' . $priceMin . ',' . $priceMax;
        }
        if ($priceMin != null && $priceMax == null) {
            $parameters = $parameters . '&price=' . $priceMin;
        }
        if ($priceMin == null && $priceMax != null) {
            ;
            $parameters = $parameters . '&price=0,' . $priceMax;
        }

        //meal
        $meal = $request->getMealtype();
        $meal = $this->handleMealTypeForDeeplink($meal);
        $parameters = $parameters . '&board=' . $meal;

        //room
        $room = $request->getRoomtype();
        $room = $this->handleRoomTypeForDeeplink($room);
        $parameters = $parameters . '&room=' . $room;

        //transfer
        $trans = $response->isOfferWithTransfer();
        if ($trans == true) {
            $parameters = $parameters . '&trans=1';
        } else {
            $parameters = $parameters . '&trans=2';
        }

        //brand
        $brand = $request->getBrand();
        $parameters = $parameters . '&brand=' . $brand;

        $response->setDeeplinkPrefix($prefix);
        $response->setDeeplinkParameter($parameters);

        return $response;
    }

    /**
     * @param array $travellers
     *
     * @return array
     */
    public function prepareTravellerList(array $travellers)
    {
        $adult = 0;
        $child = [];

        foreach ($travellers as $traveller) {
            if ($traveller != null || $traveller != 0) {
                if ($traveller > self::AGE_OF_ADULT) {
                    $adult++;
                } else {
                    if ($traveller <= self::AGE_OF_ADULT) {
                        $child[] = $traveller;
                    }
                }
            }
        }
        if (!empty(array_filter($child))) {
            $child = implode(',', $child);
        }
        $travellerList = [
            'adult' => $adult,
            'child' => $child
        ];

        return $travellerList;
    }

    /**
     * handleMealTypeForDeeplink
     *
     * @param $meal
     *
     * @return string
     */
    public function handleMealTypeForDeeplink($meal)
    {
        switch ($meal) {
            case ('ONLYSTAY'):
                return '1';
                break;
            case ('BREAKFAST'):
                return '2';
                break;
            case ('HALFBOARD'):
                return '3';
                break;
            case ('FULLBOARD'):
                return '4';
                break;
            case ('ALLINCLUSIVE'):
                return '5';
                break;
            case ('HALFBOARDPLUS'):
                return '6';
                break;
            case ('FULLBOARDPLUS'):
                return '7';
                break;
            case ('ALLINCLUSIVEPLUS'):
                return '8';
                break;
            case ('SPECIALBOARD'):
                return '9';
                break;
            default:
                return '0';
        }
    }

    /**
     * handleRoomTypeForDeeplink
     *
     * @param $room
     *
     * @return string
     */
    public function handleRoomTypeForDeeplink($room)
    {
        switch ($room) {
            case ('APPARTEMENT'):
                return '5';
                break;
            case ('BUNGALOW'):
                return '4';
                break;
            case ('DELUXEBEDROOM'):
                return '14';
                break;
            case ('DOUBLEBEDROOM'):
                return '2';
                break;
            case ('DUPLEXBEDROOM'):
                return '15';
                break;
            case ('FAMILYROOM'):
                return '3';
                break;
            case ('HOLIDAYFLAT'):
                return '12';
                break;
            case ('HOLIDAYHOUSE'):
                return '13';
                break;
            case ('MULTISHARERROOM'):
                return '11';
                break;
            case ('SINGLEBEDROOM'):
                return '1';
                break;
            case ('STUDIO'):
                return '6';
                break;
            case ('SUITE'):
                return '7';
                break;
            case ('SUPERIOR'):
                return '9';
                break;
            case ('TRIPLEBEDROOM'):
                return '10';
                break;
            case ('VILLA'):
                return '8';
                break;
            default:
                return '0';
        }
    }
}
