<?php

namespace TraveltainmentApi\WebService;

use TraveltainmentApi\Helper\TraveltainmentDeeplinkHelper;
use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\DatenAngebotResponseModel;
use Vtours\TTXmlWebService\Model\Response\Hotel;
use Vtours\TTXmlWebService\Model\Response\Meal;
use Vtours\TTXmlWebService\Model\Response\Offer;
use Vtours\TTXmlWebService\Model\Response\Region;
use Vtours\TTXmlWebService\Model\Response\SearchEngineHotelListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineRegionTreeRS;
use Vtours\TTXmlWebService\Model\Response\WeatherResponse\YearOverview;
use Vtours\TTXmlWebService\Service\HotelList\Handler as HotelListHandler;
use Vtours\TTXmlWebService\Service\OfferList\Handler as OfferListHandler;
use Vtours\TTXmlWebService\Service\RegionTree\Handler as RegionTreeHandler;

/**
 * Class TraveltainmentApiService
 *
 * @package TraveltainmentApi\WebService
 */
class TraveltainmentApiService
{

    /**
     * @var TraveltainmentDeeplinkHelper $deeplinkHelper
     */
    private $deeplinkHelper;
    /**
     * @var array $offerData
     */
    public $offerData;


    /**
     * TraveltainmentApiService constructor.
     */
    public function __construct()
    {
        $this->deeplinkHelper = new TraveltainmentDeeplinkHelper();
    }

    /**
     * @param DatenAngebotRequestModel $request
     * @param OfferListHandler         $offerListHandler
     *
     * @return array|\Vtours\TTXmlWebService\Model\DatenAngebotResponseModel|SearchEngineOfferListRS
     */
    public function getOfferByHotelCode(DatenAngebotRequestModel $request, OfferListHandler $offerListHandler)
    {
        /**
         * @var SearchEngineOfferListRS $response
         */
        $offerList = $offerListHandler->getAllHotelOffersByHotelCode($request);
        if ($offerList->getInfoMessage() == null) {
            if ($offerList->getResultSet()->getTotalNumberOfResults() != 0 && $offerList->isSuccess()) {
                $offer = $offerList->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $offerList->getHotel() ?: $offerList->getResultSet()->getListItem()[0]->getHotel();
                $region = $offerList->getRegion() ?: $offerList->getResultSet()->getListItem()[0]->getRegion();
                $responseModel = $this->createObjectData($offer, $hotel, $region, $request);

                return $responseModel;
            } else {
                return [];
            }
        } else {
            return [];
        }
    }


    /**
     * @param DatenAngebotRequestModel $requestModel
     * @param HotelListHandler         $hotelListHandler
     *
     * @return array|DatenAngebotResponseModel|SearchEngineHotelListRS
     */
    public function getOfferByRegionCode(DatenAngebotRequestModel $requestModel, HotelListHandler $hotelListHandler)
    {
        /**
         * @var SearchEngineHotelListRS $response
         */
        $hotelList = $hotelListHandler->getBestOffersByRegion($requestModel);
        if ($hotelList->getInfoMessage() == null) {
            if ($hotelList->getResultSet()->getTotalNumberOfResults() != 0 && $hotelList->isSuccess()) {
                $offer = $hotelList->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $hotelList->getResultSet()->getListItem()[0]->getHotel();
                $region = $hotelList->getRegion();
                $response = $this->createObjectData($offer, $hotel, $region, $requestModel);

                return $response;
            } else {
                return [];
            }
        } else {
            return [];
        }
    }

    /**
     * @param RegionTreeHandler $regionTreeHandler
     *
     * @return \Exception|SearchEngineRegionTreeRS
     */
    public function getCompleteRegionTree(RegionTreeHandler $regionTreeHandler)
    {
        /**
         * @var $regionTreeResponse SearchEngineRegionTreeRS
         */
        $regionTreeResponse = $regionTreeHandler->getRegionTree();
        if ($regionTreeResponse != null && $regionTreeResponse->isSuccess() == true) {
            return $regionTreeResponse;
        } else {
            return new \Exception('no regions available from TT');
        }
    }

    /**
     * @param Offer                    $offer
     * @param Hotel                    $hotel
     * @param Region                   $region
     * @param DatenAngebotRequestModel $datenAngebotRequestModel
     *
     * @return DatenAngebotResponseModel
     */
    public function createObjectData(
        Offer $offer,
        Hotel $hotel,
        Region $region,
        DatenAngebotRequestModel $datenAngebotRequestModel
    ) {
        $datenAngebotsResponseModel = new DatenAngebotResponseModel();
        if ($hotel != null) {
            $datenAngebotsResponseModel = $this->mapHotelData($datenAngebotsResponseModel, $hotel);
        }
        if ($region != null) {
            $datenAngebotsResponseModel = $this->mapRegionData($datenAngebotsResponseModel, $region);
        }
        if ($offer != null) {
            $datenAngebotsResponseModel = $this->mapOfferData($datenAngebotsResponseModel, $offer, $datenAngebotRequestModel->getOffertype());
        }
        $datenAngebotsResponseModel = $this->deeplinkHelper->createDeeplink(
            $datenAngebotsResponseModel,
            $datenAngebotRequestModel,
            $datenAngebotRequestModel->getOffertype(),
            $datenAngebotRequestModel->getCodetype()
        );

        return $datenAngebotsResponseModel;
    }

    /**
     * @param DatenAngebotResponseModel $datenAngebotsResponseModel
     * @param Hotel                     $hotel
     *
     * @return DatenAngebotResponseModel
     */
    public function mapHotelData(
        DatenAngebotResponseModel $datenAngebotsResponseModel,
        Hotel $hotel
    ) {
        $datenAngebotsResponseModel->setHotelName($hotel->getObjectName())
            ->setHotelRegion($hotel->getCity()->getCity())
            ->setHotelCityId($hotel->getCity()->getCityId())
            ->setHotelStars($hotel->getCategory()->getStars())
            ->setHotelPictureUrl($hotel->getObjectImages()->getLargeImage())
            ->setHotelIff($hotel->getObjectId())
            ->setHotelPictureUrl($hotel->getObjectImages()->getLargeImage())
            ->setHotelGiata($hotel->getAccomIds()[0]);

        return $datenAngebotsResponseModel;
    }

    /**
     * @param DatenAngebotResponseModel $datenAngebotsResponseModel
     * @param Region                    $region
     *
     * @return DatenAngebotResponseModel
     */
    public function mapRegionData(
        DatenAngebotResponseModel $datenAngebotsResponseModel,
        Region $region
    ) {
        /** @var $clima YearOverview */
        $regionClima = '';
        if ($region->getWeatherForRegion() != null) {
            foreach ($region->getWeatherForRegion()->getClimaData()->getYearOverview() as $clima) {
                if ($clima->getClimaDataType() == 'MaxTemperature') {
                    $regionClima = $clima->getValueOfMonth()->getValueOfMonth();
                }
            }
        }
        $datenAngebotsResponseModel->setHotelCountry($region->getCountry())
            ->setRegionTemperature($regionClima)
            ->setHotelRegionId($region->getTopRegionId())
            ->setDestinationId($region->getDestinationId());

        return $datenAngebotsResponseModel;
    }


    /**
     * @param DatenAngebotResponseModel $datenAngebotResponseModel
     * @param Offer                     $offer
     * @param                           $offerType
     *
     * @return DatenAngebotResponseModel
     */
    public function mapOfferData(
        DatenAngebotResponseModel $datenAngebotResponseModel,
        Offer $offer,
        $offerType
    ) {
        $datenAngebotResponseModel->setOfferId($offer->getOfferId())
            ->setOfferPrice($offer->getPriceInformation()->getOriginalPrice()->getOriginalPrice())
            ->setOfferPriceCurrency($offer->getPriceInformation()->getOriginalPrice()->getCurrency())
            ->setOfferWithFlight(false)
            ->setOfferWithTransfer(false)
            ->setOfferTravelDuration($offer->getTravelDuration());

        if ($offerType == 'Hotel') {
            /** @var Meal $meal */
            $meal = $offer->getAccommodation()->getMeal();
            $datenAngebotResponseModel->setOfferMealType($meal->getMeal())
                ->setOfferRoomType($offer->getAccommodation()->getRoom()->getRoom());
        } else {
            if ($offerType == 'Package') {
                $datenAngebotResponseModel->setOfferWithFlight(true)
                    ->setOfferMealType($offer->getPackage()->getAccommodation()->getMeal()->getMeal())
                    ->setOfferRoomType($offer->getPackage()->getAccommodation()->getRoom()->getRoom());
                if ($offer->getPackage()->getFlight()) {
                    $datenAngebotResponseModel->setOfferDepartureAirport($offer->getPackage()->getFlight()->getOutBoundFlightSegments()->getDepartureAirport()->getIataCode());
                }

                $journeyAttributes = $offer->getSpecialJourneyAttributes();
                if ($journeyAttributes != null && (strpos($journeyAttributes, 'TRANSFER')) !== false) {
                    $datenAngebotResponseModel->setOfferWithTransfer(true);
                }
            }
        }

        return $datenAngebotResponseModel;
    }
}
