<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 */


namespace TraveltainmentApi\Model;

use Pimcore\File;
use Pimcore\Model\Object\VtoursDatenRegion;
use Pimcore\Model\Object\Service;
use Vtours\TTXmlWebService\Model\Response\SubRegion;
use Vtours\TTXmlWebService\Model\Response\V2\RegionGroup;

class DatenRegionModel
{

    /**
     * gets subregions from region
     *
     * @param SubRegion $subRegions
     * @param           $parent
     */
    public function getSubRegions(array $subRegions, $parent)
    {
        foreach ($subRegions as $subRegion) {
            $this->createRegion($subRegion, $parent);
        }
    }

    /**
     * gets subregions from region
     *
     * @param RegionGroup $subRegions
     * @param           $parent
     */
    public function getSubRegionsV2($subRegions, $parent)
    {
        $regions = [];
        foreach ($subRegions as $key => $region) {
            $regions[$region->getRegionGroupId()]['name'] = $region->getName();

            foreach ($region->getRegion() as $subregions) {
                $regions[$region->getRegionGroupId()]['region'][$subregions->getRegionId()] = $subregions->getName();
            }
        }

        $this->createRegionV2($regions, $parent);
    }

    /**
     * creates region-object from tt-data
     *
     * @param array $regions
     * @param VtoursDatenRegion|null $parent
     */
    public function createRegionV2($regions, $parent)
    {

        foreach ($regions as $regionId => $region) {
            if ($parent != null) {
                $parentPath = $parent->getFullPath();
            }

            $key = File::getValidFilename($region['name']);
            $savedRegion = VtoursDatenRegion::getByPath('/vtours/daten/regionen-iff-codes/' . $parentPath != '' ? $parentPath . '/' . $key : $key);

            // first check main Groups
            if ($savedRegion === null) {
                $newRegionObject = new VtoursDatenRegion();
                $newRegionObject->setRegionIFFCode($regionId);
                $newRegionObject->setRegionName($region['name']);
                $newRegionObject->setParent($parent);

                $newRegionObject->setKey($key);
                $newRegionObject->setPublished(true);

                $newRegionObject->save();
            } else {
                // map Regions in Groups
                foreach ($region['region'] as $subRegionId => $subRegionName) {

                    $filteredSubRegionName = File::getValidFilename($subRegionName);
                    if (false === Service::pathExists($parent->getFullPath() . '/' . $key . '/' . $filteredSubRegionName)) {
                        $newRegionObject = new VtoursDatenRegion();
                        $newRegionObject->setRegionIFFCode($subRegionId);
                        $newRegionObject->setRegionName($subRegionName);
                        $newRegionObject->setParent($savedRegion);

                        $newRegionObject->setKey($filteredSubRegionName);
                        $newRegionObject->setPublished(true);

                        $newRegionObject->save();
                    }
                }
                $savedRegion->setPublished(true);
                $savedRegion->save();
            }
        }
    }

    /**
     * creates region-object from tt-data
     *
     * @param SubRegion $subRegion
     * @param VtoursDatenRegion|null $parent
     */
    public function createRegion($subRegion, $parent)
    {
        $key = File::getValidFilename($subRegion->getRegionName());
        $parentPath = '';
        if ($parent != null) {
            $parentPath = $parent->getFullPath();
        }
        /**
         * @var VtoursDatenRegion $savedRegion
         */
        $savedRegion = VtoursDatenRegion::getByPath('/vtours/daten/regionen-iff-codes/' . $parentPath != '' ? $parentPath . '/' . $key : $key);

        if ($savedRegion == null) {
            $newRegionObject = new VtoursDatenRegion();
            $newRegionObject->setRegionIFFCode($subRegion->getId());
            $newRegionObject->setRegionName($subRegion->getRegionName());
            $newRegionObject->setParent($parent);

            $newRegionObject->setKey($key);
            $newRegionObject->setPublished(true);

            $savedRegion = $newRegionObject->save();
        } else {
            if (!$this->checkValuesOfRegion($savedRegion, $subRegion)) {
                $savedRegion = $this->overwriteExistingRegion($savedRegion, $subRegion);
            }
            $savedRegion->setPublished($savedRegion->isPublished());
            $savedRegion->saveVersion();
        }
        if (!empty($subRegion->getSubRegions())) {
            $this->getSubRegions($subRegion->getSubRegions(), $savedRegion);
        }
    }

    /**
     * checks if values from incoming version are equal to existing region
     *
     * @param VtoursDatenRegion $savedRegionObject
     * @param SubRegion $subRegion
     *
     * @return bool
     */
    public function checkValuesOfRegion(VtoursDatenRegion $savedRegionObject, $subRegion)
    {
        if ($savedRegionObject->getRegionName() == $subRegion->getRegionName()
            && $savedRegionObject->getRegionIFFCode() == $subRegion->getId()
        ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * refreshs values of tt-regions
     *
     * @param VtoursDatenRegion $savedRegionObject
     * @param SubRegion $subRegion
     *
     * @return VtoursDatenRegion
     */
    public function overwriteExistingRegion(VtoursDatenRegion $savedRegionObject, $subRegion)
    {
        $savedRegionObject->setRegionIFFCode($subRegion->getId());
        $savedRegionObject->setRegionName($subRegion->getRegionName());

        return $savedRegionObject;
    }
}
