<?php

namespace TraveltainmentApi\WebService;

use GuzzleHttp\Exception\GuzzleException;
use TraveltainmentApi\Helper\TraveltainmentDeeplinkHelper;
use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\DatenAngebotResponseModel;
use Vtours\TTXmlWebService\Model\Response\Accommodation;
use Vtours\TTXmlWebService\Model\Response\Hotel;
use Vtours\TTXmlWebService\Model\Response\Meal;
use Vtours\TTXmlWebService\Model\Response\Offer;
use Vtours\TTXmlWebService\Model\Response\Region;
use Vtours\TTXmlWebService\Model\Response\SearchEngineHotelListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineRegionTreeRS;
use Vtours\TTXmlWebService\Model\Response\v2\AccommodationOffersRS;
use Vtours\TTXmlWebService\Model\Response\v2\PackageOffersRS;
use Vtours\TTXmlWebService\Model\Response\WeatherResponse\YearOverview;
use Vtours\TTXmlWebService\Service\HotelList\Handler as HotelListHandler;
use Vtours\TTXmlWebService\Service\OfferList\Handler as OfferListHandler;
use Vtours\TTXmlWebService\Service\RegionTree\Handler as RegionTreeHandler;

/**
 * Class TraveltainmentApiService
 *
 * @package TraveltainmentApi\WebService
 */
class TraveltainmentApiService
{
    const OFFER_TYPE_HOTEL = 'Hotel';
    const OFFER_TYPE_PACKAGE = 'Package';

    /**
     * @var TraveltainmentDeeplinkHelper $deeplinkHelper
     */
    private $deeplinkHelper;

    /**
     * @var array $offerData
     */
    public $offerData;

    /**
     * TraveltainmentApiService constructor.
     */
    public function __construct()
    {
        $this->deeplinkHelper = new TraveltainmentDeeplinkHelper();
    }

    /**
     * @param DatenAngebotRequestModel $request
     * @param OfferListHandler $offerListHandler
     * @param integer $apiVersion
     *
     * @return DatenAngebotResponseModel|array[]
     * @throws GuzzleException
     */
    public function getOfferByHotelCode(DatenAngebotRequestModel $request, OfferListHandler $offerListHandler, $apiVersion)
    {
        if($apiVersion !== 1) {
            /**
             * @var PackageOffersRS|AccommodationOffersRS $offerList
             */
            $offerList = $offerListHandler->getAllHotelOffersByHotelCodeV2($request);

            if($this->getOfferCount($offerList) === 0)
            {
                $logger = \Pimcore\Log\ApplicationLogger::getInstance("TraveltainmentApiService", true);
                $logger->warning('INFO: Resultset is empty (Results=' . $this->getOfferCount($offerList)
                    . ') or');

                return [];
            }

            return $this->createObjectDataV2(($request->getOffertype() === 'Package') ?
                $offerList->getPackageOffersRS() :
                $offerList->getAccommodationOffersRS(), $request);
        }


        $offerList = $offerListHandler->getAllHotelOffersByHotelCode($request);
        if ($offerList->getInfoMessage() == null) {
            if ($offerList->isSuccess() && $offerList->getResultSet()->getTotalNumberOfResults() != 0 ) {
                $offer = $offerList->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $offerList->getHotel() ?: $offerList->getResultSet()->getListItem()[0]->getHotel();
                $region = $offerList->getRegion() ?: $offerList->getResultSet()->getListItem()[0]->getRegion();

                return  $this->createObjectData($offer, $hotel, $region, $request);
            } else {
                try {
                    $logger = \Pimcore\Log\ApplicationLogger::getInstance("TraveltainmentApiService", true);
                    $logger->warning('INFO: Resultset is empty (Results=' . $offerList->getResultSet()->getTotalNumberOfResults()
                        . ') or unsuccessful (isSuccess=' . $offerList->isSuccess() . ')');
                } catch (\Exception $e) {
                }
                return [];
            }
        } else {
            try {
                $logger = \Pimcore\Log\ApplicationLogger::getInstance("TraveltainmentApiService", true);
                $logger->warning('INFO: Look at this Info Message:' . print_r($offerList->getInfoMessage(), true));
            } catch (\Exception $e) {
            }
            return [];
        }
    }


    /**
     * @param DatenAngebotRequestModel $requestModel
     * @param HotelListHandler         $hotelListHandler
     *
     * @return array|DatenAngebotResponseModel|SearchEngineHotelListRS
     */
    public function getOfferByRegionCode(DatenAngebotRequestModel $requestModel, HotelListHandler $hotelListHandler, $apiVersion)
    {
        if($apiVersion !== 1) {
            /**
             * @var PackageOffersRS|AccommodationOffersRS $offerList
             */
            $offerList = $hotelListHandler->getBestOffersByRegionV2($requestModel);

            if($this->getOfferCount($offerList) === 0)
            {
                $logger = \Pimcore\Log\ApplicationLogger::getInstance("TraveltainmentApiService", true);
                $logger->warning('INFO: Resultset is empty (Results=' . $offerList->getPackageOffersRS()->getOffers()->getOfferCount()
                    . ') or');

                return [];
            }
            return $this->createObjectDataV2(($requestModel->getOffertype() === 'Package') ?
                $offerList->getPackageOffersRS() :
                $offerList->getAccommodationOffersRS(), $requestModel);
        }


        /**
         * @var SearchEngineHotelListRS $response
         */
        $hotelList = $hotelListHandler->getBestOffersByRegion($requestModel);
        if ($hotelList->getInfoMessage() == null) {
            if ($hotelList->getResultSet()->getTotalNumberOfResults() != 0 && $hotelList->isSuccess()) {
                $offer = $hotelList->getResultSet()->getListItem()[0]->getOffer();
                $hotel = $hotelList->getResultSet()->getListItem()[0]->getHotel();
                $region = $hotelList->getRegion();
                return  $this->createObjectData($offer, $hotel, $region, $requestModel);

            } else {
                return [];
            }
        } else {
            return [];
        }
    }

    /**
     * @param RegionTreeHandler $regionTreeHandler
     *
     * @return \Exception|SearchEngineRegionTreeRS
     */
    public function getCompleteRegionTree(RegionTreeHandler $regionTreeHandler)
    {
        $regionTreeResponse = $regionTreeHandler->getRegionTree();
        if ($regionTreeResponse !== null && $regionTreeResponse->isSuccess() === true) {
            return $regionTreeResponse;
        } else {
            return new \Exception('no regions available from TT');
        }
    }

    /**
     * @param RegionTreeHandler $regionTreeHandler
     *
     * @return \Exception|SearchEngineRegionTreeRS
     */
    public function getCompleteRegionTreeV2(RegionTreeHandler $regionTreeHandler)
    {
        $regionTreeResponse = $regionTreeHandler->getRegionTreeV2();
        if ($regionTreeResponse !== null) {
            return $regionTreeResponse;
        }

        return new \Exception('no regions available from TT');
    }

    /**
     * @param Offer                    $offer
     * @param Hotel                    $hotel
     * @param Region                   $region
     * @param DatenAngebotRequestModel $datenAngebotRequestModel
     *
     * @return DatenAngebotResponseModel
     */
    public function createObjectData(
        Offer $offer,
        Hotel $hotel,
        Region $region,
        DatenAngebotRequestModel $datenAngebotRequestModel
    ) {
        $datenAngebotsResponseModel = new DatenAngebotResponseModel();
        if ($hotel != null) {
            $datenAngebotsResponseModel = $this->mapHotelData($datenAngebotsResponseModel, $hotel);
        }
        if ($region != null) {
            $datenAngebotsResponseModel = $this->mapRegionData($datenAngebotsResponseModel, $region);
        }
        if ($offer != null) {
            $datenAngebotsResponseModel = $this->mapOfferData($datenAngebotsResponseModel, $offer, $datenAngebotRequestModel->getOffertype());
        }

        return $this->deeplinkHelper->createDeeplink(
            $datenAngebotsResponseModel,
            $datenAngebotRequestModel,
            $datenAngebotRequestModel->getOffertype(),
            $datenAngebotRequestModel->getCodetype()
        );
    }


    /**
     * @param PackageOffersRS|AccommodationOffersRS $offerList
     * @param DatenAngebotRequestModel $request
     * @return DatenAngebotResponseModel
     */
    public function createObjectDataV2($offerList, DatenAngebotRequestModel $request)
    {
        $datenAngebotResponseModel = new DatenAngebotResponseModel();

        /**
         * @var \Vtours\TTXmlWebService\Model\Response\v2\HotelDictionary\Hotel $hotel
         */

        $hotel =  $offerList->getHotelDictionary()->getHotel()[0];
        $region = '';
        $regionGroupId = 0;
        $country = '';
        $temperature = 0;

        foreach($offerList->getGeoDictionary()->getCities()->getCity() as $city)
        {
            if($hotel->getHotelLocationRef()->getCityId() === $city->getCityId())
            {
                $regionGroupId = $hotel->getHotelLocationRef()->getRegionGroupId();
                $region = $city->getName();
                break;

            }
        }

        foreach($offerList->getGeoDictionary()->getCountries()->getCountry() as $country)
        {
            if($country->getCountryCode() === $hotel->getHotelLocationRef()->getCountryCode())
            {
                $country = $country->getName();
                break;
            }
        }

        //Map Hotel
        $datenAngebotResponseModel->setHotelName($hotel->getName())
            ->setHotelCityId($hotel->getHotelLocationRef()->getCityId())
            ->setHotelStars($hotel->getHotelCategory()) // category or meanRating
            ->setHotelPictureUrl($hotel->getHotelImages()->getLargeImage())
            ->setHotelIff($hotel->getHotelCodes()->getHotelIffCode())
            ->setHotelGiata($hotel->getHotelCodes()->getHotelGiataId());


        foreach($offerList->getGeoDictionary()->getRegions()->getRegion() as $regionList)
        {
            if($regionGroupId === $regionList->getRegionGroupId())
            {
                foreach($regionList->getRegionClimate()->getClimateParameter()[0]->getClimateParameterValue() as $climateValue)
                {
                    if($request->getTraveldateStart()->month === $climateValue->getMonth())
                    {
                        $temperature = (int) $climateValue->getValue();
                    }
                }
            }
        }

        //Map Region
        $datenAngebotResponseModel->setHotelCountry($country)
            ->setHotelRegion($region)
            ->setRegionTemperature($temperature)
            ->setHotelRegionId($hotel->getHotelLocationRef()->getRegionId())
            ->setDestinationId($hotel->getHotelLocationRef()->getCityId());

        //Map Offer

        /**
         * @var \Vtours\TTXmlWebService\Model\Response\v2\Offers\Offer $offer
         */
        $offer = $offerList->getOffers()->getOffer()[0];

        $datenAngebotResponseModel->setOfferId($offer->getOfferId())
            ->setOfferPrice($offer->getPriceInfo()->getPrice()->getValue())
            ->setOfferPriceCurrency($offer->getPriceInfo()->getPrice()->getCurrencyCode())
            ->setOfferWithFlight(false) // won't be available
            ->setOfferWithTransfer(false); // won't be available in hotel only
        if($request->getOffertype() === 'Package') {
            $datenAngebotResponseModel
                ->setOfferWithTransfer(null !== $offer->getOfferProperties()->getTransferInfos())
                ->setOfferRoomType(strtoupper($offer->getOfferServices()->getPackage()->getAccommodation()->getRoom()->getRoomType()))
                ->setOfferMealType(strtoupper($offer->getOfferServices()->getPackage()->getAccommodation()->getBoardType()));
        }
        $datenAngebotResponseModel->setOfferTravelDuration($offer->getTravelDateInfo()->getTripDuration());


        return $this->deeplinkHelper->createDeeplink($datenAngebotResponseModel, $request, $request->getOffertype(), $request->getCodetype());

    }

    /**
     * @param DatenAngebotResponseModel $datenAngebotsResponseModel
     * @param Hotel                     $hotel
     *
     * @return DatenAngebotResponseModel
     */
    public function mapHotelData(
        DatenAngebotResponseModel $datenAngebotsResponseModel,
        Hotel $hotel
    ) {
        $datenAngebotsResponseModel->setHotelName($hotel->getObjectName())
            ->setHotelRegion($hotel->getCity()->getCity())
            ->setHotelCityId($hotel->getCity()->getCityId())
            ->setHotelStars($hotel->getCategory()->getStars())
            ->setHotelPictureUrl($hotel->getObjectImages()->getLargeImage())
            ->setHotelIff($hotel->getObjectId())
            ->setHotelPictureUrl($hotel->getObjectImages()->getLargeImage())
            ->setHotelGiata($hotel->getAccomIds()[0]);

        return $datenAngebotsResponseModel;
    }

    /**
     * @param DatenAngebotResponseModel $datenAngebotsResponseModel
     * @param Region                    $region
     *
     * @return DatenAngebotResponseModel
     */
    public function mapRegionData(
        DatenAngebotResponseModel $datenAngebotsResponseModel,
        Region $region
    ) {
        /** @var $clima YearOverview */
        $regionClima = '';
        if ($region->getWeatherForRegion() != null) {
            foreach ($region->getWeatherForRegion()->getClimaData()->getYearOverview() as $clima) {
                if ($clima->getClimaDataType() == 'MaxTemperature') {
                    $regionClima = $clima->getValueOfMonth()->getValueOfMonth();
                }
            }
        }
        $datenAngebotsResponseModel->setHotelCountry($region->getCountry())
            ->setRegionTemperature($regionClima)
            ->setHotelRegionId($region->getTopRegionId())
            ->setDestinationId($region->getDestinationId());

        return $datenAngebotsResponseModel;
    }

    /**
     * @param DatenAngebotResponseModel $datenAngebotResponseModel
     * @param Offer                     $offer
     * @param                           $offerType
     *
     * @return DatenAngebotResponseModel
     */
    public function mapOfferData(
        DatenAngebotResponseModel $datenAngebotResponseModel,
        Offer $offer,
        $offerType
    ) {
        $datenAngebotResponseModel->setOfferId($offer->getOfferId())
            ->setOfferPrice($offer->getPriceInformation()->getOriginalPrice()->getOriginalPrice())
            ->setOfferPriceCurrency($offer->getPriceInformation()->getOriginalPrice()->getCurrency())
            ->setOfferWithFlight(false)
            ->setOfferWithTransfer(false)
            ->setOfferTravelDuration($offer->getTravelDuration());

        if (self::OFFER_TYPE_HOTEL === $offerType) {
            /** @var Meal $meal */
            $meal = $offer->getAccommodation()->getMeal();
            $datenAngebotResponseModel->setOfferMealType($meal->getMeal())
                ->setOfferRoomType($offer->getAccommodation()->getRoom()->getRoom());

            $this->overrideHotelStars($datenAngebotResponseModel, $offer->getAccommodation());
        } else {
            if (self::OFFER_TYPE_PACKAGE === $offerType) {
                $datenAngebotResponseModel->setOfferWithFlight(true)
                    ->setOfferMealType($offer->getPackage()->getAccommodation()->getMeal()->getMeal())
                    ->setOfferRoomType($offer->getPackage()->getAccommodation()->getRoom()->getRoom());
                if ($offer->getPackage()->getFlight()) {
                    $datenAngebotResponseModel->setOfferDepartureAirport($offer->getPackage()->getFlight()->getOutBoundFlightSegments()->getDepartureAirport()->getIataCode());
                }

                $journeyAttributes = $offer->getSpecialJourneyAttributes();
                if ($journeyAttributes != null && (strpos($journeyAttributes, 'TRANSFER')) !== false) {
                    $datenAngebotResponseModel->setOfferWithTransfer(true);
                }

                $this->overrideHotelStars($datenAngebotResponseModel, $offer->getPackage()->getAccommodation());
            }
        }

        return $datenAngebotResponseModel;
    }

    /**
     * @param DatenAngebotResponseModel $responseModel
     * @param Accommodation $accommodation
     */
    private function overrideHotelStars(DatenAngebotResponseModel $responseModel, Accommodation $accommodation)
    {
        $responseModel->setHotelStars($accommodation->getCategory());
    }

    private function getOfferCount($offerList)
    {
        return ($offerList instanceof PackageOffersRS)?
            $offerList->getPackageOffersRS()->getOffers()->getOfferCount() : $offerList->getAccommodationOffersRS()->getOffers()->getOfferCount();

    }
}
