<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 */

namespace TraveltainmentApi\Helper;

use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\DatenAngebotResponseModel;

/**
 * Class TraveltainmentDeeplinkHelper
 *
 * @package TraveltainmentApi\Helper
 */
class TraveltainmentDeeplinkHelper
{
    const AGE_OF_ADULT = 16;

    /**
     * create deeplink
     *
     * @param DatenAngebotResponseModel $response
     * @param DatenAngebotRequestModel  $request
     * @param                           $offerType
     * @param                           $codeType
     *
     * @return DatenAngebotResponseModel
     */
    public function createDeeplink(
        DatenAngebotResponseModel $response,
        DatenAngebotRequestModel $request,
        $offerType,
        $codeType
    ) {
        $prefix = '//booking.vtours.com/';
        $parameters = '';
        if ($offerType == 'Package') {
            $parameters = $parameters . 'ibe=package';

            //departure-airport
            $depap = null;
            $depapReq = $request->getDepartureAirport();
            $depapOffer = $response->getOfferDepartureAirport();
            if ($depapOffer != null && $depapReq == null) {
                $depap = $depapOffer;
            }

            if ($depapReq != null && $depapOffer == null) {
                $depap = $depapReq;
            } elseif ($depapOffer != null && $depapReq != null) {
                $depap = $depapReq;
            }
            if ($depap != null) {
                $parameters = $parameters . '&depap=' . $depap;
            }
        } else {
            if ($offerType == 'Hotel') {
                $parameters = $parameters . 'ibe=hotel';
            }
        }
        if ($codeType == 'TopRegionId') {
            $prefix = $prefix . 'hotel?';
        } else {
            $prefix = $prefix . 'offer?';
            //hotelcode
            $aid = $response->getHotelIff();
            $aidgiata = $response->getHotelGiata();
            if ($aid != null) {
                $parameters = $parameters . '&aid=' . $aid;
            } else {
                $parameters = $parameters . '&aidgiata=' . $aidgiata;
            }
            $cyid = $response->getHotelCityId();
            $parameters = $parameters . '&cyid=' . $cyid;
        }

        //traveldates and -duration
        $ddate = $request->getTraveldateStart();
        $ddate = $ddate->format('Y-m-d');
        $parameters = $parameters . '&ddate=' . $ddate;

        $rdate = $request->getTraveldateEnd();
        $rdate = $rdate->format('Y-m-d');
        $parameters = $parameters . '&rdate=' . $rdate;

        $durResp = $response->getOfferTravelDuration();
        $durReq = $request->getTraveldurationMin();
        if ($durReq != null) {
            $parameters = $parameters . '&dur=' . $durReq;
        } else {
            $parameters = $parameters . '&dur=' . $durResp;
        }

        //travellers
        $travellerList = $this->prepareTravellerList($request->getTravellers());
        $adult = $travellerList['adult'];
        $child = $travellerList['child'];
        if ($adult != null || $adult != 0) {
            $parameters = $parameters . '&adult=' . $adult;
        }
        if ($child != null || !empty(array_filter($child))) {
            $parameters = $parameters . '&child=' . $child;
        }
        //region
        $rid = $response->getHotelRegionId();
        $parameters = $parameters . '&rid=' . $rid;

        //country
        $nid = $this->countryIsoMapper($response->getHotelCountry());
        $parameters = $parameters . '&nid=' . $nid;

        //hotelattributes
        if (in_array('ADULTSONLY', $request->getHotelAttributes())) {
            $parameters = $parameters . '&trgrp=7';
        }
        if ((in_array('FAMILYHOLIDAY', $request->getHotelAttributes(), true))
            || (in_array('CHILDFRIENDLY', $request->getHotelAttributes(), true))) {
            $parameters = $parameters . '&hotat=27';
        }

        if (in_array('SANDYBEACH', $request->getHotelAttributes(), true)) {
            $parameters = $parameters . '&beach=2';
        }

        $priceMin = $request->getPriceMin();
        $priceMax = $request->getPriceMax();

        if ($priceMin != null && $priceMax != null) {
            $parameters .= '&price=' . $priceMin . ',' . $priceMax;
        }
        if ($priceMin != null && $priceMax == null) {
            $parameters .= '&price=' . $priceMin;
        }
        if ($priceMin == null && $priceMax != null) {
            $parameters .= '&price=0,' . $priceMax;
        }

        //meal
        $meal = (int)$this->handleMealTypeForDeeplink($request->getMealtype());
        $mealTT = (int)$this->handleMealTypeForDeeplink($response->getOfferMealType());
        if (!empty($mealTT) && $meal !== $mealTT) {
            $parameters .= ($meal > $mealTT) ? '&board=' . $mealTT.",".$meal : '&board=' . $meal.",".$mealTT ;
        } else {
            $parameters .=  '&board=' . $meal;
        }



        //room
        $room = (int)$this->handleRoomTypeForDeeplink($request->getRoomtype());
        $roomTT = (int)$this->handleRoomTypeForDeeplink($response->getOfferRoomType());
        if (!empty($roomTT) && $room !== $roomTT) {
            $parameters .= ($room > $roomTT) ? '&room=' . $roomTT.",".$room : '&room=' . $room.",".$roomTT ;
        } else {
            $parameters .=  '&room=' . $room;
        }


        //transfer
        $trans = $response->isOfferWithTransfer();
        if ($trans == true) {
            $parameters .= '&trans=1';
        } else {
            $parameters .= '&trans=2';
        }

        //brand
        $brand = $request->getBrand();
        $parameters .= '&brand=' . $brand;

        $response->setDeeplinkPrefix($prefix);
        $response->setDeeplinkParameter($parameters);

        return $response;
    }

    /**
     * @param array $travellers
     *
     * @return array
     */
    public function prepareTravellerList(array $travellers)
    {
        $adult = 0;
        $child = [];

        foreach ($travellers as $traveller) {
            if ($traveller != null || $traveller != 0) {
                if ($traveller > self::AGE_OF_ADULT) {
                    $adult++;
                } else {
                        $child[] = $traveller;
                }
            }
        }
        if (!empty(array_filter($child))) {
            $child = implode(',', $child);
        }
        return [
            'adult' => $adult,
            'child' => $child
        ];
    }

    /**
     * handleMealTypeForDeeplink
     *
     * @param $meal
     *
     * @return string
     */
    public function handleMealTypeForDeeplink($meal)
    {
        switch ($meal) {
            case ('ONLYSTAY'):
                return '1';
                break;
            case ('BREAKFAST'):
                return '2';
                break;
            case ('HALFBOARD'):
                return '3';
                break;
            case ('FULLBOARD'):
                return '4';
                break;
            case ('ALLINCLUSIVE'):
                return '5';
                break;
            case ('HALFBOARDPLUS'):
                return '6';
                break;
            case ('FULLBOARDPLUS'):
                return '7';
                break;
            case ('ALLINCLUSIVEPLUS'):
                return '8';
                break;
            case ('SPECIALBOARD'):
                return '9';
                break;
            default:
                return '0';
        }
    }

    /**
     * handleRoomTypeForDeeplink
     *
     * @param $room
     *
     * @return string
     */
    public function handleRoomTypeForDeeplink($room)
    {
        switch ($room) {
            case ('APPARTEMENT'):
                return '5';
                break;
            case ('BUNGALOW'):
                return '4';
                break;
            case ('DELUXEBEDROOM'):
                return '14';
                break;
            case ('DOUBLEBEDROOM'):
            case ('DOUBLEROOM'):
                return '2';
                break;
            case ('DUPLEXBEDROOM'):
                return '15';
                break;
            case ('FAMILYROOM'):
                return '3';
                break;
            case ('HOLIDAYFLAT'):
                return '12';
                break;
            case ('HOLIDAYHOUSE'):
                return '13';
                break;
            case ('MULTISHARERROOM'):
                return '11';
                break;
            case ('SINGLEBEDROOM'):
                return '1';
                break;
            case ('STUDIO'):
                return '6';
                break;
            case ('SUITE'):
                return '7';
                break;
            case ('SUPERIOR'):
                return '9';
                break;
            case ('TRIPLEBEDROOM'):
                return '10';
                break;
            case ('VILLA'):
                return '8';
                break;
            default:
                return '0';
        }
    }

    private function countryIsoMapper($country)
    {
        $countryArray = [
            'Afghanistan' => 'AF',
            'Albanien' => 'AL',
            'Algerien' => 'DZ',
            'Amerikanisch-Samoa' => 'AS',
            'Amerikanische Jungferninseln' => 'VI',
            'Andorra' => 'AD',
            'Angola' => 'AO',
            'Anguilla' => 'AI',
            'Antarktis' => 'AQ',
            'Antigua & Barbuda' => 'AG',
            'Argentinien' => 'AR',
            'Armenien' => 'AM',
            'Aruba' => 'AW',
            'Aserbaidschan' => 'AZ',
            'Australien' => 'AU',
            'Bahamas' => 'BS',
            'Bahrain' => 'BH',
            'Bangladesch' => 'BD',
            'Barbados' => 'BB',
            'Belgien' => 'BE',
            'Belize' => 'BZ',
            'Benin' => 'BJ',
            'Bermuda' => 'BM',
            'Bhutan' => 'BT',
            'Bolivien' => 'BO',
            'Bonaire, Sint Eustatius & Saba' => 'BQ',
            'Bosnien-Herzegowina' => 'BA',
            'Botswana' => 'BW',
            'Bouvetinsel' => 'BV',
            'Brasilien' => 'BR',
            'Britische Jungferninseln' => 'VG',
            'Britisches Territorium im Indischen Ozean' => 'IO',
            'Brunei Darussalam' => 'BN',
            'Bulgarien' => 'BG',
            'Burkina Faso' => 'BF',
            'Burundi' => 'BI',
            'Chile' => 'CL',
            'China' => 'CN',
            'Cookinseln' => 'CK',
            'Costa Rica' => 'CR',
            'Curacao' => 'CW',
            'Demokratische Republik Kongo' => 'CD',
            'Demokratische Volksrepublik Korea (Nordkorea)' => 'KP',
            'Deutschland' => 'DE',
            'Dominica' => 'DM',
            'Dominikanische Republik' => 'DO',
            'Dschibuti' => 'DJ',
            'Dänemark' => 'DK',
            'Ecuador' => 'EC',
            'El Salvador' => 'SV',
            "Elfenbeinküste (Côte d'Ivoire)" => 'CI',
            'Eritrea' => 'ER',
            'Estland' => 'EE',
            'Falklandinseln' => 'FK',
            'Fidschi' => 'FJ',
            'Finnland' => 'FI',
            'Frankreich' => 'FR',
            'Französisch-Guayana' => 'GF',
            'Französisch-Polynesien' => 'PF',
            'Französische Süd- und Antarktisgebiete' => 'TF',
            'Färöer' => 'FO',
            'Gabun' => 'GA',
            'Gambia' => 'GM',
            'Georgien' => 'GE',
            'Ghana' => 'GH',
            'Gibraltar' => 'GI',
            'Grenada' => 'GD',
            'Griechenland' => 'GR',
            'Großbritannien & Nordirland' => 'GB',
            'Grönland' => 'GL',
            'Guadeloupe' => 'GP',
            'Guam' => 'GU',
            'Guatemala' => 'GT',
            'Guernsey ' => 'GG',
            'Guinea' => 'GN',
            'Guinea-Bissau' => 'GW',
            'Guyana' => 'GY',
            'Haiti' => 'HT',
            'Heard- & McDonald-Inseln' => 'HM',
            'Honduras' => 'HN',
            'Hongkong' => 'HK',
            'Indien' => 'IN',
            'Indonesien' => 'ID',
            'Irak' => 'IQ',
            'Iran' => 'IR',
            'Irland' => 'IE',
            'Island' => 'IS',
            'Isle of Man' => 'IM',
            'Israel' => 'IL',
            'Italien' => 'IT',
            'Jamaika' => 'JM',
            'Japan' => 'JP',
            'Jemen' => 'YE',
            'Jersey ' => 'JE',
            'Jordanien' => 'JO',
            'Kaimaninseln' => 'KY',
            'Kambodscha' => 'KH',
            'Kamerun' => 'CM',
            'Kanada' => 'CA',
            'Kap Verde' => 'CV',
            'Kasachstan' => 'KZ',
            'Katar' => 'QA',
            'Kenia' => 'KE',
            'Kirgisistan' => 'KG',
            'Kiribati' => 'KI',
            'Kokosinseln' => 'CC',
            'Kolumbien' => 'CO',
            'Komoren' => 'KM',
            'Kroatien' => 'HR',
            'Kuba' => 'CU',
            'Kuwait' => 'KW',
            'Laos' => 'LA',
            'Lesotho' => 'LS',
            'Lettland' => 'LV',
            'Libanon' => 'LB',
            'Liberia' => 'LR',
            'Libyen' => 'LY',
            'Liechtenstein' => 'LI',
            'Litauen' => 'LT',
            'Luxemburg' => 'LU',
            'Macao' => 'MO',
            'Madagaskar' => 'MG',
            'Malawi' => 'MW',
            'Malaysia' => 'MY',
            'Malediven' => 'MV',
            'Mali' => 'ML',
            'Malta' => 'MT',
            'Marokko' => 'MA',
            'Marshallinseln' => 'MH',
            'Martinique' => 'MQ',
            'Mauretanien' => 'MR',
            'Mauritius' => 'MU',
            'Mayotte' => 'YT',
            'Mazedonien' => 'MK',
            'Mexiko' => 'MX',
            'Mikronesien' => 'FM',
            'Moldawien' => 'MD',
            'Monaco' => 'MC',
            'Mongolei' => 'MN',
            'Montenegro' => 'ME',
            'Montserrat' => 'MS',
            'Mosambik' => 'MZ',
            'Myanmar' => 'MM',
            'Namibia' => 'NA',
            'Nauru' => 'NR',
            'Nepal' => 'NP',
            'Neukaledonien' => 'NC',
            'Neuseeland' => 'NZ',
            'Nicaragua' => 'NI',
            'Niederlande' => 'NL',
            'Niger' => 'NE',
            'Nigeria' => 'NG',
            'Niue' => 'NU',
            'Norfolkinsel' => 'NF',
            'Norwegen' => 'NO',
            'Nördliche Marianen' => 'MP',
            'Oman' => 'OM',
            'Osttimor' => 'TL',
            'Pakistan' => 'PK',
            'Palau' => 'PW',
            'Palästinensische Autonomiegebiete' => 'PS',
            'Panama' => 'PA',
            'Papua-Neuguinea' => 'PG',
            'Paraguay' => 'PY',
            'Peru' => 'PE',
            'Philippinen' => 'PH',
            'Pitcairninseln' => 'PN',
            'Polen' => 'PL',
            'Portugal' => 'PT',
            'Puerto Rico' => 'PR',
            'Republik Kongo' => 'CG',
            'Republik Korea (Südkorea)' => 'KR',
            'Ruanda' => 'RW',
            'Rumänien' => 'RO',
            'Russland' => 'RU',
            'Réunion' => 'RE',
            'Saint Helena, Ascension & Tristan da Cunha' => 'SH',
            'Saint Kitts & Nevis' => 'KN',
            'Saint Lucia' => 'LC',
            'Saint Vincent & die Grenadinen' => 'VC',
            'Saint-Barthélemy' => 'BL',
            'Saint-Martin' => 'MF',
            'Saint-Pierre & Miquelon' => 'PM',
            'Salomonen' => 'SB',
            'Sambia' => 'ZM',
            'Samoa' => 'WS',
            'San Marino' => 'SM',
            'Saudi-Arabien' => 'SA',
            'Schweden' => 'SE',
            'Schweiz' => 'CH',
            'Senegal' => 'SN',
            'Serbien' => 'RS',
            'Seychellen' => 'SC',
            'Sierra Leone' => 'SL',
            'Simbabwe' => 'ZW',
            'Singapur' => 'SG',
            'Sint Maarten' => 'SX',
            'Slowakei' => 'SK',
            'Slowenien' => 'SI',
            'Somalia' => 'SO',
            'Spanien' => 'ES',
            'Spitzbergen (Svalbard)' => 'SJ',
            'Sri Lanka' => 'LK',
            'Staat Vatikanstadt' => 'VA',
            'Sudan' => 'SD',
            'Suriname' => 'SR',
            'Swasiland' => 'SZ',
            'Syrien' => 'SY',
            'São Tomé & Príncipe' => 'ST',
            'Südafrika' => 'ZA',
            'Südgeorgien & die Südlichen Sandwichinseln' => 'GS',
            'Südsudan' => 'SS',
            'Tadschikistan' => 'TJ',
            'Taiwan' => 'TW',
            'Tansania' => 'TZ',
            'Thailand' => 'TH',
            'Togo' => 'TGRR',
            'Tokelau' => 'TK',
            'Tonga' => 'TO',
            'Trinidad & Tobago' => 'TT',
            'Tschad' => 'TD',
            'Tschechien' => 'CZ',
            'Tunesien' => 'TN',
            'Turkmenistan' => 'TM',
            'Turks & Caicosinseln' => 'TC',
            'Tuvalu' => 'TV',
            'Türkei' => 'TR',
            'Uganda' => 'UG',
            'Ukraine' => 'UA',
            'Ungarn' => 'HU',
            'United States Minor Outlying Islands' => 'UM',
            'Uruguay' => 'UY',
            'USA' => 'US',
            'Usbekistan' => 'UZ',
            'Vanuatu' => 'VU',
            'Venezuela' => 'VE',
            'Vereinigte Arabische Emirate' => 'AE',
            'Vietnam' => 'VN',
            'Wallis & Futuna' => 'WF',
            'Weihnachtsinsel' => 'CX',
            'Weißrussland (Belarus)' => 'BY',
            'Westsahara' => 'EH',
            'Zentralafrikanische Republik' => 'CF',
            'Zypern' => 'CY',
            'Ägypten' => 'EG',
            'Äquatorialguinea' => 'GQ',
            'Äthiopien' => 'ET',
            'Åland' => 'AX',
            'Österreich' => 'AT'

        ];

        return $countryArray[$country];

    }
}
