<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 *   @copyright 2017 Lifestyle Webconsulting GmbH
 */

namespace Vtours\TTXmlWebService\Service;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Response;
use Vtours\TTXmlWebService\Model\Request\SearchEngineCityListRQ;
use Vtours\TTXmlWebService\Model\Request\SearchEngineHotelListRQ;
use Vtours\TTXmlWebService\Model\Request\SearchEngineRegionTreeRQ;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;

/**
 * Class TravelTainmentService
 *
 * @package Vtours\TTXmlWebService\Service
 */
class TravelTainmentService
{

    private $client;
    private $serializer;

    /**
     * TravelTainmentService constructor.
     */
    public function __construct()
    {
        $this->client = new Client();
        $this->serializer = \JMS\Serializer\SerializerBuilder::create()->build();

    }

    /**
     * get RegionTree - list of all regionIds
     *
     * @return Exception|\Psr\Http\Message\ResponseInterface
     */
    public function getRegionTree()
    {
        $searchEngineRegionTreeRequest = new SearchEngineRegionTreeRQ();
        $xml = $this->serializer->serialize($searchEngineRegionTreeRequest, 'xml');

        $regionTreeUrl = '/Search/Hotel/RegionTree';

        try {
            return $this->send($xml, $regionTreeUrl);

        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * TT - SearchEngineOfferListRequest/ -response
     *
     * @param array $durationDays
     * @param array $durationPeriod
     * @param       $mealType
     * @param       $hotelId
     * @param       $type
     *
     * @return array|Exception|\JMS\Serializer\scalar|mixed|object
     */
    public function getAllHotelOffersByGiata(array $durationDays, array $durationPeriod, $mealType, $hotelId, $type)
    {
        $searchEngineOfferListRequest = new \Vtours\TTXmlWebService\Model\Request\SearchEngineOfferListRQ($durationDays, $durationPeriod, $mealType,
            $hotelId);
        $xml = $this->serializer->serialize($searchEngineOfferListRequest, 'xml');

        $offersByHotelUrl = '/Search/' . $type . '/OfferList';

        try {
            /**
             * @var Response $response
             */
            $response = $this->send($xml, $offersByHotelUrl);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineOfferListRS::class, 'xml');

        } catch (Exception $e) {
            return $e;
        }

    }

    /**
     * TT - get SearchEngineCityListRequest/ -response
     *
     * @param array $durationDays
     * @param       $mealType
     * @param array $cityIds
     * @param array $topRegionIds
     *
     * @return Exception|\Psr\Http\Message\ResponseInterface
     */
    public function getBestCityOffer(array $durationDays, $mealType, array $cityIds, array $topRegionIds)
    {
        $searchEngineCityListRequest = new SearchEngineCityListRQ($durationDays, $mealType, $cityIds, $topRegionIds);
        $xml = $this->serializer->serialize($searchEngineCityListRequest, 'xml');

        //url nur hotel
        $offersByCityUrl = '/Search/Hotel/CityList';
        //url hotel mit flug
//        $offersByCityUrl = '/Search/Package/CityList';
        //url nur flug?
//        $offersByCityUrl = '/Search/Flight/CityList';


        try {
            return $this->send($xml, $offersByCityUrl);

        } catch (Exception $e) {
            return $e;

        }
    }

    /**
     * TT - SearchEngineHotelListRequest/ -response
     * configured to receive only one offer
     *
     * @param       $hotelId
     * @param array $duration
     * @param       $mealType
     *
     * @return Exception|\Psr\Http\Message\ResponseInterface
     */
    public function getBestPackageOfferByGiata($hotelId, array $duration, $mealType)
    {
        $searchEngineHotelListRequest = new SearchEngineHotelListRQ($duration, $mealType, $hotelId, null);
        $xml = $this->serializer->serialize($searchEngineHotelListRequest, 'xml');

        $offersByHotelUri = '/Search/Package/HotelList';

        try {
            return $this->send($xml, $offersByHotelUri);

        } catch (Exception $e) {
            return $e;

        }
    }

    /**
     * TT - SearchEngineHotelListREquest/ -response
     * configured for CityTrips
     *
     * @param array $duration
     * @param       $mealType
     * @param       $regionId
     *
     * @return Exception|\Psr\Http\Message\ResponseInterface
     */
    public function getBestHotelOffersByRegion(array $duration, $mealType, $regionId)
    {
        $searchEngineHotelListRequest = new SearchEngineHotelListRQ($duration, $mealType, null, $regionId);
        $xml = $this->serializer->serialize($searchEngineHotelListRequest, 'xml');

        $offersByHotelUri = '/Search/Hotel/HotelList';

        try {
            return $this->send($xml, $offersByHotelUri);

        } catch (Exception $e) {
            return $e;

        }
    }

    /**
     * sends requests to TT and returns response from TT
     *
     * @param $request
     * @param $path
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function send($request, $path)
    {
        $baseUrl = 'http://de.ttxml.traveltainment.eu/TTXml-1.7/Dispatcher';

        $fullPath = $baseUrl . $path;

        $response = $this->client->post($fullPath, [
            'auth' => ['MKT_655000_DE', '193893Hjtt0!aaaqq', 'digest'],
            'body' => $request
        ]);

        return $response;
    }
}
