<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 */

namespace Vtours\TTXmlWebService\Service;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Response;
use Vtours\TTXmlWebService\Model\Request\SearchEngineCityListRQ;
use Vtours\TTXmlWebService\Model\Request\SearchEngineHotelListRQ;
use Vtours\TTXmlWebService\Model\Request\SearchEngineRegionTreeRQ;
use Vtours\TTXmlWebService\Model\Response\SearchEngineCityListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineHotelListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;
use Vtours\TTXmlWebService\Model\Response\SearchEngineRegionTreeRS;

/**
 * Class TravelTainmentService
 *
 * @package Vtours\TTXmlWebService\Service
 */
class TravelTainmentService
{

    private $client;
    private $serializer;

    /**
     * TravelTainmentService constructor.
     */
    public function __construct()
    {
        $this->client = new Client();
        $this->serializer = \JMS\Serializer\SerializerBuilder::create()->build();

    }

    /**
     * get RegionTree - list of all regionIds
     * used to get all regionids and names -> needed for filtering suggester in herostage
     *
     * @return SearchEngineRegionTreeRS  | Exception
     * @throws Exception
     */
    public function getRegionTree()
    {
        $searchEngineRegionTreeRequest = new SearchEngineRegionTreeRQ();
        $xml = $this->serializer->serialize($searchEngineRegionTreeRequest, 'xml');

        $regionTreeUrl = '/Search/Hotel/RegionTree';

        try {
            /**
             * @var Response $response
             */
            $response = $this->send($xml, $regionTreeUrl);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineRegionTreeRS::class, 'xml');

        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * TT - SearchEngineOfferListRequest/ -response
     * used for getting cheapest offer for special hotel
     *
     * @param array $durationDays
     * @param array $durationPeriod
     * @param array $pricing
     * @param array $hotelAttributes
     * @param       $mealType
     * @param array $travellers
     * @param array $specialJourneyAttributes
     * @param       $brand
     * @param       $departureAirport
     * @param       $hotelId
     * @param       $hotelCodeType
     * @param       $type
     *
     * @return bool|Exception|string
     */
    public function getAllHotelOffersByHotelCode(
        array $durationDays,
        array $durationPeriod,
        array $pricing,
        array $hotelAttributes,
        $mealType,
        array $travellers,
        array $specialJourneyAttributes,
        $brand,
        $departureAirport,
        $hotelId,
        $hotelCodeType,
        $type
    ) {

        $searchEngineOfferListRequest = new \Vtours\TTXmlWebService\Model\Request\SearchEngineOfferListRQ($durationDays, $durationPeriod, $pricing,
            $hotelAttributes, $mealType,
            $travellers, $specialJourneyAttributes, $brand, $departureAirport,
            $hotelId, $hotelCodeType);
        $xml = $this->serializer->serialize($searchEngineOfferListRequest, 'xml');

        $offersByHotelUrl = '/Search/' . $type . '/OfferList';

        try {
            /**
             * @var Response $response
             */
            $response = $this->send($xml, $offersByHotelUrl);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineOfferListRS::class, 'xml');

        } catch (Exception $e) {
            return $e;
        }

    }

    /**
     * TT - get SearchEngineCityListRequest/ -response
     * actual unnused
     *
     * @param array $durationDays
     * @param       $mealType
     * @param array $cityIds
     * @param array $topRegionIds
     *
     * @return SearchEngineCityListRS | Exception
     * @throws Exception
     */
    public function getBestCityOffer(array $durationDays, $mealType, array $cityIds, array $topRegionIds)
    {
        $searchEngineCityListRequest = new SearchEngineCityListRQ($durationDays, $mealType, $cityIds, $topRegionIds);
        $xml = $this->serializer->serialize($searchEngineCityListRequest, 'xml');

        //url nur hotel
        $offersByCityUrl = '/Search/Hotel/CityList';
        //url hotel mit flug
//        $offersByCityUrl = '/Search/Package/CityList';
        //url nur flug?
//        $offersByCityUrl = '/Search/Flight/CityList';

        try {
            /**
             * @var Response $response
             */
            $response = $this->send($xml, $offersByCityUrl);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineCityListRS::class, 'xml');

        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * TT - SearchEngineHotelListRequest/ -response
     * configured to receive only one offer
     *
     * @param       $hotelId
     * @param array $duration
     * @param array $durationPeriod
     * @param       $mealType
     * @param int   $adults
     *
     * @return SearchEngineHotelListRS | Exception
     * @throws Exception
     */
    public function getBestPackageOfferByGiata($hotelId, array $duration, array $durationPeriod, $mealType, $adults, $offertype)
    {
        $searchEngineHotelListRequest = new SearchEngineHotelListRQ($duration, $durationPeriod, $mealType, $adults, $hotelId, null);
        $xml = $this->serializer->serialize($searchEngineHotelListRequest, 'xml');

        $offersByHotelUri = '/Search/' . $offertype . '/HotelList';

        try {
            /**
             * @var Response $response
             */
            $response = $this->send($xml, $offersByHotelUri);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineHotelListRS::class, 'xml');

        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * TT - SearchEngineHotelListRequest/ -response
     * used to get cheapest offer of special city
     *
     * @param array $durationDays
     * @param array $durationPeriod
     * @param array $pricing
     * @param array $hotelAttributes
     * @param       $mealType
     * @param array $travellers
     * @param array $specialJourneyAttributes
     * @param       $brand
     * @param       $departureAirport
     * @param null  $hotelId
     * @param null  $hotelCodeType
     * @param       $regionId
     * @param       $offertype
     *
     * @return array|Exception|\JMS\Serializer\scalar|mixed|object
     */
    public function getBestHotelOffersByRegion(
        array $durationDays,
        array $durationPeriod,
        array $pricing,
        array $hotelAttributes,
        $mealType,
        array $travellers,
        array $specialJourneyAttributes,
        $brand,
        $departureAirport,
        $regionId,
        $offertype
    ) {
        $searchEngineHotelListRequest = new SearchEngineHotelListRQ($durationDays, $durationPeriod, $pricing, $hotelAttributes, $mealType,
            $travellers, $specialJourneyAttributes, $brand, $departureAirport, $regionId);
        $xml = $this->serializer->serialize($searchEngineHotelListRequest, 'xml');

        $offersByHotelUri = '/Search/' . $offertype . '/HotelList';

        try {
            /**
             * @var Response $response
             */
            $response = $this->send($xml, $offersByHotelUri);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineOfferListRS::class, 'xml');

        } catch (Exception $e) {
            return $e;
        }
    }

    /**
     * sends requests to TT and returns response from TT
     *
     * @param $request
     * @param $path
     *
     * @return \Psr\Http\Message\ResponseInterface
     */
    public function send($request, $path)
    {
        $baseUrl = 'http://de.ttxml.traveltainment.eu/TTXml-1.7/Dispatcher';

        $fullPath = $baseUrl . $path;

        $response = $this->client->post($fullPath, [
            'auth' => ['MKT_655000_DE', '193893Hjtt0!aaaqq', 'digest'],
            'body' => $request
        ]);

        return $response;
    }
}
