<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 */

namespace Vtours\TTXmlWebService\Service\MappingHelper;

use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\Options;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\Search;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\Hotel;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\DepartureAirportList;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\Journey;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\PriceSpan;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\TravelDateRequest\TravelDateSpan;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\TravelDateRequest\TravelDurationSpan;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\Traveller;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\MealType;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\RoomType;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\TourOperator;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\Trip;

/**
 * Class GeneralMappingHelper
 *
 * @package Vtours\TTXmlWebService\Service\MappingHelper
 */
class GeneralMappingHelper
{
    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return \Vtours\TTXmlWebService\Model\Request\SearchRequest\Search
     */
    public function mapSearchEngineRequest(DatenAngebotRequestModel $request)
    {
        $journey = $this->mapJourney($request);
        $hotel = $this->mapHotel($request);
        $tourOperator = $this->mapTourOperator($request);

        //----TRIP----
        $trip = new Trip();
        $trip
            ->setHotel($hotel)
            ->setTourOperator($tourOperator)
            ->setJourney($journey);

        $options = $this->mapOptions();

        //----SEARCH----
        $search = new Search();
        $search
            ->setOptions($options)
            ->setTrip($trip);

        return $search;
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return Journey
     */
    private function mapJourney(DatenAngebotRequestModel $request)
    {
        /**
         * @var Journey $journey
         */
        $journey = new Journey();
        $journey->setDepartureAirportCountry('DE');
        $travellers = $request->getTravellers();
        $travellerList = null;
        foreach ($travellers as $traveller) {
            $t = new Traveller();
            $t->setAge($traveller);
            $travellerList[] = $t;
        }
        $journey = $journey->setTravellerList($travellerList);
        $journey->setTravelDateSpan($this->mapTravelDateSpan($request));
        $journey->setTravelDurationSpan($this->mapTravelDurationSpan($request));
        $journey->setPriceSpan($this->mapPriceSpan($request));

        $departureAirportList = new DepartureAirportList();

        if (!empty($request->getDepartureAirport())) {
            $airports = $request->getDepartureAirport();

            $departureAirportList->setDepartureAirportList([$airports]);
            $departureAirportList->setWeightage(50);
        }
        $journey->setDepartureAirportList($departureAirportList);

        if (!empty($request->getSpecialJourneyAttributes())) {
            $specials = $request->getSpecialJourneyAttributes();
            $journey->setSpecialJourneyAttributes($specials);
        }

        return $journey;
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return PriceSpan
     */
    public function mapPriceSpan(DatenAngebotRequestModel $request)
    {
        if ($request->getPriceMin() != null || $request->getPriceMax() != null) {
            $priceSpan = new PriceSpan();
            $priceSpan->setCurrencyCode('EUR');
            $minPrice = $request->getPriceMin();
            $maxPrice = $request->getPriceMax();
            if ($minPrice != null) {
                $priceSpan->setMinPrice($minPrice);
            }
            if ($maxPrice != null) {
                $priceSpan->setMaxPrice($maxPrice);
            }

            return $priceSpan;
        }
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return TravelDurationSpan
     */
    public function mapTravelDurationSpan(DatenAngebotRequestModel $request)
    {
        $travelDurationSpan = new TravelDurationSpan();
        if (!empty($request->getTravelDurationMin()) || !empty($request->getTravelDurationMax())) {
            $minDays = $request->getTravelDurationMin();
            $maxDays = $request->getTravelDurationMax();
            if ($minDays != null) {
                $travelDurationSpan->setMinDays($minDays);
            }
            if ($maxDays != null) {
                $travelDurationSpan->setMaxDays($maxDays);
            }
            $travelDurationSpan->setWeightage(60);
        }

        return $travelDurationSpan;
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return TravelDateSpan
     */
    public function mapTravelDateSpan(DatenAngebotRequestModel $request)
    {
        $travelDateSpan = new TravelDateSpan();
        $departureDate = $request->getTravelDateStart();
        $returnDate = $request->getTravelDateEnd();
        if ($departureDate != null) {
            $travelDateSpan->setDepartureDate(date($departureDate));
        }
        if ($returnDate != null) {
            $travelDateSpan->setReturnDate(date($returnDate));
        }

        return $travelDateSpan;
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return Hotel
     */
    private function mapHotel(DatenAngebotRequestModel $request)
    {
        $hotel = new Hotel();
        if ($request->getMealType() != null) {
            $meal = $request->getMealType();
            $mealType = new MealType();
            $mealType->setMealType($meal);
            $mealType->setWeightage(60);
            $hotel->setMealType($mealType);
        }

        if ($request->getRoomType() != null) {
            $room = $request->getRoomType();
            $roomType = new RoomType();
            $roomType->setRoom($room);
            $roomType->setWeightage(60);
            $hotel->setRoomType($roomType);
        }
        if (!empty($request->getHotelAttributes())) {
            $hotelAttributes = ($request->getHotelAttributes());
            $hotel->setHotelAttributes($hotelAttributes);
        }

        return $hotel;
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return TourOperator
     */
    private function mapTourOperator(DatenAngebotRequestModel $request)
    {
        $touroperator = new TourOperator();
        if ($request->getBrand() != null) {
            $brand = $request->getBrand();
            $touroperator->setLimit([$brand]);
        }

        return $touroperator;
    }

    /**
     * @return Options
     */
    private function mapOptions()
    {
        $options = new Options();
        $options
            ->setResultsPerPage(10)
            ->setResultOffset(0)
            ->setSorting('PRICE')
            ->setAdditionalCurrencies('EUR');

        return $options;
    }
}
