<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 *  Any unauthorized use of this software without a valid license is a violation of the license agreement
 *  and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 */

namespace Vtours\TTXmlWebService\Service\OfferList;

use GuzzleHttp\ClientInterface;
use JMS\Serializer\SerializerInterface;
use Psr\Http\Message\ResponseInterface;
use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\Response\SearchEngineOfferListRS;
use Vtours\TTXmlWebService\Model\Request\SearchRequest\TripRequest\JourneyRequest\Traveller;

#V2 Response
use Vtours\TTXmlWebService\Model\Response\V2\AccommodationOffersRS;
use Vtours\TTXmlWebService\Model\Response\V2\PackageOffersRS;


/**
 * Class Handler
 *
 * @package Vtours\TTXmlWebService\Service\OfferList
 */
class Handler
{
    const API_URI = '/TTXml-1.7/Dispatcher/Search/%s/OfferList';

    const apiUrlPackageOffer = 'https://marketing-de-ibe.ws.traveltainment.eu/ttgateway-web-v3/rest/PackageSearch/packageOffers';
    const apiUrlAccommodationOffer = 'https://marketing-de-ibe.ws.traveltainment.eu/ttgateway-web-v3/rest/AccommodationSearch/accommodationOffers';

    const apiUrlAuth = 'https://auth.ws.traveltainment.eu/auth/realms/SystemUser-BasicAccessLevel/protocol/openid-connect/token';

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * Handler constructor.
     *
     * @param Mapper $mapper
     * @param SerializerInterface $serializer
     * @param ClientInterface $client
     */
    public function __construct(Mapper $mapper, SerializerInterface $serializer, ClientInterface $client)
    {
        $this->mapper = $mapper;
        $this->serializer = $serializer;
        $this->client = $client;
    }


    /**
     * @return mixed|string
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    private function oauth2AccessToken()
    {
        $requestOptions = [
            'headers' => [],
            'form_params' => [
                'grant_type' => 'password',
                'client_id' => 'gateway',
                'username' => 'mkt_655000_de',
                'password' => 'Laura2020!',
            ],
        ];


        $response = $this->client->request('POST', self::apiUrlAuth, $requestOptions);

        if ((int)$response->getStatusCode() === 200) {
            $responseArray = json_decode($response->getBody()->getContents(), true);

            return $responseArray['access_token'];
        }

        return '';
    }


    /**
     * @param DatenAngebotRequestModel $request
     * @return array|
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getAllHotelOffersByHotelCodeV2(DatenAngebotRequestModel $request)
    {
        $requestModel = $this->mapper->mapRequestByHotelCodeV2($request);
        $requestJson = $this->serializer->serialize($requestModel,'json');

        $requestOptions = [
            'headers' => ['Authorization' => 'Bearer ' . $this->oauth2AccessToken(), 'content-type' => 'application/json'],
            'body' => $requestJson,
        ];
        try {
            $response = $this->client->request('POST', ($request->getOffertype() === 'Package') ? self::apiUrlPackageOffer : self::apiUrlAccommodationOffer, $requestOptions);
            return $this->serializer->deserialize($response->getBody()->getContents() , ($request->getOffertype() === 'Package') ? PackageOffersRS::class : AccommodationOffersRS::class , 'json');

        } catch (ClientException $exception) {
            $logger = \Pimcore\Log\ApplicationLogger::getInstance("TTXmlWebService", true);
            $logger->alert('ERROR: While TT Post Request: ' . $exception->getMessage());

            return $exception;
        }
    }


    public function getAllHotelOffersByHotelCode(DatenAngebotRequestModel $request)
    {
        $requestModel = $this->mapper->mapRequestByHotelCode($request);
        $requestXml = $this->serializer->serialize($requestModel, 'xml');

        $requestUri = $this->buildUriFromRequest($request);

        $requestOptions = [
            'headers' => [],
            'auth' => ['MKT_655000_DE', '193893Hjtt0!aaaqq', 'digest'],
            'body' => $requestXml,
        ];

        try {
            /** @var ResponseInterface $response */
            $response = $this->client->request('POST', $requestUri, $requestOptions);
            $result = $response->getBody()->getContents();

            $this->logXmlFiles($request, $requestXml, $result);

            return $this->serializer->deserialize($result, SearchEngineOfferListRS::class, 'xml');
        } catch (\Exception $exception) {
            // return false or null? or maybe do not catch exception in pimcore context?
            try {
                $logger = \Pimcore\Log\ApplicationLogger::getInstance("TTXmlWebService", true);
                $logger->alert('ERROR: While TT Post Request: ' . $exception->getMessage());
            } catch (\Exception $e) {
            }

            return $exception;
        }
    }

    /**
     * @param DatenAngebotRequestModel $request
     * @param $requestXml
     * @param $responseXml
     */
    private function logXmlFiles(DatenAngebotRequestModel $request, $requestXml, $responseXml)
    {
        $logger = \Pimcore\Log\ApplicationLogger::getInstance("TTXmlWebService", true);

        $path = PIMCORE_LOG_FILEOBJECT_DIRECTORY;
        $random = rand(000000000000000, 999999999999999);
        $requestXmlFilename = "request_" . $request->getCode() . "_" . $request->getBrand() . "_" . $random;
        $responseXmlFilename = "response_" . $request->getCode() . "_" . $request->getBrand() . "_" . $random;

        $fileObjectRequest = new \Pimcore\Log\FileObject($requestXml, "$path/$requestXmlFilename.xml");
        $fileObjectResponse = new \Pimcore\Log\FileObject($responseXml, "$path/$responseXmlFilename.xml");

        $logger->setFileObject($fileObjectRequest);
        $logger->info("INFO: The request XML has been logged as " . $fileObjectRequest->getSystemPath());
        $logger->setFileObject($fileObjectResponse);
        $logger->info("INFO: The response XML has been logged as " . $fileObjectResponse->getSystemPath());

    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return string
     */
    private function buildUriFromRequest(DatenAngebotRequestModel $request)
    {
        return sprintf(self::API_URI, $request->getOfferType());
    }
}
