<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Vtours\TTXmlWebService\Service\HotelList;

use GuzzleHttp\ClientInterface;
use JMS\Serializer\SerializerInterface;
use Psr\Http\Message\ResponseInterface;
use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\Response\SearchEngineHotelListRS;

/**
 * Class Handler
 *
 * @package Vtours\TTXmlWebService\Service\HotelList
 */
class Handler
{
    const API_URI = '/TTXml-1.7/Dispatcher/Search/%s/HotelList';

    const API_URL_PACKAGE_OFFER = 'https://marketing-de-ibe.ws.traveltainment.eu/ttgateway-web-v3/rest/PackageSearch/packageOffers';
    const API_URL_ACCOMMODATION_OFFER = 'https://marketing-de-ibe.ws.traveltainment.eu/ttgateway-web-v3/rest/AccommodationSearch/accommodationOffers';

    const API_URL_AUTH = 'https://auth.ws.traveltainment.eu/auth/realms/SystemUser-BasicAccessLevel/protocol/openid-connect/token';

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * Handler constructor.
     *
     * @param Mapper              $mapper
     * @param SerializerInterface $serializer
     * @param ClientInterface     $client
     */
    public function __construct(Mapper $mapper, SerializerInterface $serializer, ClientInterface $client)
    {
        $this->mapper = $mapper;
        $this->serializer = $serializer;
        $this->client = $client;
    }

    /**
     * @return mixed|string
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    private function oauth2AccessToken()
    {
        $requestOptions = [
            'headers' => [],
            'form_params' => [
                'grant_type' => 'password',
                'client_id' => 'gateway',
                'username' => 'mkt_655000_de',
                'password' => 'Laura2020!',
            ],
        ];


        $response = $this->client->request('POST', self::API_URL_AUTH, $requestOptions);

        if ((int)$response->getStatusCode() === 200) {
            $responseArray = json_decode($response->getBody()->getContents(), true);

            return $responseArray['access_token'];
        }

        return '';
    }


    /**
     * @param DatenAngebotRequestModel $request
     * @return array|bool|Exception|float|int|object|string|ClientException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function getBestOffersByRegionV2(DatenAngebotRequestModel $request)
    {
        $requestModel = $this->mapper->mapRequestByRegionV2($request);
        $requestJson = $this->serializer->serialize($requestModel,'json');

        $requestOptions = [
            'headers' => ['Authorization' => 'Bearer ' . $this->oauth2AccessToken(), 'content-type' => 'application/json'],
            'body' => $requestJson,
        ];
        try {
            $response = $this->client->request('POST', ($request->getOffertype() === 'Package') ? self::API_URL_PACKAGE_OFFER : self::API_URL_ACCOMMODATION_OFFER, $requestOptions);
            return $this->serializer->deserialize($response->getBody()->getContents() , ($request->getOffertype() === 'Package') ? PackageOffersRS::class : AccommodationOffersRS::class , 'json');

        } catch (ClientException $exception) {
            $logger = \Pimcore\Log\ApplicationLogger::getInstance("TTXmlWebService", true);
            $logger->alert('ERROR: While TT Post Request: ' . $exception->getMessage());

            return $exception;
        }
    }


    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return array|\Exception|\JMS\Serializer\scalar|object
     */
    public function getBestOffersByRegion(DatenAngebotRequestModel $request)
    {
        $requestXml = $this->createRequestXml($request);

        $requestUri = $this->buildUriFromRequest($request);

        $requestOptions = [
            'headers' => [],
            'auth'    => ['MKT_655000_DE', '193893Hjtt0!aaaqq', 'digest'],
            'body'    => $requestXml,
        ];

        try {
            /** @var ResponseInterface $response */
            $response = $this->client->request('POST', $requestUri, $requestOptions);
            $result = $response->getBody()->getContents();

            return $this->serializer->deserialize($result, SearchEngineHotelListRS::class, 'xml');
        } catch (\Exception $exception) {
            // return false or null? or maybe do not catch exception in pimcore context?
            return $exception;
        }
    }

    /**
     * @param \Vtours\TTXmlWebService\Model\DatenAngebotRequestModel $request
     *
     * @return string
     */
    private function createRequestXml(DatenAngebotRequestModel $request)
    {
        $requestModel = $this->mapper->mapRequestByRegion($request);

        return $this->serializer->serialize($requestModel, 'xml');
    }

    /**
     * @param $request
     *
     * @return string
     */
    private function buildUriFromRequest($request)
    {
        return sprintf(self::API_URI, $request->getOfferType());
    }
}
