<?php

/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Vtours\TTXmlWebService\Service\HotelList;

use Exception;
use Ramsey\Uuid\Uuid;
use Vtours\TTXmlWebService\Model\DatenAngebotRequestModel;
use Vtours\TTXmlWebService\Model\Request\SearchEngineHotelListRQ;
use Vtours\TTXmlWebService\Model\Request\SelectionRequest\Selection;
use Vtours\TTXmlWebService\Model\Request\v2\AccommodationOfferMainRQ;
use Vtours\TTXmlWebService\Model\Request\v2\AccommodationOffersRQ\AccommodationOfferRQ;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\AccomFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\AccomPropertiesFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\AccomSelectors;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\DateAndTimeFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\DateRange;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\InboundTransportDateAndTimeFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\OutboundTransportDateAndTimeFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\PriceFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferFilters\TravelDurationFilter;
use Vtours\TTXmlWebService\Model\Request\v2\OfferOptions;
use Vtours\TTXmlWebService\Model\Request\v2\OfferTravellers;
use Vtours\TTXmlWebService\Model\Request\v2\PackageOffersMainRQ;
use Vtours\TTXmlWebService\Model\Request\v2\PackageOffersRQ\PackageOffersRQ;
use Vtours\TTXmlWebService\Model\Request\v2\RQMeta;
use Vtours\TTXmlWebService\Service\MappingHelper\GeneralMappingHelper;

/**
 * Class Mapper
 *
 * @package Vtours\TTXmlWebService\Service\HotelList
 */
class Mapper
{
    /**
     * @var GeneralMappingHelper
     */
    private $generalMappingHelper;

    /**
     * Mapper constructor.
     *
     * @param GeneralMappingHelper $generalMappingHelper
     */
    public function __construct(GeneralMappingHelper $generalMappingHelper)
    {
        $this->generalMappingHelper = $generalMappingHelper;
    }

    /**
     * @param DatenAngebotRequestModel $request
     *
     * @return SearchEngineHotelListRQ
     */
    public function mapRequestByRegion(DatenAngebotRequestModel $request)
    {
        $requestSearchModel = $this->generalMappingHelper->mapSearchEngineRequest($request);

        $selection = new Selection();
        $selection->setRegionIds($request->getCode());

        $searchEngineHotelListRQ = new SearchEngineHotelListRQ();
        $searchEngineHotelListRQ
            ->setSelection($selection)
            ->setSearch($requestSearchModel);

        return $searchEngineHotelListRQ;
    }

    /**
     * @param DatenAngebotRequestModel $request
     * @return PackageOffersMainRQ
     */
    public function mapRequestByRegionV2(DatenAngebotRequestModel $request)
    {
        return ($request->getOffertype()  === 'Package') ? $this->packageOfferRQ($request) : $this->accommodationOfferRQ($request);
    }


    /**
     * @param $request
     * @return AccommodationOfferMainRQ
     * @throws Exception
     */
    private function accommodationOfferRQ($request)
    {
        $accommodationOfferRQ = new AccommodationOfferMainRQ();
        $accommodationOfferRequest = new AccommodationOfferRQ();

        // MetaData
        $packageOfferMeta = new RQMeta();
        $packageOfferMeta->setRequestId(UUID::uuid4());
        $packageOfferMeta->setTransactonId(UUID::uuid4());
        $accommodationOfferRequest->setRQMeta($packageOfferMeta);

        //travellers
        $travellersList = [];
        foreach ($request->getTravellers() as $key => $value) {
            $travellersList[$key] = ["Age" => $value];
        }

        $travellers = new OfferTravellers();
        $travellers->setTravellerList($travellersList);
        $accommodationOfferRequest->setTravellers($travellers);

        //OfferFilters
        $offerFilters = new OfferFilters();

        //Travel Duration Filter
        $travelDurationFilter = new OfferFilters\TravelDurationFilter();
        $travelDurationFilter->setDurationKind((
            "Package" === $request->getOffertype()) ?
            "Trip" :
            "OvernightStays"
        );
        $travelDurationFilter->setMinDuration((int)$request->getTraveldurationMin());
        $travelDurationFilter->setMaxDuration((int)$request->getTraveldurationMax());
        $offerFilters->setTravelDurationFilter($travelDurationFilter);

        //Price Filter
        if ($request->getPriceMax() !== null) {
            $priceFilter = new PriceFilter();
            $priceFilter->setMaxPrice((int)$request->getPriceMax());
            $offerFilters->setPriceFilter($priceFilter);
        }

        //accomFilter
        $accomFilter = new AccomFilter();
        $accomSelectors = new AccomSelectors();

        switch ($request->getCodeType()){
            case 'IFF':
                $accomSelectors->setHotelIffCodes([(int)$request->getCode()]);
                break;
            case 'Giata':
                $accomSelectors->setHotelGiataIDs([(int)$request->getCode()]);
                break;
            case 'TopRegionID':
                $accomSelectors->setRegionIds([(int)$request->getCode()]);
                break;
            default:
                break;
        }

        $accomFilter->setAccomSelectors($accomSelectors);
        $offerFilters->setAccomFilter($accomFilter);

        if (null !== $request->getMealType()  || null !== $request->getRoomType()) {
            $accomPropertiesFilter = new AccomPropertiesFilter();
            if (null !== $request->getMealType()) {
                $mappingMealType = [
                    "HALFBOARD" => "HalfBoard",
                    "FULLBOARD" => "FullBoard",
                    "BREAKFAST" => "Breakfast",
                    "ALLINCLUSIVE" => "AllInclusive",
                    "ALLINCLUSIVEPLUS" => "AllInclusiveSuperior",
                    "HALFBOARDPLUS" => "HalfBoardSuperior",
                    "FULLBOARDPLUS" => "FullBoardSuperior",
                    "UNKNOWN" => "NoBoard",
                    "ONLYSTAY" => "NoBoard",

                ];
                $accomPropertiesFilter->setMinBoardType($mappingMealType[$request->getMealType()]);
            }

            if (null !== $request->getRoomType()) {
                $mappingRoomType = [
                    "APPARTEMENT" => "Apartment",
                    "BUNGALOW" => "Bungalow",
                    "DELUXEBEDROOM" => "DeluxeRoom",
                    "DOUBLEBEDROOM" => "DoubleRoom",
                    "DUPLEXBEDROOM" => "DuplexRoom",
                    "FAMILYROOM" => "FamilyRoom",
                    "HOLIDAYFLAT" => "HolidayFlat",
                    "HOLIDAYHOUSE" => "SummerCottage",
                    "MULTISHAREROOM" => "MultiBedroom",
                    "PROGRAM" => "AccordingDescription",
                    "SINGLEBEDROOM" => "SingleRoom",
                    "STUDIO" => "Studio",
                    "SUITE" => "Suite",
                    "SUPERIOR" => "SuperiorRoom",
                    "TRIPLEBEDROOM" => "ThreeBedRoom",
                    "UNKNOWN" => "AccordingDescription",
                    "VILLA" => "Villa",
                ];


                $accomPropertiesFilter->setRoomTypes([$mappingRoomType[$request->getRoomType()]]);
            }

            $offerFilters->setAccomPropertiesFilter($accomPropertiesFilter);
        }

        //set filter in general request
        $accommodationOfferRequest->setOfferFilters($offerFilters);

        // set options in general request
        $offerOptions = new OfferOptions();
        $accommodationOfferRequest->setOfferOptions($offerOptions);


        $accommodationOfferRQ->setAccommodationOfferRQ($accommodationOfferRequest);

        return $accommodationOfferRQ;

    }


    /**
     * @throws Exception
     */
    private function packageOfferRQ($request)
    {
        $packageOffersMainRQ = new PackageOffersMainRQ();
        $packageOffersRequest = new PackageOffersRQ();

        // MetaData
        $packageOfferMeta = new RQMeta();
        $packageOfferMeta->setRequestId(UUID::uuid4());
        $packageOfferMeta->setTransactonId(UUID::uuid4());
        $packageOffersRequest->setRQMeta($packageOfferMeta);


        //travellers
        $travellersList = [];
        foreach ($request->getTravellers() as $key => $value) {
            $travellersList[$key] = ["Age" => $value];
        }

        $travellers = new OfferTravellers();
        $travellers->setTravellerList($travellersList);
        $packageOffersRequest->setTravellers($travellers);

        //OfferFilters
        $offerFilters = new OfferFilters();

        //DateAndTimeFilter
        $dateAndTimeFilter = new DateAndTimeFilter();
        $outboundTransportDateAndTimeFilter = new OutboundTransportDateAndTimeFilter();
        $inboundTransportDateAndTimeFilter = new InboundTransportDateAndTimeFilter();

        $dateRange = new DateRange();
        $dateRange->setMinDate((new \DateTime($request->getTraveldateStart()->date))->format('Y-m-d'));
        $outboundTransportDateAndTimeFilter->setDateRange($dateRange);

        $dateRange = new DateRange();
        $dateRange->setMaxDate((new \DateTime($request->getTraveldateEnd()->date))->format('Y-m-d'));
        $inboundTransportDateAndTimeFilter->setDateRange($dateRange);

        $dateAndTimeFilter->setInboundTransportDateAndTimeFilter($inboundTransportDateAndTimeFilter);
        $dateAndTimeFilter->setOutboundTransportDateAndTimeFilter($outboundTransportDateAndTimeFilter);


        $offerFilters->setDateAndTimeFilter($dateAndTimeFilter);

        //StationFilter
        if (!empty($request->getDepartureAirport())) {
            $stationFilter = new OfferFilters\StationFilter();
            $departureStationFilter = new OfferFilters\DepartureStationFilter();
            $departureStationFilter->setStationCodes([$request->getDepartureAirport()]);
            $stationFilter->setDepatureStationFilter($departureStationFilter);
            $offerFilters->setStationFilter($stationFilter);
        }


        //Travel Duration Filter
        $travelDurationFilter = new TravelDurationFilter();
        $travelDurationFilter->setDurationKind((
            "Package" === $request->getOffertype()) ?
            "Trip" :
            "OvernightStays"
        );
        $travelDurationFilter->setMinDuration((int)$request->getTraveldurationMin());
        $travelDurationFilter->setMaxDuration((int)$request->getTraveldurationMax());
        $offerFilters->setTravelDurationFilter($travelDurationFilter);

        //Price Filter
        if ($request->getPriceMax() !== null) {
            $priceFilter = new PriceFilter();
            $priceFilter->setMaxPrice((int)$request->getPriceMax());
            $offerFilters->setPriceFilter($priceFilter);
        }

        //accomFilter
        $accomFilter = new AccomFilter();
        $accomSelectors = new AccomSelectors();
        switch ($request->getCodeType()) {
            case 'IFF':
                $accomSelectors->setHotelIffCodes([(int)$request->getCode()]);
                break;
            case 'Giata':
                $accomSelectors->setHotelGiataIDs([(int)$request->getCode()]);
                break;
            case 'TopRegionID':
                $accomSelectors->setRegionIds([(int)$request->getCode()]);
                break;
            default:
                break;
        }
        $accomFilter->setAccomSelectors($accomSelectors);
        $offerFilters->setAccomFilter($accomFilter);

        if (null !== $request->getMealType()  || null !== $request->getRoomType()) {
            $accomPropertiesFilter = new AccomPropertiesFilter();
            if (null !== $request->getMealType()) {
                $mappingMealType = [
                    "HALFBOARD" => "HalfBoard",
                    "FULLBOARD" => "FullBoard",
                    "BREAKFAST" => "Breakfast",
                    "ALLINCLUSIVE" => "AllInclusive",
                    "ALLINCLUSIVEPLUS" => "AllInclusiveSuperior",
                    "HALFBOARDPLUS" => "HalfBoardSuperior",
                    "FULLBOARDPLUS" => "FullBoardSuperior",
                    "UNKNOWN" => "NoBoard",
                    "ONLYSTAY" => "NoBoard",

                ];
                $accomPropertiesFilter->setMinBoardType($mappingMealType[$request->getMealType()]);
            }

            if (null !== $request->getRoomType()) {
                $mappingRoomType = [
                    "APPARTEMENT" => "Apartment",
                    "BUNGALOW" => "Bungalow",
                    "DELUXEBEDROOM" => "DeluxeRoom",
                    "DOUBLEBEDROOM" => "DoubleRoom",
                    "DUPLEXBEDROOM" => "DuplexRoom",
                    "FAMILYROOM" => "FamilyRoom",
                    "HOLIDAYFLAT" => "HolidayFlat",
                    "HOLIDAYHOUSE" => "SummerCottage",
                    "MULTISHAREROOM" => "MultiBedroom",
                    "PROGRAM" => "AccordingDescription",
                    "SINGLEBEDROOM" => "SingleRoom",
                    "STUDIO" => "Studio",
                    "SUITE" => "Suite",
                    "SUPERIOR" => "SuperiorRoom",
                    "TRIPLEBEDROOM" => "ThreeBedRoom",
                    "UNKNOWN" => "AccordingDescription",
                    "VILLA" => "Villa",
                ];


                $accomPropertiesFilter->setRoomTypes([$mappingRoomType[$request->getRoomType()]]);
            }

            $offerFilters->setAccomPropertiesFilter($accomPropertiesFilter);
        }

        //set filter in general request
        $packageOffersRequest->setOfferFilters($offerFilters);

        // set options in general request
        $offerOptions = new OfferOptions();
        $packageOffersRequest->setOfferOptions($offerOptions);


        //map all to the mothership
        $packageOffersMainRQ->setPackageOffersRQ($packageOffersRequest);

        return $packageOffersMainRQ;
    }
}
