<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright 2020 Lifestyle Webconsulting GmbH
 * @link http://www.life-style.de
 */

namespace Vtours\VmixSync\Service;

use ReflectionException;
use Vtours\VmixSync\Converter\ResponseConverter;
use Vtours\VmixSync\Exception\VmixSyncException;
use Vtours\VmixSync\Model\Config;
use Vtours\VmixSync\Model\Request;
use Vtours\VmixSync\Model\Response;
use Vtours\VmixSync\Model\VmixDataInterface;

/**
 * Class VmixService
 * @package Vtours\VmixSync\Service
 */
class VmixService
{
    const CHECK_MD5_SUFFIX = 'vto_mid_office_processing_web_service#001';

    /**
     * @param int $bookingId
     * @param VmixDataInterface $vmixData
     * @param Config $config
     * @return Response
     * @throws VmixSyncException
     */
    public function syncToVmix($bookingId, VmixDataInterface $vmixData, Config $config)
    {
        $request = (new Request())
            ->setBookingId($bookingId)
            ->setCheckMd5($bookingId . static::CHECK_MD5_SUFFIX)
            ->setVmixData($vmixData);

        try {
            $response = $this->request($request, $config);
        } catch (VmixSyncException $exception) {
            throw new VmixSyncException('VmixSyncError: ' . $exception->getMessage(), $exception->getCode(), $exception);
        } catch (ReflectionException $exception) {
            throw new VmixSyncException('VmixSyncError: ' . $exception->getMessage(), $exception->getCode(), $exception);
        }

        return $response;
    }

    /**
     * @param Request $request
     * @param Config $config
     * @return Response
     * @throws ReflectionException
     * @throws VmixSyncException
     */
    private function request(Request $request, Config $config)
    {
        // Curl is used instead of guzzle to reduce dependencies and increase compatibility

        $body = http_build_query([
            'data' => base64_encode(serialize($request->asArray()))
        ]);

        $handler = curl_init();

        curl_setopt_array($handler, [
            CURLOPT_URL => $config->getBaseUrl()  . '/' . $config->getPath(),
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $body,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $config->getTimeout()
        ]);

        $response = curl_exec($handler);
        curl_close($handler);

        return ResponseConverter::convert($response);
    }
}
