<?php
/**
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license is a violation of the license agreement
 * and will be prosecuted by civil and criminal law.
 *
 * @copyright 2017 Lifestyle Webconsulting GmbH
 * @link      www.life-style.de
 */

namespace Vtours\Website\Model;

use Doctrine\Common\Cache\CacheProvider;
use Vtours\Website\Model\Navigation\Route;
use Vtours\Website\Model\Stage\Element\MetaData;

/**
 * Class Page
 *
 * @package Vtours\Website\Model
 */
class Page
{

    /**
     * @var string
     */
    const CACHE_PREFIX = 'vtours_page_';

    /**
     * @var CacheProvider
     */
    private $cacheDriver;

    /**
     * @var int
     */
    private $pageId;

    /**
     * @var array
     */
    private $stages = [];

    /**
     * @var string
     */
    private $templateName = '_layout.twig';

    /**
     * @var string
     */
    private $twoColTemplateName = '_layout_twocols.twig';

    /**
     * @var string
     */
    private $linkName;

    /**
     * @var MetaData
     */
    private $metaData;

    /**
     * @var string
     */
    private $path;

    /**
     * @var bool
     */
    private $menu;

    /**
     * @var string
     */
    private $serviceTitle;

    /**
     * @var Route
     */
    private $serviceTitleRoute;

    /**
     * @var Route[]
     */
    private $menuItems;

    /**
     * @var Route
     */
    private $menuHome;

    /**
     * @return int
     */
    public function getPageId()
    {
        return $this->pageId;
    }

    /**
     * @param int $pageId
     *
     * @return Page
     */
    public function setPageId($pageId)
    {
        $this->pageId = $pageId;

        return $this;
    }

    /**
     * @return array
     */
    public function getStages()
    {
        if ($this->stages === []) {
            $this->load();
        }

        return $this->stages;
    }

    /**
     * setStages
     *
     * @param $stages
     *
     * @return $this
     */
    public function setStages($stages)
    {
        $this->stages = $stages;

        return $this;
    }

    /**
     * getStageByName
     *
     * @param $modelName
     *
     * @return mixed|null
     */
    public function getStageByName($modelName)
    {

        foreach ($this->stages as $stage) {
            if ($stage instanceof $modelName) {
                return $stage;
            }
        }

        return null;
    }

    /**
     * getFilteredStages
     *
     * @param array $filter
     *
     * @return array
     */
    public function getFilteredStages(array $filter)
    {
        $stages = [];
        foreach ($this->stages as $stage) {
            if (!array_search(get_class($stage), $filter)) {
                $stages[] = $stage;
            }
        }

        return $stages;
    }

    /**
     * @return string
     */
    public function getTemplateName()
    {
        return $this->templateName;
    }

    /**
     * @param string $templateName
     *
     * @return Page
     */
    public function setTemplateName($templateName)
    {
        $this->templateName = $templateName;

        return $this;
    }

    /**
     * @return string
     */
    public function getTwoColTemplateName()
    {
        return $this->twoColTemplateName;
    }

    /**
     * @param string $twoColTemplateName
     *
     * @return Page
     */
    public function setTwoColTemplateName($twoColTemplateName)
    {
        $this->twoColTemplateName = $twoColTemplateName;

        return $this;
    }

    /**
     * @return string
     */
    public function getLinkName()
    {
        return $this->linkName;
    }

    /**
     * @param string $linkName
     *
     * @return $this
     */
    public function setLinkName($linkName)
    {
        $this->linkName = $linkName;

        return $this;
    }

    /**
     * @return MetaData
     */
    public function getMetaData()
    {
        return $this->metaData;
    }

    /**
     * @param MetaData $metaData
     *
     * @return Page
     */
    public function setMetaData($metaData)
    {
        $this->metaData = $metaData;

        return $this;
    }

    /**
     * @return string
     */
    public function getPath()
    {
        return $this->path;
    }

    /**
     * @param string $path
     *
     * @return Page
     */
    public function setPath($path)
    {
        $this->path = $path;

        return $this;
    }

    /**
     * @return bool
     */
    public function isMenu()
    {
        return $this->menu;
    }

    /**
     * @param bool $menu
     *
     * @return Page
     */
    public function setMenu($menu)
    {
        $this->menu = $menu;

        return $this;
    }

    /**
     * determineTemplateName
     *
     * @return string
     */
    public function determineTemplateName()
    {
        if ($this->menu) {
            return $this->twoColTemplateName;
        }

        return $this->templateName;
    }

    /**
     * @return string
     */
    public function getServiceTitle()
    {
        return $this->serviceTitle;
    }

    /**
     * @param string $serviceTitle
     *
     * @return Page
     */
    public function setServiceTitle($serviceTitle)
    {
        $this->serviceTitle = $serviceTitle;

        return $this;
    }

    /**
     * @return Route
     */
    public function getServiceTitleRoute()
    {
        return $this->serviceTitleRoute;
    }

    /**
     * @param Route $serviceTitleObject
     *
     * @return Page
     */
    public function setServiceTitleRoute($serviceTitleObject)
    {
        $this->serviceTitleRoute = $serviceTitleObject;

        return $this;
    }

    /**
     * @return Route[]
     */
    public function getMenuItems()
    {
        return $this->menuItems;
    }

    /**
     * @param Route[] $menuItems
     *
     * @return Page
     */
    public function setMenuItems($menuItems)
    {
        $this->menuItems = $menuItems;

        return $this;
    }

    /**
     * @return Route
     */
    public function getMenuHome()
    {
        return $this->menuHome;
    }

    /**
     * @param Route $menuHome
     *
     * @return Page
     */
    public function setMenuHome($menuHome)
    {
        $this->menuHome = $menuHome;

        return $this;
    }

    /**
     * @return string
     */
    public function getCacheKey()
    {

        return static::CACHE_PREFIX . $this->pageId . '_' . hash('sha256', $this->pageId);
    }

    /**
     * @return CacheProvider
     */
    public function getCacheDriver()
    {
        return $this->cacheDriver;
    }

    /**
     * @param CacheProvider $cacheDriver
     *
     * @return Page
     */
    public function setCacheDriver($cacheDriver)
    {
        $this->cacheDriver = $cacheDriver;

        return $this;
    }

    /**
     * load
     */
    public function load()
    {
        $data = $this->cacheDriver->fetch($this->getCacheKey());

        $this->linkName = $data['linkName'];
        $this->stages = empty($data['stages']) ? [] : $data['stages'];
        $this->metaData = $data['metaData'];
        $this->path = $data['path'];
        $this->menu = $data['menu'];
        $this->serviceTitle = $data['serviceTitle'];
        $this->serviceTitleRoute = $data['serviceTitleRoute'];
        $this->menuItems = $data['menuItems'];
        $this->menuHome = $data['menuHome'];

        return $this;
    }

    /**
     * save
     */
    public function save()
    {
        $data = [
            'linkName' => $this->linkName,
            'stages' => $this->stages,
            'metaData' => $this->metaData,
            'path' => $this->path,
            'menu' => $this->menu,
            'serviceTitle' => $this->serviceTitle,
            'serviceTitleRoute' => $this->serviceTitleRoute,
            'menuItems' => $this->menuItems,
            'menuHome' => $this->menuHome,
        ];
        $this->cacheDriver->save($this->getCacheKey(), $data);

        return $this;
    }
}
